var __defProp = Object.defineProperty;
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};

// src/datalink.ts
var datalink_exports = {};
__export(datalink_exports, {
  ConnectionsResponse: () => ConnectionsResponse,
  DATALINK_PROTOCOL: () => DATALINK_PROTOCOL,
  DEFAULT_ARCH: () => DEFAULT_ARCH,
  DEFAULT_PROGRAM: () => DEFAULT_PROGRAM,
  DataLinkConnection: () => DataLinkConnection,
  DataLinkIdStats: () => DataLinkIdStats,
  DataLinkPacket: () => DataLinkPacket,
  DataLinkResponse: () => DataLinkResponse,
  DataLinkStats: () => DataLinkStats,
  ENDSTREAM: () => ENDSTREAM,
  ERROR: () => ERROR,
  ID: () => ID,
  INFO: () => INFO,
  IRIS_RINGSERVER_URL: () => IRIS_RINGSERVER_URL,
  MAX_PROC_NUM: () => MAX_PROC_NUM,
  MODE: () => MODE,
  MSEED3_TYPE: () => MSEED3_TYPE,
  MSEED_TYPE: () => MSEED_TYPE,
  OK: () => OK,
  PACKET: () => PACKET,
  QUERY_MODE: () => QUERY_MODE,
  STREAM: () => STREAM,
  STREAM_MODE: () => STREAM_MODE,
  StatusResponse: () => StatusResponse,
  StreamStat: () => StreamStat,
  StreamsResponse: () => StreamsResponse,
  ThreadStat: () => ThreadStat,
  USER_BROWSER: () => USER_BROWSER,
  daliDateTime: () => daliDateTime,
  dateTimeToHPTime: () => dateTimeToHPTime,
  hpTimeToDateTime: () => hpTimeToDateTime,
  stringToUint8Array: () => stringToUint8Array
});

// src/util.ts
var util_exports = {};
__export(util_exports, {
  BINARY_MIME: () => BINARY_MIME,
  JSONAPI_MIME: () => JSONAPI_MIME,
  JSON_MIME: () => JSON_MIME,
  SVG_MIME: () => SVG_MIME,
  SVG_NS: () => SVG_NS,
  TEXT_MIME: () => TEXT_MIME,
  UTC_OPTIONS: () => UTC_OPTIONS,
  WAY_FUTURE: () => WAY_FUTURE,
  XHTML_NS: () => XHTML_NS,
  XML_MIME: () => XML_MIME,
  asStringDictionary: () => asStringDictionary,
  calcClockOffset: () => calcClockOffset,
  checkLuxonValid: () => checkLuxonValid,
  checkProtocol: () => checkProtocol,
  checkStringOrDate: () => checkStringOrDate,
  cloneFetchInitObj: () => cloneFetchInitObj,
  createSVGElement: () => createSVGElement,
  dataViewToString: () => dataViewToString,
  defaultFetchInitObj: () => defaultFetchInitObj,
  default_fetch: () => default_fetch,
  doBoolGetterSetter: () => doBoolGetterSetter,
  doFetchWithTimeout: () => doFetchWithTimeout,
  doFloatGetterSetter: () => doFloatGetterSetter,
  doIntGetterSetter: () => doIntGetterSetter,
  doMomentGetterSetter: () => doMomentGetterSetter,
  doStringGetterSetter: () => doStringGetterSetter,
  downloadBlobAsFile: () => downloadBlobAsFile,
  durationEnd: () => durationEnd,
  errorFetch: () => errorFetch,
  getFetch: () => getFetch,
  hasArgs: () => hasArgs,
  hasNoArgs: () => hasNoArgs,
  isDef: () => isDef,
  isError: () => isError,
  isNonEmptyStringArg: () => isNonEmptyStringArg,
  isNumArg: () => isNumArg,
  isObject: () => isObject,
  isStringArg: () => isStringArg,
  isoToDateTime: () => isoToDateTime,
  log: () => log,
  makeParam: () => makeParam,
  makePostParam: () => makePostParam,
  meanOfSlice: () => meanOfSlice,
  reErrorWithMessage: () => reErrorWithMessage,
  setDefaultFetch: () => setDefaultFetch,
  startDuration: () => startDuration,
  startEnd: () => startEnd,
  stringify: () => stringify,
  toError: () => toError,
  toIsoWoZ: () => toIsoWoZ,
  toJSDate: () => toJSDate,
  updateVersionText: () => updateVersionText,
  validEndTime: () => validEndTime,
  validStartTime: () => validStartTime,
  warn: () => warn
});

// src/version.ts
var version = "3.1.4";

// src/util.ts
import { DateTime, Duration, Interval, FixedOffsetZone } from "luxon";
var XML_MIME = "application/xml";
var JSON_MIME = "application/json";
var JSONAPI_MIME = "application/vnd.api+json";
var SVG_MIME = "image/svg+xml";
var TEXT_MIME = "text/plain";
var BINARY_MIME = "application/octet-stream";
var UTC_OPTIONS = { zone: FixedOffsetZone.utcInstance };
function hasArgs(value) {
  return arguments.length !== 0 && typeof value !== "undefined";
}
function hasNoArgs(value) {
  return arguments.length === 0 || typeof value === "undefined";
}
function isStringArg(value) {
  return arguments.length !== 0 && (typeof value === "string" || isObject(value) && value instanceof String);
}
function isNumArg(value) {
  return arguments.length !== 0 && (typeof value === "number" || isObject(value) && value instanceof Number);
}
function isNonEmptyStringArg(value) {
  return arguments.length !== 0 && isStringArg(value) && value.length !== 0;
}
function isObject(obj) {
  return obj !== null && typeof obj === "object";
}
function isDef(value) {
  return value !== null && value !== void 0;
}
function reErrorWithMessage(err, message) {
  let out;
  if (!isDef(err)) {
    out = new Error(`${message}`);
  } else if (typeof err === "string") {
    out = new Error(`${message} ${err}`);
  } else if (err instanceof Error) {
    err.message = `${message} ${err.message}`;
    out = err;
  } else {
    out = new Error(`${message} ${stringify(err)}`);
  }
  return out;
}
function asStringDictionary(inobj) {
  if (typeof inobj !== "object") {
    throw new Error(`Expect obj to be object, but was ${stringify(inobj)}`);
  }
  const obj = inobj;
  return obj;
}
function doStringGetterSetter(inobj, field, value) {
  const hiddenField = `_${field}`;
  const obj = asStringDictionary(inobj);
  if (hasNoArgs(value) || value === null) {
    obj[hiddenField] = void 0;
  } else if (isStringArg(value)) {
    obj[hiddenField] = value;
  } else {
    throw new Error(
      `${field} value argument is optional or string, but was type ${typeof value}, '${value}' `
    );
  }
  return inobj;
}
function doBoolGetterSetter(inobj, field, value) {
  const hiddenField = `_${field}`;
  const obj = asStringDictionary(inobj);
  if (hasNoArgs(value) || value === null) {
    obj[hiddenField] = void 0;
  } else if (value === true || value === false) {
    obj[hiddenField] = value;
  } else {
    throw new Error(
      `${field} value argument is optional or boolean, but was type ${typeof value}, '${value}' `
    );
  }
  return inobj;
}
function doIntGetterSetter(inobj, field, value) {
  const hiddenField = `_${field}`;
  const obj = asStringDictionary(inobj);
  if (hasNoArgs(value) || value === null) {
    obj[hiddenField] = void 0;
  } else if (isNumArg(value)) {
    obj[hiddenField] = value;
  } else if (isStringArg(value) && Number.isFinite(Number(value))) {
    obj[hiddenField] = parseInt(value);
  } else {
    throw new Error(
      `${field} value argument is optional or number, but was type ${typeof value}, '${value}' `
    );
  }
  return inobj;
}
function doFloatGetterSetter(inobj, field, value) {
  const hiddenField = `_${field}`;
  const obj = asStringDictionary(inobj);
  if (hasNoArgs(value) || value === null) {
    obj[hiddenField] = void 0;
  } else if (isNumArg(value)) {
    obj[hiddenField] = value;
  } else if (isStringArg(value) && Number.isFinite(Number(value))) {
    obj[hiddenField] = parseFloat(value);
  } else {
    throw new Error(
      `value argument is optional or number, but was type ${typeof value}, '${value}' `
    );
  }
  return obj;
}
function doMomentGetterSetter(inobj, field, value) {
  const hiddenField = `_${field}`;
  const obj = asStringDictionary(inobj);
  if (hasNoArgs(value) || value === null) {
    obj[hiddenField] = void 0;
  } else if (isDef(value) && isObject(value) && DateTime.isDateTime(value)) {
    obj[hiddenField] = value;
  } else if (isDef(value) && DateTime.isDateTime(checkStringOrDate(value))) {
    obj[hiddenField] = checkStringOrDate(value);
  } else {
    throw new Error(
      `${field} value argument is optional, DateTime, date or date string, but was type ${typeof value}, '${stringify(
        value
      )}' `
    );
  }
  return obj;
}
function dataViewToString(dataView) {
  let out = "";
  for (let i = 0; i < dataView.byteLength; i++) {
    out += String.fromCharCode(dataView.getUint8(i));
  }
  return out;
}
function log(msg) {
  if (console) {
    console.log(`${stringify(msg)}`);
  }
  if (typeof document !== "undefined" && document !== null) {
    const p = document.createElement("p");
    p.textContent = `${stringify(msg)}`;
    const divDebug = document.querySelector("div#debug");
    if (isDef(divDebug)) {
      divDebug.appendChild(p);
    }
  }
}
function isError(error) {
  return typeof error === "object" && error !== null && error instanceof Error;
}
function toError(maybeError) {
  if (isError(maybeError)) return maybeError;
  try {
    return new Error(JSON.stringify(maybeError));
  } catch {
    return new Error(String(maybeError));
  }
}
function warn(msg) {
  if (console) {
    console.assert(false, `${stringify(msg)}`);
  }
  if (typeof document !== "undefined" && document !== null) {
    const p = document.createElement("p");
    p.textContent = `${stringify(msg)}`;
    document.querySelector("div#debug").appendChild(p);
  }
}
function stringify(value) {
  if (typeof value === "string") {
    return value;
  } else if (typeof value === "number") {
    return value.toString();
  } else if (typeof value === "boolean") {
    return value ? "true" : "false";
  } else if (typeof value === "undefined") {
    return "undefined";
  } else if (typeof value === "function") {
    return "function " + value.name;
  } else if (typeof value === "object") {
    if (value) {
      if (DateTime.isDateTime(value)) {
        const dateTimeValue = value;
        const s = dateTimeValue.toISO();
        return dateTimeValue.isValid && s ? s : `Invalid DateTime: ${dateTimeValue.invalidReason}: ${dateTimeValue.invalidExplanation}`;
      } else {
        return `${value?.constructor?.name} ${String(value)}`;
      }
    } else {
      return `${value}`;
    }
  } else {
    return "<unknown" + typeof value + "???>";
  }
}
function isoToDateTime(val) {
  if (val.toLowerCase() === "now") {
    return DateTime.utc();
  }
  return DateTime.fromISO(val, UTC_OPTIONS);
}
function startEnd(start, end) {
  if (isStringArg(start)) {
    start = isoToDateTime(start);
  }
  if (isStringArg(end)) {
    end = isoToDateTime(end);
  }
  return Interval.fromDateTimes(start, end);
}
function startDuration(start, duration) {
  if (isStringArg(start)) {
    start = isoToDateTime(start);
  }
  if (isStringArg(duration)) {
    duration = Duration.fromISO(duration);
  } else if (isNumArg(duration)) {
    duration = Duration.fromMillis(1e3 * duration);
  }
  if (duration.valueOf() < 0) {
    return Interval.before(start, duration.negate());
  } else {
    return Interval.after(start, duration);
  }
}
function durationEnd(duration, end) {
  if (isStringArg(end)) {
    end = isoToDateTime(end);
  }
  if (isStringArg(duration)) {
    duration = Duration.fromISO(duration);
  } else if (isNumArg(duration)) {
    duration = Duration.fromMillis(1e3 * duration);
  }
  if (duration.valueOf() < 0) {
    return Interval.after(end, duration.negate());
  } else {
    return Interval.before(end, duration);
  }
}
function calcClockOffset(serverTimeUTC) {
  return DateTime.utc().diff(serverTimeUTC).toMillis() * 1e3;
}
var WAY_FUTURE = DateTime.fromISO("2500-01-01T00:00:00Z");
function checkStringOrDate(d) {
  if (DateTime.isDateTime(d)) {
    return d;
  } else if (d instanceof Date) {
    return DateTime.fromJSDate(d, UTC_OPTIONS);
  } else if (isNumArg(d)) {
    return DateTime.fromMillis(d, UTC_OPTIONS);
  } else if (isNonEmptyStringArg(d)) {
    const lc = d.toLowerCase();
    if (d.length === 0 || lc === "now") {
      return DateTime.utc();
    } else {
      return isoToDateTime(d);
    }
  }
  throw new Error(`unknown date type: ${stringify(d)} ${typeof d}`);
}
function makeParam(name, val) {
  return `${name}=${encodeURIComponent(stringify(val))}&`;
}
function makePostParam(name, val) {
  return name + "=" + stringify(val) + "\n";
}
function toIsoWoZ(date) {
  if (date.isValid) {
    let out = date.toISO();
    if (out == null) {
      throw new Error(`Bad date: ${stringify(date)}`);
    }
    if (out.endsWith("Z")) {
      out = out.substring(0, out.length - 1);
    }
    return out;
  } else {
    throw new Error(`${date.invalidReason}: ${date.invalidExplanation}`);
  }
}
function validStartTime(interval) {
  const d = interval.start;
  if (d == null) {
    throw new Error(`Bad interval: ${stringify(interval)}`);
  }
  return d;
}
function validEndTime(interval) {
  const d = interval.end;
  if (d == null) {
    throw new Error(`Bad interval: ${stringify(interval)}`);
  }
  return d;
}
function toJSDate(d) {
  if (!d) {
    throw new Error(`Null/undef DateTime: ${d}`);
  }
  if (!d.isValid) {
    throw new Error(`${d.invalidReason}: ${d.invalidExplanation}`);
  }
  return d.toJSDate();
}
function checkLuxonValid(d, msg) {
  if (d == null) {
    const m = msg ? msg : "";
    throw new Error(`Null luxon value: ${d} ${m}`);
  }
  if (!d.isValid) {
    const m = msg ? msg : "";
    throw new Error(
      `Invalid Luxon: ${typeof d} ${d?.constructor?.name} ${d.invalidReason}: ${d.invalidExplanation} ${m}`
    );
  }
  return d;
}
function checkProtocol() {
  let _protocol = "http:";
  if (typeof document !== "undefined" && document !== null && "location" in document && "protocol" in document.location && "https:" === document.location.protocol) {
    _protocol = "https:";
  }
  return _protocol;
}
function defaultFetchInitObj(mimeType) {
  const headers = {};
  if (isStringArg(mimeType)) {
    headers.Accept = mimeType;
  }
  return {
    cache: "no-cache",
    redirect: "follow",
    mode: "cors",
    referrer: "seisplotjs",
    headers
  };
}
function cloneFetchInitObj(fetchInit) {
  const out = {};
  if (fetchInit) {
    for (const [key, value] of Object.entries(fetchInit)) {
      if (Array.isArray(value)) {
        out[key] = value.slice();
      } else {
        out[key] = value;
      }
    }
  }
  return out;
}
function errorFetch(_url, _init) {
  throw new Error("There is no fetch!?!?!");
}
var default_fetch = null;
function setDefaultFetch(fetcher) {
  if (fetcher != null) {
    default_fetch = fetcher;
  }
}
function getFetch() {
  if (default_fetch != null) {
    return default_fetch;
  } else if (window != null) {
    return window.fetch;
  } else if (global != null) {
    return global.fetch;
  } else {
    return errorFetch;
  }
}
function doFetchWithTimeout(url, fetchInit, timeoutSec2, fetcher) {
  const controller = new AbortController();
  const signal = controller.signal;
  if (!fetcher) {
    fetcher = getFetch();
  }
  if (!fetcher) {
    fetcher = window.fetch;
  }
  let internalFetchInit = isDef(fetchInit) ? fetchInit : defaultFetchInitObj();
  internalFetchInit = cloneFetchInitObj(internalFetchInit);
  if (internalFetchInit.redirect === "follow" && internalFetchInit.method === "POST") {
    internalFetchInit.redirect = "manual";
  }
  if (!isDef(timeoutSec2)) {
    timeoutSec2 = 30;
  }
  setTimeout(() => controller.abort(), timeoutSec2 * 1e3);
  internalFetchInit.signal = signal;
  let absoluteUrl;
  if (url instanceof URL) {
    absoluteUrl = url;
  } else if (isStringArg(url)) {
    if (url.startsWith("http://") || url.startsWith("https://")) {
      absoluteUrl = new URL(url);
    } else {
      absoluteUrl = new URL(url, document.URL);
    }
  } else {
    throw new Error(`url must be string or URL, ${stringify(url)}`);
  }
  log(
    `attempt to fetch ${internalFetchInit.method ? internalFetchInit.method : ""} ${stringify(
      absoluteUrl
    )}`
  );
  const fetchForRedirect = fetcher;
  return fetcher(absoluteUrl.href, internalFetchInit).catch((err) => {
    log("fetch failed, possible CORS or PrivacyBadger or NoScript?");
    throw err;
  }).then(function(response) {
    if (response.ok || response.status === 404) {
      return response;
    } else if (response.status >= 300 && response.status <= 399) {
      if (checkProtocol() === "http:" && absoluteUrl.href.startsWith("http://")) {
        const httpsUrl = new URL(`https://${absoluteUrl.href.slice(7)}`);
        const method = internalFetchInit.method ? internalFetchInit.method : "";
        log(
          `attempt fetch redirect ${response.status} ${method} to ${stringify(httpsUrl)}`
        );
        return fetchForRedirect(httpsUrl.href, internalFetchInit).then(
          (httpsResponse) => {
            if (httpsResponse.ok || httpsResponse.status === 404) {
              return httpsResponse;
            } else {
              return response.text().then((text) => {
                throw new Error(
                  `fetch response was redirect for http and failed for https. ${response.ok} ${response.status}, ${httpsResponse.ok} ${httpsResponse.status} 
${text}`
                );
              });
            }
          }
        );
      }
    }
    return response.text().then((text) => {
      throw new Error(
        `fetch response was not ok. ${response.ok} ${response.status}
${text}`
      );
    });
  });
}
function downloadBlobAsFile(data, filename, mimeType = "application/octet-stream") {
  if (!data) {
    throw new Error("data is empty");
  }
  if (!filename) filename = "filetodownload.txt";
  const blob = new Blob([data], { type: mimeType });
  const e = document.createEvent("MouseEvents");
  const a = document.createElement("a");
  a.download = filename;
  a.href = window.URL.createObjectURL(blob);
  a.dataset.downloadurl = [mimeType, a.download, a.href].join(":");
  e.initMouseEvent(
    "click",
    true,
    false,
    window,
    0,
    0,
    0,
    0,
    0,
    false,
    false,
    false,
    false,
    0,
    null
  );
  a.dispatchEvent(e);
}
function meanOfSlice(dataSlice, totalPts) {
  if (dataSlice.length < 8) {
    return (
      // @ts-expect-error different array types confuses typescript
      dataSlice.reduce(function(acc, val) {
        return acc + val;
      }, 0) / totalPts
    );
  } else {
    const byTwo = Math.floor(dataSlice.length / 2);
    return meanOfSlice(dataSlice.slice(0, byTwo), totalPts) + meanOfSlice(dataSlice.slice(byTwo, dataSlice.length), totalPts);
  }
}
var SVG_NS = "http://www.w3.org/2000/svg";
var XHTML_NS = "http://www.w3.org/1999/xhtml";
function createSVGElement(name) {
  return document.createElementNS(SVG_NS, name);
}
function updateVersionText(selector = "#sp-version") {
  document.querySelectorAll(selector).forEach((el) => {
    el.textContent = version;
  });
}

// src/miniseed.ts
var miniseed_exports = {};
__export(miniseed_exports, {
  BTime: () => BTime,
  Blockette: () => Blockette,
  Blockette100: () => Blockette100,
  Blockette1000: () => Blockette1000,
  Blockette1001: () => Blockette1001,
  D_TYPECODE: () => D_TYPECODE,
  DataHeader: () => DataHeader,
  DataRecord: () => DataRecord,
  MINISEED_MIME: () => MINISEED_MIME,
  M_TYPECODE: () => M_TYPECODE,
  Q_TYPECODE: () => Q_TYPECODE,
  R_TYPECODE: () => R_TYPECODE,
  areContiguous: () => areContiguous,
  byChannel: () => byChannel,
  checkByteSwap: () => checkByteSwap,
  createSeismogramSegment: () => createSeismogramSegment,
  merge: () => merge,
  mergeSegments: () => mergeSegments,
  parseBTime: () => parseBTime,
  parseBlockette: () => parseBlockette,
  parseDataRecords: () => parseDataRecords,
  parseSingleDataRecord: () => parseSingleDataRecord,
  parseSingleDataRecordHeader: () => parseSingleDataRecordHeader,
  seismogramPerChannel: () => seismogramPerChannel,
  seismogramSegmentPerChannel: () => seismogramSegmentPerChannel
});
import { DateTime as DateTime5, Duration as Duration5 } from "luxon";

// src/fdsnsourceid.ts
var fdsnsourceid_exports = {};
__export(fdsnsourceid_exports, {
  EMPTY_LOC_CODE: () => EMPTY_LOC_CODE,
  FDSNSourceId: () => FDSNSourceId,
  FDSN_PREFIX: () => FDSN_PREFIX,
  LocationSourceId: () => LocationSourceId,
  NetworkSourceId: () => NetworkSourceId,
  NslcId: () => NslcId,
  SEP: () => SEP,
  SourceIdSorter: () => SourceIdSorter,
  StationSourceId: () => StationSourceId,
  bandCodeForRate: () => bandCodeForRate,
  parseSourceId: () => parseSourceId
});
var FDSN_PREFIX = "FDSN:";
var SEP = "_";
var FDSNSourceId = class _FDSNSourceId {
  networkCode;
  stationCode;
  locationCode;
  bandCode;
  sourceCode;
  subsourceCode;
  constructor(networkCode, stationCode, locationCode, bandCode, sourceCode, subsourceCode) {
    this.networkCode = networkCode;
    this.stationCode = stationCode;
    this.locationCode = locationCode;
    this.bandCode = bandCode;
    this.sourceCode = sourceCode;
    this.subsourceCode = subsourceCode;
  }
  static createUnknown(sampRate, source, subsource) {
    const s = source ? source : "Y";
    const ss = subsource ? subsource : "X";
    return new _FDSNSourceId("XX", "ABC", "", bandCodeForRate(sampRate), s, ss);
  }
  static parse(id) {
    if (!id.startsWith(FDSN_PREFIX)) {
      throw new Error(`sourceid must start with ${FDSN_PREFIX}: ${id}`);
    }
    const items = id.slice(FDSN_PREFIX.length).split(SEP);
    if (items.length === 6) {
      return new _FDSNSourceId(
        items[0],
        items[1],
        items[2],
        items[3],
        items[4],
        items[5]
      );
    } else {
      throw new Error(
        `FDSN sourceid must have 6 items for channel; separated by '${SEP}': ${id}`
      );
    }
  }
  static fromNslc(net, sta, loc, channelCode) {
    let band;
    let source;
    let subsource;
    if (channelCode.length === 3) {
      band = channelCode.charAt(0);
      source = channelCode.charAt(1);
      subsource = channelCode.charAt(2);
    } else {
      const b_s_ss = /(\w)_(\w+)_(\w+)/;
      const match = b_s_ss.exec(channelCode);
      if (match) {
        band = match[1];
        source = match[2];
        subsource = match[3];
      } else {
        throw new Error(
          `channel code must be length 3 or have 3 items separated by '${SEP}': ${channelCode}`
        );
      }
    }
    return new _FDSNSourceId(net, sta, loc, band, source, subsource);
  }
  static fromNslcId(nslcId) {
    return _FDSNSourceId.fromNslc(
      nslcId.networkCode,
      nslcId.stationCode,
      nslcId.locationCode,
      nslcId.channelCode
    );
  }
  static parseNslc(nslc, sep = ".") {
    const items = nslc.split(sep);
    if (items.length < 4) {
      throw new Error(
        `channel nslc must have 4 items separated by '${sep}': ${nslc}`
      );
    }
    return _FDSNSourceId.fromNslc(items[0], items[1], items[2], items[3]);
  }
  stationSourceId() {
    return new StationSourceId(this.networkCode, this.stationCode);
  }
  networkSourceId() {
    return new NetworkSourceId(this.networkCode);
  }
  asNslc() {
    let chanCode;
    if (this.bandCode.length === 1 && this.sourceCode.length === 1 && this.subsourceCode.length === 1) {
      chanCode = `${this.bandCode}${this.sourceCode}${this.subsourceCode}`;
    } else {
      chanCode = `${this.bandCode}${SEP}${this.sourceCode}${SEP}${this.subsourceCode}`;
    }
    return new NslcId(
      this.networkCode,
      this.stationCode,
      this.locationCode,
      chanCode
    );
  }
  /**
   * returns a channel code. If this is an old style NSLC, it will be 3 chars,
   * but if either source or subsouce is more than one char, it will be
   * three fields delimited by underscores.
   *
   * @returns the channel code part of the id
   */
  formChannelCode() {
    return this.asNslc().channelCode;
  }
  toString() {
    return `${FDSN_PREFIX}${this.networkCode}${SEP}${this.stationCode}${SEP}${this.locationCode}${SEP}${this.bandCode}${SEP}${this.sourceCode}${SEP}${this.subsourceCode}`;
  }
  toStringNoPrefix() {
    return `${this.networkCode}${SEP}${this.stationCode}${SEP}${this.locationCode}${SEP}${this.bandCode}${SEP}${this.sourceCode}${SEP}${this.subsourceCode}`;
  }
  equals(other) {
    if (!other) {
      return false;
    }
    return this.toString() === other.toString();
  }
  clone() {
    return new _FDSNSourceId(
      this.networkCode,
      this.stationCode,
      this.locationCode,
      this.bandCode,
      this.sourceCode,
      this.subsourceCode
    );
  }
};
var NetworkSourceId = class _NetworkSourceId {
  networkCode;
  constructor(networkCode) {
    this.networkCode = networkCode;
  }
  static parse(id) {
    if (!id.startsWith(FDSN_PREFIX)) {
      throw new Error(`sourceid must start with ${FDSN_PREFIX}: ${id}`);
    }
    const items = id.slice(FDSN_PREFIX.length).split(SEP);
    if (items.length === 1) {
      return new _NetworkSourceId(items[0]);
    } else {
      throw new Error(
        `FDSN network sourceid must have 1 items; separated by '${SEP}': ${id}`
      );
    }
    return new _NetworkSourceId(items[0]);
  }
  toString() {
    return `${FDSN_PREFIX}${this.networkCode}`;
  }
  equals(other) {
    return this.toString() === other.toString();
  }
};
var StationSourceId = class _StationSourceId {
  networkCode;
  stationCode;
  constructor(networkCode, stationCode) {
    this.networkCode = networkCode;
    this.stationCode = stationCode;
  }
  static parse(id) {
    if (!id.startsWith(FDSN_PREFIX)) {
      throw new Error(`station sourceid must start with ${FDSN_PREFIX}: ${id}`);
    }
    const items = id.slice(FDSN_PREFIX.length).split(SEP);
    if (items.length === 2) {
      return new _StationSourceId(items[0], items[1]);
    } else {
      throw new Error(
        `FDSN station sourceid must have 2 items; separated by '${SEP}': ${id}`
      );
    }
    return new _StationSourceId(items[0], items[1]);
  }
  toString() {
    return `${FDSN_PREFIX}${this.networkCode}${SEP}${this.stationCode}`;
  }
  networkSourceId() {
    return new NetworkSourceId(this.networkCode);
  }
  equals(other) {
    return this.toString() === other.toString();
  }
};
var LocationSourceId = class {
  networkCode;
  stationCode;
  locationCode;
  constructor(networkCode, stationCode, locationCode) {
    this.networkCode = networkCode;
    this.stationCode = stationCode;
    this.locationCode = locationCode;
  }
  toString() {
    return `${FDSN_PREFIX}${this.networkCode}${SEP}${this.stationCode}${SEP}${this.locationCode}`;
  }
  equals(other) {
    return this.toString() === other.toString();
  }
};
function bandCodeForRate(sampRate, resp_lb) {
  if (!sampRate) {
    return "I";
  }
  if (sampRate >= 5e3) {
    return "J";
  } else if (sampRate >= 1e3 && sampRate < 5e3) {
    if (resp_lb && resp_lb < 0.1) {
      return "F";
    }
    return "G";
  } else if (sampRate >= 250 && sampRate < 1e3) {
    if (resp_lb && resp_lb < 0.1) {
      return "C";
    }
    return "D";
  } else if (sampRate >= 80 && sampRate < 250) {
    if (resp_lb && resp_lb < 0.1) {
      return "H";
    }
    return "E";
  } else if (sampRate >= 10 && sampRate < 80) {
    if (resp_lb && resp_lb < 0.1) {
      return "B";
    }
    return "S";
  } else if (sampRate > 1.05 && sampRate < 10) {
    return "M";
  } else if (sampRate >= 0.95 && sampRate <= 1.05) {
    return "L";
  } else if (sampRate >= 0.1 && sampRate < 1) {
    return "V";
  } else if (sampRate >= 0.01 && sampRate < 0.1) {
    return "U";
  } else if (sampRate >= 1e-3 && sampRate < 0.01) {
    return "W";
  } else if (sampRate >= 1e-4 && sampRate < 1e-3) {
    return "R";
  } else if (sampRate >= 1e-5 && sampRate < 1e-4) {
    return "P";
  } else if (sampRate >= 1e-6 && sampRate < 1e-5) {
    return "T";
  } else if (sampRate < 1e-6) {
    return "Q";
  } else {
    throw new Error(`Unable to calc band code for: ${sampRate} ${resp_lb}`);
  }
}
var EMPTY_LOC_CODE = "--";
var NslcId = class _NslcId {
  networkCode;
  stationCode;
  locationCode;
  channelCode;
  constructor(net, sta, loc, chan) {
    this.networkCode = net;
    this.stationCode = sta;
    this.locationCode = loc;
    this.channelCode = chan;
  }
  static parse(nslc, sep = ".") {
    const items = nslc.split(SEP);
    if (items.length !== 4) {
      throw new Error(
        `NSLC id must have 4 items; separated by '${sep}': ${nslc}`
      );
    }
    return new _NslcId(items[0], items[1], items[2], items[3]);
  }
  toString() {
    return `${this.networkCode}_${this.stationCode}_${this.locationCode}_${this.channelCode}`;
  }
  equals(other) {
    if (this.networkCode !== other.networkCode) {
      return false;
    }
    if (this.stationCode !== other.stationCode) {
      return false;
    }
    const myLoc = this.locationCode === EMPTY_LOC_CODE ? "" : this.locationCode;
    const otherLoc = other.locationCode === EMPTY_LOC_CODE ? "" : other.locationCode;
    if (myLoc !== otherLoc) {
      return false;
    }
    if (this.channelCode !== other.channelCode) {
      return false;
    }
    return true;
  }
};
function parseSourceId(id) {
  if (!id.startsWith(FDSN_PREFIX)) {
    throw new Error(`sourceid must start with ${FDSN_PREFIX}: ${id}`);
  }
  const items = id.slice(FDSN_PREFIX.length).split(SEP);
  if (items.length === 1) {
    return new NetworkSourceId(items[0]);
  } else if (items.length === 2) {
    return new StationSourceId(items[0], items[1]);
  } else if (items.length !== 6) {
    throw new Error(
      `FDSN sourceid must have 6 items for channel, 2 for station or 1 for network; separated by '${SEP}': ${id}`
    );
  }
  return new FDSNSourceId(
    items[0],
    items[1],
    items[2],
    items[3],
    items[4],
    items[5]
  );
}
function SourceIdSorter(aSid, bSid) {
  if (aSid.networkCode !== bSid.networkCode) {
    return aSid.networkCode.localeCompare(bSid.networkCode);
  }
  if (aSid.stationCode !== bSid.stationCode) {
    return aSid.stationCode.localeCompare(bSid.stationCode);
  }
  if (aSid.locationCode !== bSid.locationCode) {
    return aSid.locationCode.localeCompare(bSid.locationCode);
  }
  if (aSid.bandCode !== bSid.bandCode) {
    return aSid.bandCode.localeCompare(bSid.bandCode);
  }
  if (aSid.sourceCode !== bSid.sourceCode) {
    return aSid.sourceCode.localeCompare(bSid.sourceCode);
  }
  return aSid.subsourceCode.localeCompare(bSid.subsourceCode);
}

// src/seismogramsegment.ts
var seismogramsegment_exports = {};
__export(seismogramsegment_exports, {
  COUNT_UNIT: () => COUNT_UNIT,
  SeismogramSegment: () => SeismogramSegment
});
import { Duration as Duration3, Interval as Interval2 } from "luxon";

// src/scale.ts
var scale_exports = {};
__export(scale_exports, {
  AMPLITUDE_MODE: () => AMPLITUDE_MODE,
  AlignmentLinkedTimeScale: () => AlignmentLinkedTimeScale,
  AmplitudeScalable: () => AmplitudeScalable,
  FixedHalfWidthAmplitudeScale: () => FixedHalfWidthAmplitudeScale,
  IndividualAmplitudeScale: () => IndividualAmplitudeScale,
  LinkedAmplitudeScale: () => LinkedAmplitudeScale,
  LinkedTimeScale: () => LinkedTimeScale,
  MinMaxable: () => MinMaxable,
  TimeScalable: () => TimeScalable
});
import { Duration as Duration2 } from "luxon";
var AMPLITUDE_MODE = /* @__PURE__ */ ((AMPLITUDE_MODE2) => {
  AMPLITUDE_MODE2["Raw"] = "raw";
  AMPLITUDE_MODE2["Zero"] = "zero";
  AMPLITUDE_MODE2["MinMax"] = "minmax";
  AMPLITUDE_MODE2["Mean"] = "mean";
  return AMPLITUDE_MODE2;
})(AMPLITUDE_MODE || {});
var _lastId = 0;
var MinMaxable = class _MinMaxable {
  min;
  max;
  constructor(min, max) {
    this.min = min;
    this.max = max;
  }
  get middle() {
    return (this.min + this.max) / 2;
  }
  get halfWidth() {
    return this.fullWidth / 2;
  }
  get fullWidth() {
    return this.max - this.min;
  }
  union(omm) {
    if (omm) {
      return new _MinMaxable(
        Math.min(this.min, omm.min),
        Math.max(this.max, omm.max)
      );
    } else {
      return this;
    }
  }
  expandPercentage(percent) {
    return _MinMaxable.fromMiddleHalfWidth(
      this.middle,
      this.halfWidth * percent
    );
  }
  /**
   * This as a d3 style 2 element array.
   *
   * @returns length 2 array of min then max
   */
  asArray() {
    return [this.min, this.max];
  }
  toString() {
    return `${this.min} to ${this.max}, mid: ${this.middle} hw: ${this.halfWidth}`;
  }
  /**
   * Create MinMaxable from a d3 style two element array.
   *
   * @param  minmax  array of min then max
   * @returns       new MinMaxable
   */
  static fromArray(minmax) {
    if (minmax.length < 2) {
      throw new Error(`array must have lenght 2, ${minmax.length}`);
    }
    return new _MinMaxable(minmax[0], minmax[1]);
  }
  static fromMiddleHalfWidth(mid, halfWidth) {
    return new _MinMaxable(mid - halfWidth, mid + halfWidth);
  }
};
var AmplitudeScalable = class {
  minMax;
  constructor(minMax) {
    if (minMax) {
      this.minMax = minMax;
    } else {
      this.minMax = new MinMaxable(0, 0);
    }
  }
  // eslint-disable-next-line no-unused-vars
  notifyAmplitudeChange(_middle, _halfWidth) {
  }
  get middle() {
    return this.minMax.middle;
  }
  get halfWidth() {
    return this.minMax.halfWidth;
  }
  get fullWidth() {
    return this.minMax.fullWidth;
  }
  get min() {
    return this.minMax.min;
  }
  get max() {
    return this.minMax.max;
  }
  toString() {
    return this.minMax.toString();
  }
};
var TimeScalable = class {
  alignmentTimeOffset;
  duration;
  constructor(alignmentTimeOffset, duration) {
    this.alignmentTimeOffset = alignmentTimeOffset;
    this.duration = duration;
  }
  // eslint-disable-next-line no-unused-vars
  notifyTimeRangeChange(_alignmentTimeOffset, _duration) {
  }
};
var LinkedAmplitudeScale = class {
  /**
   * @private
   */
  _graphSet;
  _halfWidth;
  _recalcTimeoutID;
  _scaleId;
  constructor(graphList) {
    this._scaleId = ++_lastId;
    const glist = graphList ? graphList : [];
    this._halfWidth = 0;
    this._graphSet = new Set(glist);
    this._recalcTimeoutID = null;
  }
  get halfWidth() {
    return this._halfWidth;
  }
  set halfWidth(val) {
    if (this._halfWidth !== val) {
      this._halfWidth = val;
      this.notifyAll().catch((m) => {
        console.warn(`problem recalc halfWidth: ${m}`);
      });
    }
  }
  /**
   * Links new Seismograph with this amplitude scale.
   *
   * @param   graphList Array of AmplitudeScalable to link
   */
  linkAll(graphList) {
    graphList.forEach((graph) => {
      if ("notifyAmplitudeChange" in graph) {
        this._graphSet.add(graph);
      } else if ("amp_scalable" in graph) {
        this._graphSet.add(graph.amp_scalable);
      } else {
      }
    });
    this.recalculate().catch((m) => {
      console.warn(`problem recalc linkAll: ${m}`);
    });
  }
  /**
   * Link new Seismograph with this amplitude scale.
   *
   * @param   graph AmplitudeScalable to link
   */
  link(graph) {
    this.linkAll([graph]);
  }
  /**
   * Unlink Seismograph with this amplitude scale.
   *
   * @param   graph AmplitudeScalable to unlink
   */
  unlink(graph) {
    this._graphSet.delete(graph);
    this.recalculate().catch((m) => {
      console.warn(`problem recalc unlink: ${m}`);
    });
  }
  /**
   * Recalculate the best amplitude scale for all Seismographs. Causes a redraw.
   *
   * @returns array of promise of best amp scales
   */
  recalculate() {
    const maxHalfRange = this.graphList.reduce((acc, cur) => {
      return acc > cur.halfWidth ? acc : cur.halfWidth;
    }, 0);
    let promiseOut;
    if (this.halfWidth !== maxHalfRange) {
      this.halfWidth = maxHalfRange;
      promiseOut = this._internalNotifyAll();
    } else {
      promiseOut = Promise.all(this.graphList.map((g) => Promise.resolve(g)));
    }
    return promiseOut;
  }
  _internalNotifyAll() {
    const hw = this.halfWidth;
    return Promise.all(
      this.graphList.map((g) => {
        return new Promise((resolve) => {
          setTimeout(() => {
            g.notifyAmplitudeChange(g.middle, hw);
            resolve(g);
          }, 10);
        });
      })
    );
  }
  notifyAll() {
    return this._internalNotifyAll();
  }
  get graphList() {
    return Array.from(this._graphSet.values());
  }
};
var IndividualAmplitudeScale = class extends LinkedAmplitudeScale {
  constructor(graphList) {
    super(graphList);
  }
  recalculate() {
    return this.notifyAll();
  }
  notifyAll() {
    return Promise.all(
      this.graphList.map((g) => {
        return new Promise((resolve) => {
          setTimeout(() => {
            g.notifyAmplitudeChange(g.middle, g.halfWidth);
            resolve(g);
          }, 10);
        });
      })
    );
  }
};
var FixedHalfWidthAmplitudeScale = class extends LinkedAmplitudeScale {
  constructor(halfWidth, graphList) {
    super(graphList);
    this.halfWidth = halfWidth;
  }
  recalculate() {
    return this.notifyAll();
  }
  notifyAll() {
    const hw = this.halfWidth;
    return Promise.all(
      this.graphList.map((g) => {
        return new Promise((resolve) => {
          setTimeout(() => {
            g.notifyAmplitudeChange(g.middle, hw);
            resolve(g);
          }, 10);
        });
      })
    );
  }
};
var LinkedTimeScale = class {
  /**
   * @private
   */
  _graphSet;
  _originalDuration;
  _originalOffset;
  _zoomedDuration;
  _zoomedOffset;
  _scaleId;
  constructor(graphList, originalDuration, originalOffset, scaleId) {
    if (scaleId) {
      this._scaleId = scaleId;
    } else {
      this._scaleId = -1;
    }
    const glist = graphList ? graphList : [];
    this._graphSet = new Set(glist);
    this._originalDuration = Duration2.fromMillis(0);
    this._originalOffset = Duration2.fromMillis(0);
    this._zoomedDuration = null;
    this._zoomedOffset = null;
    if (isDef(originalDuration)) {
      this._originalDuration = originalDuration;
      this._zoomedDuration = originalDuration;
    } else if (glist.length > 0) {
      this._originalDuration = glist.reduce((acc, cur) => {
        return acc > cur.duration ? acc : cur.duration;
      }, Duration2.fromMillis(0));
    }
    if (originalOffset) {
      this._originalOffset = originalOffset;
    } else {
      this._originalOffset = Duration2.fromMillis(0);
    }
    this.recalculate().catch((m) => {
      console.warn(`problem recalc constructor: ${m}`);
    });
  }
  /**
   * Link new TimeScalable with this time scale.
   *
   * @param   graph TimeScalable to link
   */
  link(graph) {
    this.linkAll([graph]);
  }
  /**
   * Links TimeScalable with this time scale. Each
   * object in the array should either be a TimeScalable
   * or have a time_scalable field that is a TimeScalable.
   *
   * @param   graphList Array of TimeScalable to link
   */
  linkAll(graphList) {
    graphList.forEach((graph) => {
      if ("notifyTimeRangeChange" in graph) {
        this._graphSet.add(graph);
      } else if ("time_scalable" in graph) {
        this._graphSet.add(graph.time_scalable);
      } else {
      }
    });
    this.recalculate().catch((m) => {
      console.warn(`problem recalc linkAll: ${m}`);
    });
  }
  /**
   * Unlink TimeScalable with this amplitude scale.
   *
   * @param   graph TimeScalable to unlink
   */
  unlink(graph) {
    this._graphSet.delete(graph);
    this.recalculate().catch((m) => {
      console.warn(`problem recalc unlink: ${m}`);
    });
  }
  zoom(startOffset, duration) {
    this._zoomedDuration = duration;
    this._zoomedOffset = startOffset;
    this.recalculate().catch((m) => {
      console.warn(`problem recalc zoom: ${m}`);
    });
  }
  unzoom() {
    this._zoomedDuration = null;
    this._zoomedOffset = null;
    this.recalculate().catch((m) => {
      console.warn(`problem recalc unzoom: ${m}`);
    });
  }
  get offset() {
    return this._zoomedOffset ? this._zoomedOffset : this._originalOffset;
  }
  set offset(offset) {
    this._originalOffset = offset;
    this._zoomedOffset = offset;
    this.recalculate().catch((m) => {
      console.warn(`problem recalc set offset: ${m}`);
    });
  }
  get duration() {
    return isDef(this._zoomedDuration) ? this._zoomedDuration : this._originalDuration;
  }
  set duration(duration) {
    if (!isDef(duration)) {
      throw new Error(`Duration must be defined`);
    }
    this._originalDuration = duration;
    this._zoomedDuration = duration;
    this.recalculate().catch((m) => {
      console.warn(`problem recalc set duration: ${m}`);
    });
  }
  get origOffset() {
    return this._originalOffset;
  }
  get origDuration() {
    return this._originalDuration;
  }
  /**
   * Recalculate the best time scale for all Seismographs. Causes a redraw.
   * @returns promise to array of all linked items
   */
  recalculate() {
    if (!isDef(this._zoomedDuration) || this._originalDuration.toMillis() === 0) {
      this.graphList.forEach((graph) => {
        if (graph && graph.duration > this._originalDuration) {
          this._originalDuration = graph.duration;
        }
      });
    }
    return this.notifyAll();
  }
  notifyAll() {
    return Promise.all(
      this.graphList.map((g) => {
        return new Promise((resolve) => {
          setTimeout(() => {
            if (g != null) {
              g.notifyTimeRangeChange(this.offset, this.duration);
            }
            resolve(g);
          }, 10);
        });
      })
    );
  }
  get graphList() {
    return Array.from(this._graphSet.values());
  }
};
var AlignmentLinkedTimeScale = class extends LinkedTimeScale {
  constructor(graphList, originalDuration, originalOffset, scaleId) {
    super(graphList, originalDuration, originalOffset, scaleId);
  }
  /**
   * Does no calculation, just causes a redraw.
   * @returns promise to all linked items
   */
  recalculate() {
    return this.notifyAll();
  }
  notifyAll() {
    return Promise.all(
      this.graphList.map((g) => {
        return new Promise((resolve) => {
          setTimeout(() => {
            if (g != null) {
              g.notifyTimeRangeChange(this.offset, this.duration);
            }
            resolve(g);
          }, 10);
        });
      })
    );
  }
};

// src/seedcodec.ts
var seedcodec_exports = {};
__export(seedcodec_exports, {
  ASCII: () => ASCII,
  CDSN: () => CDSN,
  CodecException: () => CodecException,
  DOUBLE: () => DOUBLE,
  DWWSSN: () => DWWSSN,
  EncodedDataSegment: () => EncodedDataSegment,
  FLOAT: () => FLOAT,
  INT24: () => INT24,
  INTEGER: () => INTEGER,
  SHORT: () => SHORT,
  SRO: () => SRO,
  STEIM1: () => STEIM1,
  STEIM2: () => STEIM2,
  UnsupportedCompressionType: () => UnsupportedCompressionType,
  decodeSteim1: () => decodeSteim1,
  decodeSteim2: () => decodeSteim2,
  decompress: () => decompress,
  isFloatCompression: () => isFloatCompression
});
var ASCII = 0;
var SHORT = 1;
var INT24 = 2;
var INTEGER = 3;
var FLOAT = 4;
var DOUBLE = 5;
var STEIM1 = 10;
var STEIM2 = 11;
var CDSN = 16;
var SRO = 30;
var DWWSSN = 32;
var CodecException = class extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.name = "CodecException";
  }
};
var UnsupportedCompressionType = class extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.name = "UnsupportedCompressionType";
  }
};
function isFloatCompression(compressionType) {
  if (compressionType === FLOAT || compressionType === DOUBLE) {
    return true;
  }
  return false;
}
var EncodedDataSegment = class {
  compressionType;
  dataView;
  numSamples;
  littleEndian;
  constructor(compressionType, dataView, numSamples, littleEndian) {
    this.compressionType = compressionType;
    this.dataView = dataView;
    this.numSamples = numSamples;
    this.littleEndian = littleEndian;
  }
  isFloatCompression() {
    return isFloatCompression(this.compressionType);
  }
  decode() {
    return decompress(
      this.compressionType,
      this.dataView,
      this.numSamples,
      this.littleEndian
    );
  }
};
function decompress(compressionType, dataView, numSamples, littleEndian) {
  if (numSamples === 0) {
    return new Int32Array(0);
  }
  let out;
  let offset = 0;
  let i;
  switch (compressionType) {
    case SHORT:
    case DWWSSN:
      if (dataView.byteLength < 2 * numSamples) {
        throw new CodecException(
          "Not enough bytes for " + numSamples + " 16 bit data points, only " + dataView.byteLength + " bytes."
        );
      }
      out = new Int32Array(numSamples);
      for (i = 0; i < numSamples; i++) {
        out[i] = dataView.getInt16(offset, littleEndian);
        offset += 2;
      }
      break;
    case INTEGER:
      if (dataView.byteLength < 4 * numSamples) {
        throw new CodecException(
          "Not enough bytes for " + numSamples + " 32 bit data points, only " + dataView.byteLength + " bytes."
        );
      }
      out = new Int32Array(numSamples);
      for (i = 0; i < numSamples; i++) {
        out[i] = dataView.getInt32(offset, littleEndian);
        offset += 4;
      }
      break;
    case FLOAT:
      if (dataView.byteLength < 4 * numSamples) {
        throw new CodecException(
          "Not enough bytes for " + numSamples + " 32 bit data points, only " + dataView.byteLength + " bytes."
        );
      }
      out = new Float32Array(numSamples);
      for (i = 0; i < numSamples; i++) {
        out[i] = dataView.getFloat32(offset, littleEndian);
        offset += 4;
      }
      break;
    case DOUBLE:
      if (dataView.byteLength < 8 * numSamples) {
        throw new CodecException(
          "Not enough bytes for " + numSamples + " 64 bit data points, only " + dataView.byteLength + " bytes."
        );
      }
      out = new Float64Array(numSamples);
      for (i = 0; i < numSamples; i++) {
        out[i] = dataView.getFloat64(offset, littleEndian);
        offset += 8;
      }
      break;
    case STEIM1:
      out = decodeSteim1(dataView, numSamples, littleEndian, 0);
      break;
    case STEIM2:
      out = decodeSteim2(dataView, numSamples, littleEndian, 0);
      break;
    default:
      throw new UnsupportedCompressionType(
        "Type " + compressionType + " is not supported at this time."
      );
  }
  return out;
}
function decodeSteim1(dataView, numSamples, littleEndian, bias) {
  if (dataView.byteLength % 64 !== 0) {
    throw new CodecException(
      "encoded data length is not multiple of 64 bytes (" + dataView.byteLength + ")"
    );
  }
  const buf = new ArrayBuffer(4 * numSamples);
  const samples = new Int32Array(buf);
  let tempSamples;
  const numFrames = dataView.byteLength / 64;
  let current = 0;
  let start = 0;
  let firstData = 0;
  let lastValue = 0;
  let i, j;
  for (i = 0; i < numFrames; i++) {
    tempSamples = extractSteim1Samples(dataView, i * 64, littleEndian);
    firstData = 0;
    if (i === 0) {
      lastValue = bias;
      start = tempSamples[1];
      firstData = 3;
      if (bias === 0) lastValue = start - tempSamples[3];
    }
    for (j = firstData; j < tempSamples.length && current < numSamples; j++) {
      samples[current] = lastValue + tempSamples[j];
      lastValue = samples[current];
      current++;
    }
  }
  if (current !== numSamples) {
    throw new CodecException(
      "Number of samples decompressed doesn't match number in header: " + current + " !== " + numSamples
    );
  }
  return samples;
}
function extractSteim1Samples(dataView, offset, littleEndian) {
  const nibbles = dataView.getInt32(offset, littleEndian);
  let currNibble = 0;
  const temp = [];
  let currNum = 0;
  let i, n;
  for (i = 0; i < 16; i++) {
    currNibble = nibbles >> 30 - i * 2 & 3;
    switch (currNibble) {
      case 0:
        if (offset === 0) {
          temp[currNum++] = dataView.getInt32(offset + i * 4, littleEndian);
        }
        break;
      case 1:
        for (n = 0; n < 4; n++) {
          temp[currNum] = dataView.getInt8(offset + i * 4 + n);
          currNum++;
        }
        break;
      case 2:
        for (n = 0; n < 4; n += 2) {
          temp[currNum] = dataView.getInt16(offset + i * 4 + n, littleEndian);
          currNum++;
        }
        break;
      case 3:
        temp[currNum++] = dataView.getInt32(offset + i * 4, littleEndian);
        break;
      default:
        throw new CodecException("unreachable case: " + currNibble);
    }
  }
  return temp;
}
function decodeSteim2(dataView, numSamples, swapBytes, bias) {
  if (dataView.byteLength % 64 !== 0) {
    throw new CodecException(
      "encoded data length is not multiple of 64 bytes (" + dataView.byteLength + ")"
    );
  }
  const buf = new ArrayBuffer(4 * numSamples);
  const samples = new Int32Array(buf);
  let tempSamples;
  const numFrames = dataView.byteLength / 64;
  let current = 0;
  let start = 0;
  let firstData = 0;
  let lastValue = 0;
  for (let i = 0; i < numFrames; i++) {
    tempSamples = extractSteim2Samples(dataView, i * 64, swapBytes);
    firstData = 0;
    if (i === 0) {
      lastValue = bias;
      start = tempSamples[1];
      firstData = 3;
      if (bias === 0) lastValue = start - tempSamples[3];
    }
    for (let j = firstData; j < tempSamples.length && current < numSamples; j++) {
      samples[current] = lastValue + tempSamples[j];
      lastValue = samples[current];
      current++;
    }
  }
  if (current !== numSamples) {
    throw new CodecException(
      "Number of samples decompressed doesn't match number in header: " + current + " !== " + numSamples
    );
  }
  return samples;
}
function extractSteim2Samples(dataView, offset, swapBytes) {
  const nibbles = dataView.getUint32(offset, swapBytes);
  let currNibble = 0;
  let dnib = 0;
  const temp = new Int32Array(106);
  let tempInt;
  let currNum = 0;
  let diffCount = 0;
  let bitSize = 0;
  let headerSize = 0;
  for (let i = 0; i < 16; i++) {
    currNibble = nibbles >> 30 - i * 2 & 3;
    switch (currNibble) {
      case 0:
        if (offset === 0) {
          temp[currNum++] = dataView.getInt32(offset + i * 4, swapBytes);
        }
        break;
      case 1:
        temp[currNum++] = dataView.getInt8(offset + i * 4);
        temp[currNum++] = dataView.getInt8(offset + i * 4 + 1);
        temp[currNum++] = dataView.getInt8(offset + i * 4 + 2);
        temp[currNum++] = dataView.getInt8(offset + i * 4 + 3);
        break;
      case 2:
        tempInt = dataView.getUint32(offset + i * 4, swapBytes);
        dnib = tempInt >> 30 & 3;
        switch (dnib) {
          case 1:
            temp[currNum++] = tempInt << 2 >> 2;
            break;
          case 2:
            temp[currNum++] = tempInt << 2 >> 17;
            temp[currNum++] = tempInt << 17 >> 17;
            break;
          case 3:
            temp[currNum++] = tempInt << 2 >> 22;
            temp[currNum++] = tempInt << 12 >> 22;
            temp[currNum++] = tempInt << 22 >> 22;
            break;
          default:
            throw new CodecException(
              `Unknown case currNibble=${currNibble} dnib=${dnib} for chunk ${i} offset ${offset}, nibbles: ${nibbles}`
            );
        }
        break;
      case 3:
        tempInt = dataView.getUint32(offset + i * 4, swapBytes);
        dnib = tempInt >> 30 & 3;
        diffCount = 0;
        bitSize = 0;
        headerSize = 0;
        switch (dnib) {
          case 0:
            headerSize = 2;
            diffCount = 5;
            bitSize = 6;
            break;
          case 1:
            headerSize = 2;
            diffCount = 6;
            bitSize = 5;
            break;
          case 2:
            headerSize = 4;
            diffCount = 7;
            bitSize = 4;
            break;
          default:
            throw new CodecException(
              `Unknown case currNibble=${currNibble} dnib=${dnib} for chunk ${i} offset ${offset}, nibbles: ${nibbles}`
            );
        }
        if (diffCount > 0) {
          for (let d = 0; d < diffCount; d++) {
            temp[currNum++] = tempInt << headerSize + d * bitSize >> (diffCount - 1) * bitSize + headerSize;
          }
        }
        break;
      default:
        throw new CodecException(`Unknown case currNibble=${currNibble}`);
    }
  }
  return temp.slice(0, currNum);
}

// src/seismogramsegment.ts
var COUNT_UNIT = "count";
var SeismogramSegment = class _SeismogramSegment {
  /** Array of y values */
  _y;
  _compressed;
  /**
   * the sample rate in hertz
   *
   * @private
   */
  _sampleRate;
  /** @private */
  _startTime;
  _endTime_cache;
  _endTime_cache_numPoints;
  _sourceId;
  yUnit;
  _highlow;
  constructor(yArray, sampleRate, startTime, sourceId) {
    if (yArray instanceof Int32Array || yArray instanceof Float32Array || yArray instanceof Float64Array) {
      this._y = yArray;
      this._compressed = null;
    } else if (Array.isArray(yArray) && yArray.every((ee) => ee instanceof EncodedDataSegment)) {
      this._compressed = yArray;
      this._y = null;
    } else if (Array.isArray(yArray) && yArray.every((ee) => typeof ee === "number")) {
      this._y = Float64Array.from(yArray);
      this._compressed = null;
    } else {
      this._compressed = null;
      this._y = null;
    }
    if (sampleRate <= 0) {
      throw new Error(`SampleRate must be positive number: ${sampleRate}`);
    }
    this._sampleRate = sampleRate;
    this._startTime = checkStringOrDate(startTime);
    this.yUnit = COUNT_UNIT;
    this._sourceId = sourceId ? sourceId : FDSNSourceId.createUnknown(sampleRate);
    this._endTime_cache = null;
    this._endTime_cache_numPoints = 0;
  }
  /**
   * Y data of the seismogram. Decompresses data if needed.
   *
   * @returns y data as typed array
   */
  get y() {
    let out;
    if (this._y) {
      out = this._y;
    } else {
      if (!this.isEncoded()) {
        throw new Error("Seismogram not y as TypedArray or encoded.");
      }
      const outLen = this.numPoints;
      if (this._compressed === null) {
        throw new Error("Seismogram not y as TypedArray or encoded.");
      }
      if (this._compressed[0].compressionType === DOUBLE) {
        out = new Float64Array(outLen);
      } else if (this._compressed[0].compressionType === FLOAT) {
        out = new Float32Array(outLen);
      } else {
        out = new Int32Array(outLen);
      }
      let currIdx = 0;
      for (const c of this._compressed) {
        const cData = c.decode();
        for (let i = 0; i < c.numSamples; i++) {
          out[currIdx + i] = cData[i];
        }
        currIdx += c.numSamples;
      }
      this._y = out;
      this._compressed = null;
    }
    return out;
  }
  set y(value) {
    this._y = value;
    this._invalidate_endTime_cache();
  }
  get start() {
    return this.startTime;
  }
  set start(value) {
    this.startTime = value;
  }
  get startTime() {
    return this._startTime;
  }
  set startTime(value) {
    this._startTime = checkStringOrDate(value);
    this._invalidate_endTime_cache();
  }
  get end() {
    return this.endTime;
  }
  get endTime() {
    if (!this._endTime_cache || this._endTime_cache_numPoints !== this.numPoints) {
      this._endTime_cache_numPoints = this.numPoints;
      this._endTime_cache = this.timeOfSample(
        this._endTime_cache_numPoints - 1
      );
    }
    return this._endTime_cache;
  }
  get timeRange() {
    return Interval2.fromDateTimes(this.startTime, this.endTime);
  }
  get sampleRate() {
    return this._sampleRate;
  }
  set sampleRate(value) {
    this._sampleRate = value;
    this._invalidate_endTime_cache();
  }
  get samplePeriod() {
    return 1 / this.sampleRate;
  }
  get numPoints() {
    let out = 0;
    if (this._y) {
      out = this._y.length;
    } else if (this._compressed) {
      for (const c of this._compressed) {
        out += c.numSamples;
      }
    }
    return out;
  }
  get networkCode() {
    return this._sourceId.networkCode;
  }
  get stationCode() {
    return this._sourceId.stationCode;
  }
  get locationCode() {
    return this._sourceId.locationCode;
  }
  get channelCode() {
    return this._sourceId.formChannelCode();
  }
  /**
   * Checks if the data is encoded
   *
   * @returns true if encoded, false otherwise
   */
  isEncoded() {
    if (this._y && this._y.length > 0) {
      return false;
    } else if (this._compressed && this._compressed.length > 0) {
      return true;
    } else {
      return false;
    }
  }
  /**
   * Gets encoded data, if it is.
   *
   * @returns array of encoded data segments
   * @throws Error if data is not encoded
   */
  getEncoded() {
    const compressed = this._compressed;
    if (this.isEncoded() && compressed != null) {
      return compressed;
    } else {
      throw new Error("Data is not encoded.");
    }
  }
  yAtIndex(i) {
    if (i >= 0) {
      return this.y[i];
    } else {
      return this.y[this.numPoints + i];
    }
  }
  /**
   * Finds the min and max values of a SeismogramSegment, with an optional
   * accumulator for use with gappy data.
   *
   * @param minMaxAccumulator optional initialized accumulator as an array
   * of two numbers, min and max
   * @returns min, max as arry of length two
   */
  findMinMax(minMaxAccumulator) {
    let minAmp = Number.MAX_SAFE_INTEGER;
    let maxAmp = -1 * minAmp;
    if (minMaxAccumulator) {
      minAmp = minMaxAccumulator.min;
      maxAmp = minMaxAccumulator.max;
    }
    const yData = this.y;
    for (let n = 0; n < yData.length; n++) {
      if (minAmp > yData[n]) {
        minAmp = yData[n];
      }
      if (maxAmp < yData[n]) {
        maxAmp = yData[n];
      }
    }
    return new MinMaxable(minAmp, maxAmp);
  }
  /**
   * Time of the i-th sample, indexed from zero.
   * If i is negative, counting from end, so
   * timeOfSample(-1) is time of last data point;
   *
   * @param  i               sample index
   * @returns   time
   */
  timeOfSample(i) {
    if (i >= 0) {
      return this.startTime.plus(
        Duration3.fromMillis(1e3 * i / this.sampleRate)
      );
    } else {
      return this.startTime.plus(
        Duration3.fromMillis(1e3 * (this.numPoints + i) / this.sampleRate)
      );
    }
  }
  indexOfTime(t) {
    if (t < this.startTime || t > this.endTime.plus(Duration3.fromMillis(1e3 / this.sampleRate))) {
      return -1;
    }
    return Math.round(
      t.diff(this.startTime).toMillis() * this.sampleRate / 1e3
    );
  }
  hasCodes() {
    return isDef(this._sourceId);
  }
  /**
   * return network, station, location and channels codes as one string.
   * Uses this.channel if it exists, this.seismogram if not.
   *
   * @returns nslc codes separated by '.'
   */
  get nslc() {
    return this.codes();
  }
  get nslcId() {
    return this._sourceId.asNslc();
  }
  /**
   * return network, station, location and channels codes as one string
   *
   * @param sep separator, defaults to '.'
   * @returns nslc codes separated by sep
   */
  codes(sep = ".") {
    return (this.networkCode ? this.networkCode : "") + sep + (this.stationCode ? this.stationCode : "") + sep + (this.locationCode ? this.locationCode : "") + sep + (this.channelCode ? this.channelCode : "");
  }
  seisId() {
    const out = `${this.sourceId.toString()}_${this.startTime.toISO()}_${this.endTime.toISO()}`;
    return out.replace(/\./g, "_").replace(/:/g, "");
  }
  /**
   * return FDSN source id.
   *
   * @returns FDSN source id
   */
  get sourceId() {
    return this._sourceId;
  }
  set sourceId(sid) {
    this._sourceId = sid;
  }
  clone() {
    let out;
    if (isDef(this._y)) {
      out = this.cloneWithNewData(this._y.slice());
    } else if (this.isEncoded()) {
      out = this.cloneWithNewData(Array.from(this.getEncoded()));
    } else {
      throw new Error("no _y and no _compressed");
    }
    return out;
  }
  cloneWithNewData(clonedData, clonedStartTime = this._startTime) {
    const out = new _SeismogramSegment(
      clonedData,
      this.sampleRate,
      clonedStartTime,
      this._sourceId.clone()
    );
    out.yUnit = this.yUnit;
    return out;
  }
  cut(timeRange) {
    checkLuxonValid(timeRange);
    if (timeRange.start == null || timeRange.end == null || timeRange.end < this._startTime || timeRange.start > this.endTime) {
      return null;
    }
    let sIndex = 0;
    if (timeRange.start > this._startTime) {
      const milliDiff = timeRange.start.diff(this._startTime).toMillis();
      const offset = milliDiff * this.sampleRate / 1e3;
      sIndex = Math.floor(offset);
    }
    let eIndex = this.y.length;
    if (timeRange.end < this.endTime) {
      const milliDiff = this.endTime.diff(timeRange.end).toMillis();
      const offset = milliDiff * this.sampleRate / 1e3;
      eIndex = this.y.length - Math.floor(offset);
    }
    const cutY = this.y.slice(sIndex, eIndex);
    const out = this.cloneWithNewData(
      cutY,
      this._startTime.plus(
        Duration3.fromMillis(1e3 * sIndex / this.sampleRate)
      )
    );
    return out;
  }
  _invalidate_endTime_cache() {
    this._endTime_cache = null;
    this._endTime_cache_numPoints = 0;
  }
};

// src/seismogram.ts
var seismogram_exports = {};
__export(seismogram_exports, {
  COUNT_UNIT: () => COUNT_UNIT2,
  NonContiguousData: () => NonContiguousData,
  Seismogram: () => Seismogram,
  SeismogramDisplayData: () => SeismogramDisplayData,
  SeismogramDisplayStats: () => SeismogramDisplayStats,
  calcMinMax: () => calcMinMax,
  ensureIsSeismogram: () => ensureIsSeismogram,
  findMaxDuration: () => findMaxDuration,
  findMaxDurationOfType: () => findMaxDurationOfType,
  findMinMax: () => findMinMax,
  findMinMaxOfSDD: () => findMinMaxOfSDD,
  findMinMaxOfSeismograms: () => findMinMaxOfSeismograms,
  findMinMaxOverRelativeTimeRange: () => findMinMaxOverRelativeTimeRange,
  findMinMaxOverTimeRange: () => findMinMaxOverTimeRange,
  findStartEnd: () => findStartEnd,
  findStartEndOfSeismograms: () => findStartEndOfSeismograms,
  uniqueChannels: () => uniqueChannels,
  uniqueQuakes: () => uniqueQuakes,
  uniqueStations: () => uniqueStations2
});
import { DateTime as DateTime4, Duration as Duration4, Interval as Interval4 } from "luxon";

// src/distaz.ts
var distaz_exports = {};
__export(distaz_exports, {
  DistAzOutput: () => DistAzOutput,
  degtokm: () => degtokm,
  distaz: () => distaz,
  kmPerDeg: () => kmPerDeg,
  kmtodeg: () => kmtodeg
});
var kmPerDeg = 111.19;
function degtokm(deg) {
  return deg * kmPerDeg;
}
function kmtodeg(km) {
  return km / kmPerDeg;
}
var DistAzOutput = class {
  delta;
  az;
  baz;
  stalat;
  stalon;
  evtlat;
  evtlon;
  constructor(delta, az, baz) {
    this.delta = delta ? delta : 0;
    this.az = az ? az : 0;
    this.baz = baz ? baz : 0;
  }
  get distance() {
    return this.delta;
  }
  get distanceKm() {
    return degtokm(this.delta);
  }
  get distanceDeg() {
    return this.delta;
  }
  get azimuth() {
    return this.az;
  }
  get backazimuth() {
    return this.baz;
  }
};
function distaz(lat1, lon1, lat2, lon2) {
  if (lat1 === lat2 && lon1 === lon2) {
    const result2 = new DistAzOutput(0, 0, 0);
    result2.stalat = lat1;
    result2.stalon = lon1;
    result2.evtlat = lat2;
    result2.evtlon = lon2;
    return result2;
  }
  const rad = 2 * Math.PI / 360;
  const sph = 1 / 298.257;
  const scolat = Math.PI / 2 - Math.atan((1 - sph) * (1 - sph) * Math.tan(lat1 * rad));
  const ecolat = Math.PI / 2 - Math.atan((1 - sph) * (1 - sph) * Math.tan(lat2 * rad));
  const slon = lon1 * rad;
  const elon = lon2 * rad;
  const a = Math.sin(scolat) * Math.cos(slon);
  const b = Math.sin(scolat) * Math.sin(slon);
  const c = Math.cos(scolat);
  const d = Math.sin(slon);
  const e = -Math.cos(slon);
  const g = -c * e;
  const h = c * d;
  const k = -Math.sin(scolat);
  const aa = Math.sin(ecolat) * Math.cos(elon);
  const bb = Math.sin(ecolat) * Math.sin(elon);
  const cc = Math.cos(ecolat);
  const dd = Math.sin(elon);
  const ee = -Math.cos(elon);
  const gg = -cc * ee;
  const hh = cc * dd;
  const kk = -Math.sin(ecolat);
  const del = Math.acos(a * aa + b * bb + c * cc);
  const result_delta = del / rad;
  const baz_rhs1 = (aa - d) * (aa - d) + (bb - e) * (bb - e) + cc * cc - 2;
  const baz_rhs2 = (aa - g) * (aa - g) + (bb - h) * (bb - h) + (cc - k) * (cc - k) - 2;
  let dbaz = Math.atan2(baz_rhs1, baz_rhs2);
  if (dbaz < 0) {
    dbaz = dbaz + 2 * Math.PI;
  }
  let result_baz = dbaz / rad;
  const daz_rhs1 = (a - dd) * (a - dd) + (b - ee) * (b - ee) + c * c - 2;
  const daz_rhs2 = (a - gg) * (a - gg) + (b - hh) * (b - hh) + (c - kk) * (c - kk) - 2;
  let daz = Math.atan2(daz_rhs1, daz_rhs2);
  if (daz < 0) {
    daz = daz + 2 * Math.PI;
  }
  let result_az = daz / rad;
  if (Math.abs(result_baz - 360) < 1e-5) result_baz = 0;
  if (Math.abs(result_az - 360) < 1e-5) result_az = 0;
  const result = new DistAzOutput(result_delta, result_az, result_baz);
  result.stalat = lat1;
  result.stalon = lon1;
  result.evtlat = lat2;
  result.evtlon = lon2;
  return result;
}

// src/stationxml.ts
var stationxml_exports = {};
__export(stationxml_exports, {
  AbstractFilterType: () => AbstractFilterType,
  Author: () => Author,
  CHANNEL_CLICK_EVENT: () => CHANNEL_CLICK_EVENT,
  COUNT_UNIT_NAME: () => COUNT_UNIT_NAME,
  Channel: () => Channel,
  CoefficientsFilter: () => CoefficientsFilter,
  Comment: () => Comment,
  DataAvailability: () => DataAvailability,
  Decimation: () => Decimation,
  Equipment: () => Equipment,
  FAKE_EMPTY_XML: () => FAKE_EMPTY_XML,
  FAKE_START_DATE: () => FAKE_START_DATE,
  FIR: () => FIR,
  FIX_INVALID_STAXML: () => FIX_INVALID_STAXML,
  Gain: () => Gain,
  INVALID_NUMBER: () => INVALID_NUMBER,
  InstrumentSensitivity: () => InstrumentSensitivity,
  Network: () => Network,
  PolesZeros: () => PolesZeros,
  Response: () => Response2,
  STAML_NS: () => STAML_NS,
  STATION_CLICK_EVENT: () => STATION_CLICK_EVENT,
  Span: () => Span,
  Stage: () => Stage,
  Station: () => Station,
  allChannels: () => allChannels,
  allStations: () => allStations,
  convertToAuthor: () => convertToAuthor,
  convertToChannel: () => convertToChannel,
  convertToComment: () => convertToComment,
  convertToDataAvailability: () => convertToDataAvailability,
  convertToDecimation: () => convertToDecimation,
  convertToEquipment: () => convertToEquipment,
  convertToGain: () => convertToGain,
  convertToInstrumentSensitivity: () => convertToInstrumentSensitivity,
  convertToNetwork: () => convertToNetwork,
  convertToResponse: () => convertToResponse,
  convertToStage: () => convertToStage,
  convertToStation: () => convertToStation,
  createChannelClickEvent: () => createChannelClickEvent,
  createInterval: () => createInterval,
  createStationClickEvent: () => createStationClickEvent,
  extractComplex: () => extractComplex,
  fetchStationXml: () => fetchStationXml,
  findChannels: () => findChannels,
  parseStationXml: () => parseStationXml,
  parseUtil: () => parseUtil,
  uniqueNetworks: () => uniqueNetworks,
  uniqueSourceIds: () => uniqueSourceIds,
  uniqueStations: () => uniqueStations
});

// src/oregondsputil.ts
var oregondsputil_exports = {};
__export(oregondsputil_exports, {
  Allpass: () => Allpass,
  AnalogPrototype: () => AnalogPrototype,
  BANDPASS: () => BANDPASS,
  Butterworth: () => Butterworth,
  CDFT: () => CDFT,
  CenteredDifferentiator: () => CenteredDifferentiator,
  CenteredHilbertTransform: () => CenteredHilbertTransform,
  ChebyshevI: () => ChebyshevI,
  ChebyshevII: () => ChebyshevII,
  Complex: () => Complex,
  ComplexAnalyticSignal: () => ComplexAnalyticSignal,
  EquirippleBandpass: () => EquirippleBandpass,
  EquirippleFIRFilter: () => EquirippleFIRFilter,
  EquirippleHalfBand: () => EquirippleHalfBand,
  EquirippleHighpass: () => EquirippleHighpass,
  EquirippleLowpass: () => EquirippleLowpass,
  FIRTypeI: () => FIRTypeI,
  FIRTypeII: () => FIRTypeII,
  FIRTypeIII: () => FIRTypeIII,
  HIGHPASS: () => HIGHPASS,
  HammingWindow: () => HammingWindow,
  HanningWindow: () => HanningWindow,
  IIRFilter: () => IIRFilter,
  Interpolator: () => Interpolator,
  LOWPASS: () => LOWPASS,
  LagrangePolynomial: () => LagrangePolynomial,
  OregonDSP: () => OregonDSP,
  OverlapAdd: () => OverlapAdd,
  PassbandType: () => PassbandType,
  Polynomial: () => Polynomial,
  RDFT: () => RDFT,
  Rational: () => Rational,
  Sequence: () => Sequence,
  StaggeredDifferentiator: () => StaggeredDifferentiator,
  StaggeredHilbertTranform: () => StaggeredHilbertTranform,
  ThiranAllpass: () => ThiranAllpass,
  Window: () => Window,
  complexFromPolar: () => complexFromPolar,
  createComplex: () => createComplex
});
import OregonDSPTop from "oregondsp";
var OregonDSP = OregonDSPTop.com.oregondsp.signalProcessing;
var fft = OregonDSP.fft;
var equiripple = OregonDSP.filter.fir.equiripple;
var fir = OregonDSP.filter.fir;
var iir = OregonDSP.filter.iir;
var CDFT = fft.CDFT;
var RDFT = fft.RDFT;
var CenteredDifferentiator = equiripple.CenteredDifferentiator;
var CenteredHilbertTransform = equiripple.CenteredHilbertTransform;
var EquirippleBandpass = equiripple.EquirippleBandpass;
var EquirippleFIRFilter = equiripple.EquirippleFIRFilter;
var EquirippleHalfBand = equiripple.EquirippleHalfBand;
var EquirippleHighpass = equiripple.EquirippleHighpass;
var EquirippleLowpass = equiripple.EquirippleLowpass;
var FIRTypeI = equiripple.FIRTypeI;
var FIRTypeII = equiripple.FIRTypeII;
var FIRTypeIII = equiripple.FIRTypeIII;
var StaggeredDifferentiator = equiripple.StaggeredDifferentiator;
var StaggeredHilbertTranform = equiripple.StaggeredHilbertTranform;
var ComplexAnalyticSignal = fir.ComplexAnalyticSignal;
var Interpolator = fir.Interpolator;
var OverlapAdd = fir.OverlapAdd;
var Allpass = iir.Allpass;
var AnalogPrototype = iir.AnalogPrototype;
var Butterworth = iir.Butterworth;
var ChebyshevI = iir.ChebyshevI;
var ChebyshevII = iir.ChebyshevII;
var Complex = iir.Complex;
var IIRFilter = iir.IIRFilter;
var PassbandType = iir.PassbandType;
var ThiranAllpass = iir.ThiranAllpass;
var LOWPASS = iir.PassbandType.LOWPASS;
var BANDPASS = iir.PassbandType.BANDPASS;
var HIGHPASS = iir.PassbandType.HIGHPASS;
var LagrangePolynomial = OregonDSP.filter.LagrangePolynomial;
var Polynomial = OregonDSP.filter.Polynomial;
var Rational = OregonDSP.filter.Rational;
var HammingWindow = OregonDSP.HammingWindow;
var HanningWindow = OregonDSP.HanningWindow;
var Sequence = OregonDSP.Sequence;
var Window = OregonDSP.Window;
function complexFromPolar(amp, phase) {
  const real = amp * Math.cos(phase);
  const imag = amp * Math.sin(phase);
  return new OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.Complex(
    real,
    imag
  );
}
function createComplex(real, imag) {
  return new OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.Complex(
    real,
    imag
  );
}

// src/stationxml.ts
import { DateTime as DateTime3, Interval as Interval3 } from "luxon";
var STAML_NS = "http://www.fdsn.org/xml/station/1";
var COUNT_UNIT_NAME = "count";
var FIX_INVALID_STAXML = true;
var INVALID_NUMBER = -99999;
var FAKE_START_DATE = DateTime3.fromISO("1900-01-01T00:00:00Z");
var FAKE_EMPTY_XML = '<?xml version="1.0" encoding="ISO-8859-1"?> <FDSNStationXML xmlns="http://www.fdsn.org/xml/station/1" schemaVersion="1.0" xsi:schemaLocation="http://www.fdsn.org/xml/station/1 http://www.fdsn.org/xml/station/fdsn-station-1.0.xsd" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:iris="http://www.fdsn.org/xml/station/1/iris"> </FDSNStationXML>';
var CHANNEL_CLICK_EVENT = "channelclick";
var STATION_CLICK_EVENT = "stationclick";
function createChannelClickEvent(sta, mouseclick) {
  const detail = {
    mouseevent: mouseclick,
    channel: sta
  };
  return new CustomEvent(CHANNEL_CLICK_EVENT, { detail });
}
function createStationClickEvent(sta, mouseclick) {
  const detail = {
    mouseevent: mouseclick,
    station: sta
  };
  return new CustomEvent(STATION_CLICK_EVENT, { detail });
}
var Network = class {
  networkCode;
  _startDate;
  _endDate;
  restrictedStatus;
  description;
  totalNumberStations;
  stations;
  constructor(networkCode) {
    this.networkCode = networkCode;
    this._startDate = FAKE_START_DATE;
    this._endDate = null;
    this.description = "";
    this.restrictedStatus = "";
    this.stations = [];
    this.totalNumberStations = null;
  }
  get sourceId() {
    return new NetworkSourceId(this.networkCode ? this.networkCode : "");
  }
  get startDate() {
    return this._startDate;
  }
  set startDate(value) {
    this._startDate = checkStringOrDate(value);
  }
  get endDate() {
    return this._endDate;
  }
  set endDate(value) {
    if (!isDef(value)) {
      this._endDate = null;
    } else {
      this._endDate = checkStringOrDate(value);
    }
  }
  get timeRange() {
    return createInterval(this.startDate, this.endDate);
  }
  codes() {
    return this.networkCode;
  }
  isActiveAt(d) {
    if (!isDef(d)) {
      d = DateTime3.utc();
    }
    return this.timeRange.contains(d);
  }
  isTempNet() {
    const first = this.networkCode.charAt(0);
    return first === "X" || first === "Y" || first === "Z" || first >= "0" && first <= "9";
  }
};
var Station = class {
  network;
  stationCode;
  sourceID;
  /** @private */
  _startDate;
  /** @private */
  _endDate;
  restrictedStatus;
  name;
  latitude;
  longitude;
  elevation;
  waterLevel;
  comments;
  equipmentList;
  dataAvailability;
  identifierList;
  description;
  geology;
  vault;
  channels;
  constructor(network, stationCode) {
    this.network = network;
    this.name = "";
    this.description = "";
    this.sourceID = null;
    this.restrictedStatus = "";
    this._startDate = FAKE_START_DATE;
    this._endDate = null;
    this.stationCode = stationCode;
    this.channels = [];
    this.latitude = INVALID_NUMBER;
    this.longitude = INVALID_NUMBER;
    this.elevation = 0;
    this.waterLevel = null;
    this.comments = [];
    this.equipmentList = [];
    this.dataAvailability = null;
    this.geology = "";
    this.vault = "";
    this.identifierList = [];
  }
  get sourceId() {
    return new StationSourceId(
      this.networkCode ? this.networkCode : "",
      this.stationCode ? this.stationCode : ""
    );
  }
  get startDate() {
    return this._startDate;
  }
  set startDate(value) {
    this._startDate = checkStringOrDate(value);
  }
  get endDate() {
    return this._endDate;
  }
  set endDate(value) {
    if (!isDef(value)) {
      this._endDate = null;
    } else {
      this._endDate = checkStringOrDate(value);
    }
  }
  get timeRange() {
    return createInterval(this.startDate, this.endDate);
  }
  get networkCode() {
    return this.network.networkCode;
  }
  isActiveAt(d) {
    if (!isDef(d)) {
      d = DateTime3.utc();
    }
    return this.timeRange.contains(d);
  }
  codes(sep = ".") {
    return this.network.codes() + sep + this.stationCode;
  }
};
var Channel = class {
  station;
  /** @private */
  _locationCode;
  channelCode;
  /** @private */
  _sourceId;
  /** @private */
  _startDate;
  /** @private */
  _endDate;
  restrictedStatus;
  latitude;
  longitude;
  elevation;
  depth;
  azimuth;
  dip;
  sampleRate;
  waterLevel = null;
  comments = [];
  equipmentList = [];
  dataAvailability = null;
  identifierList = [];
  description = "";
  response;
  sensor;
  preamplifier;
  datalogger;
  constructor(station, channelCode, locationCode) {
    this.station = station;
    this._startDate = FAKE_START_DATE;
    this._endDate = null;
    this.response = null;
    this.sensor = null;
    this.preamplifier = null;
    this.datalogger = null;
    this.restrictedStatus = "";
    this.azimuth = INVALID_NUMBER;
    this.dip = INVALID_NUMBER;
    this.latitude = INVALID_NUMBER;
    this.longitude = INVALID_NUMBER;
    this.depth = 0;
    this.elevation = 0;
    this.sampleRate = 0;
    if (channelCode.length !== 3) {
      throw new Error(`Channel code must be 3 chars: ${channelCode}`);
    }
    this.channelCode = channelCode;
    this._locationCode = locationCode;
    if (!locationCode) {
      this._locationCode = "";
    }
    if (!(this._locationCode.length === 2 || this._locationCode.length === 0)) {
      throw new Error(
        `locationCode must be 2 chars, or empty: "${locationCode}"`
      );
    }
  }
  get sourceId() {
    if (this._sourceId) {
      return this._sourceId;
    }
    return FDSNSourceId.fromNslc(
      this.networkCode,
      this.stationCode,
      this.locationCode,
      this.channelCode
    );
  }
  get nslcId() {
    return new NslcId(
      this.networkCode ? this.networkCode : "",
      this.stationCode ? this.stationCode : "",
      this.locationCode && this.locationCode !== "--" ? this.locationCode : "",
      this.channelCode ? this.channelCode : ""
    );
  }
  get startDate() {
    return this._startDate;
  }
  set startDate(value) {
    this._startDate = checkStringOrDate(value);
  }
  get endDate() {
    return this._endDate;
  }
  set endDate(value) {
    if (!isDef(value)) {
      this._endDate = null;
    } else {
      this._endDate = checkStringOrDate(value);
    }
  }
  get timeRange() {
    return createInterval(this.startDate, this.endDate);
  }
  get locationCode() {
    return this._locationCode;
  }
  set locationCode(value) {
    this._locationCode = value;
    if (!value) {
      this._locationCode = "";
    }
  }
  get stationCode() {
    return this.station.stationCode;
  }
  get networkCode() {
    return this.station.networkCode;
  }
  /**
   * Checks if this channel has sensitivity defined, within the response.
   *
   * @returns          true if instrumentSensitivity exits
   */
  hasInstrumentSensitivity() {
    return isDef(this.response) && isDef(this.response.instrumentSensitivity);
  }
  set instrumentSensitivity(value) {
    if (!isDef(this.response)) {
      this.response = new Response2(value);
    } else {
      this.response.instrumentSensitivity = value;
    }
  }
  get instrumentSensitivity() {
    if (isDef(this.response) && isDef(this.response.instrumentSensitivity)) {
      return this.response.instrumentSensitivity;
    } else {
      throw new Error("no Response or InstrumentSensitivity defined");
    }
  }
  /**
   * return network, station, location and channels codes as one string.
   *
   * @returns net.sta.loc.chan
   */
  get nslc() {
    return this.codes();
  }
  /**
   * return network, station, location and channels codes as one string.
   *
   * @param sep separator, defaults to dot '.'
   * @returns net.sta.loc.chan
   */
  codes(sep = ".") {
    return this.station.codes(sep) + sep + this.locationCode + sep + this.channelCode;
  }
  isActiveAt(d) {
    if (!isDef(d)) {
      d = DateTime3.utc();
    }
    return this.timeRange.contains(d);
  }
};
var InstrumentSensitivity = class {
  sensitivity;
  frequency;
  inputUnits;
  outputUnits;
  constructor(sensitivity, frequency, inputUnits, outputUnits) {
    this.sensitivity = sensitivity;
    this.frequency = frequency;
    this.inputUnits = inputUnits;
    this.outputUnits = outputUnits;
  }
};
var Equipment = class {
  resourceId;
  type;
  description;
  manufacturer;
  vendor;
  model;
  serialNumber;
  installationDate;
  removalDate;
  calibrationDateList;
  constructor() {
    this.resourceId = "";
    this.type = "";
    this.description = "";
    this.manufacturer = "";
    this.vendor = "";
    this.model = "";
    this.serialNumber = "";
    this.installationDate = null;
    this.removalDate = null;
    this.calibrationDateList = [];
  }
};
var Response2 = class {
  instrumentSensitivity;
  stages;
  constructor(instrumentSensitivity, stages) {
    if (instrumentSensitivity) {
      this.instrumentSensitivity = instrumentSensitivity;
    } else {
      this.instrumentSensitivity = null;
    }
    if (stages) {
      this.stages = stages;
    } else {
      this.stages = [];
    }
  }
};
var Stage = class {
  filter;
  decimation;
  gain;
  constructor(filter, decimation, gain) {
    this.filter = filter;
    this.decimation = decimation;
    this.gain = gain;
  }
};
var AbstractFilterType = class {
  inputUnits;
  outputUnits;
  name;
  description;
  constructor(inputUnits, outputUnits) {
    this.inputUnits = inputUnits;
    this.outputUnits = outputUnits;
    this.description = "";
    this.name = "";
  }
};
var PolesZeros = class extends AbstractFilterType {
  pzTransferFunctionType;
  normalizationFactor;
  normalizationFrequency;
  zeros;
  poles;
  constructor(inputUnits, outputUnits) {
    super(inputUnits, outputUnits);
    this.pzTransferFunctionType = "";
    this.normalizationFactor = 1;
    this.normalizationFrequency = 0;
    this.zeros = new Array(0);
    this.poles = new Array(0);
  }
};
var FIR = class extends AbstractFilterType {
  symmetry;
  numerator;
  constructor(inputUnits, outputUnits) {
    super(inputUnits, outputUnits);
    this.symmetry = "none";
    this.numerator = [1];
  }
};
var CoefficientsFilter = class extends AbstractFilterType {
  cfTransferFunction;
  numerator;
  denominator;
  constructor(inputUnits, outputUnits) {
    super(inputUnits, outputUnits);
    this.cfTransferFunction = "";
    this.numerator = [1];
    this.denominator = new Array(0);
  }
};
var Decimation = class {
  inputSampleRate;
  factor;
  offset;
  delay;
  correction;
  constructor(inputSampleRate, factor) {
    this.inputSampleRate = inputSampleRate;
    this.factor = factor;
  }
};
var Gain = class {
  value;
  frequency;
  constructor(value, frequency) {
    this.value = value;
    this.frequency = frequency;
  }
};
var Span = class {
  interval;
  numberSegments = 0;
  maximumTimeTear;
  constructor(interval) {
    this.maximumTimeTear = null;
    this.interval = interval;
  }
};
var DataAvailability = class {
  extent;
  spanList;
  constructor() {
    this.extent = null;
    this.spanList = [];
  }
};
var Comment = class {
  id = null;
  subject = null;
  value;
  beginEffectiveTime = null;
  endEffectiveTime = null;
  authorList = [];
  constructor(value) {
    this.value = value;
  }
};
var Author = class {
  name = null;
  agency = null;
  email = null;
  phone = null;
};
function parseStationXml(rawXml) {
  const top = rawXml.documentElement;
  if (!top) {
    throw new Error("No documentElement in XML");
  }
  const netArray = Array.from(top.getElementsByTagNameNS(STAML_NS, "Network"));
  const out = [];
  for (const n of netArray) {
    out.push(convertToNetwork(n));
  }
  return out;
}
function convertToNetwork(xml) {
  let netCode = "";
  try {
    netCode = _requireAttribute(xml, "code");
    const out = new Network(netCode);
    out.startDate = _requireAttribute(xml, "startDate");
    const rs = _grabAttribute(xml, "restrictedStatus");
    if (isNonEmptyStringArg(rs)) {
      out.restrictedStatus = rs;
    }
    const desc = _grabFirstElText(xml, "Description");
    if (isNonEmptyStringArg(desc)) {
      out.description = desc;
    }
    if (_grabAttribute(xml, "endDate")) {
      out.endDate = _grabAttribute(xml, "endDate");
    }
    const totSta = xml.getElementsByTagNameNS(STAML_NS, "TotalNumberStations");
    if (totSta && totSta.length > 0) {
      out.totalNumberStations = _grabFirstElInt(xml, "TotalNumberStations");
    }
    const staArray = Array.from(
      xml.getElementsByTagNameNS(STAML_NS, "Station")
    );
    const stations = [];
    for (const s of staArray) {
      stations.push(convertToStation(out, s));
    }
    out.stations = stations;
    return out;
  } catch (err) {
    throw reErrorWithMessage(err, netCode);
  }
}
function convertToStation(network, xml) {
  let staCode = "";
  try {
    staCode = _requireAttribute(xml, "code");
    if (!isNonEmptyStringArg(staCode)) {
      throw new Error("station code missing in station!");
    }
    const out = new Station(network, staCode);
    out.startDate = _requireAttribute(xml, "startDate");
    const rs = _grabAttribute(xml, "restrictedStatus");
    if (isNonEmptyStringArg(rs)) {
      out.restrictedStatus = rs;
    }
    const lat = _grabFirstElFloat(xml, "Latitude");
    if (isNumArg(lat)) {
      out.latitude = lat;
    }
    const lon = _grabFirstElFloat(xml, "Longitude");
    if (isNumArg(lon)) {
      out.longitude = lon;
    }
    const elev = _grabFirstElFloat(xml, "Elevation");
    if (isNumArg(elev)) {
      out.elevation = elev;
    }
    const waterLevel = _grabFirstElFloat(xml, "WaterLevel");
    if (isNumArg(waterLevel)) {
      out.waterLevel = waterLevel;
    }
    const vault = _grabFirstElText(xml, "Vault");
    if (isStringArg(vault)) {
      out.vault = vault;
    }
    const geology = _grabFirstElText(xml, "Geology");
    if (isStringArg(geology)) {
      out.geology = geology;
    }
    const name = _grabFirstElText(_grabFirstEl(xml, "Site"), "Name");
    if (isStringArg(name)) {
      out.name = name;
    }
    const endDate = _grabAttribute(xml, "endDate");
    if (isDef(endDate)) {
      out.endDate = _grabAttribute(xml, "endDate");
    }
    const description = _grabFirstElText(xml, "Description");
    if (isDef(description)) {
      out.description = description;
    }
    const identifierList = Array.from(
      xml.getElementsByTagNameNS(STAML_NS, "Identifier")
    );
    out.identifierList = identifierList.map((el) => {
      return el.textContent ? el.textContent : "";
    });
    const dataAvailEl = _grabFirstEl(xml, "DataAvailability");
    if (isDef(dataAvailEl)) {
      out.dataAvailability = convertToDataAvailability(dataAvailEl);
    }
    const commentArray = Array.from(
      xml.getElementsByTagNameNS(STAML_NS, "Comment")
    );
    const comments = [];
    for (const c of commentArray) {
      comments.push(convertToComment(c));
    }
    out.comments = comments;
    const equipmentArray = Array.from(
      xml.getElementsByTagNameNS(STAML_NS, "Equipment")
    );
    const equipmentList = [];
    for (const c of equipmentArray) {
      equipmentList.push(convertToEquipment(c));
    }
    out.equipmentList = equipmentList;
    const chanArray = Array.from(
      xml.getElementsByTagNameNS(STAML_NS, "Channel")
    );
    const channels = [];
    for (const c of chanArray) {
      channels.push(convertToChannel(out, c));
    }
    out.channels = channels;
    return out;
  } catch (err) {
    throw reErrorWithMessage(err, staCode);
  }
}
function convertToChannel(station, xml) {
  let locCode = "";
  const chanCode = "";
  try {
    locCode = _grabAttribute(xml, "locationCode");
    if (!isNonEmptyStringArg(locCode)) {
      locCode = "";
    }
    const chanCode2 = _requireAttribute(xml, "code");
    const out = new Channel(station, chanCode2, locCode);
    out.startDate = checkStringOrDate(_requireAttribute(xml, "startDate"));
    const rs = _grabAttribute(xml, "restrictedStatus");
    if (isNonEmptyStringArg(rs)) {
      out.restrictedStatus = rs;
    }
    const lat = _grabFirstElFloat(xml, "Latitude");
    if (isNumArg(lat)) {
      out.latitude = lat;
    }
    const lon = _grabFirstElFloat(xml, "Longitude");
    if (isNumArg(lon)) {
      out.longitude = lon;
    }
    const elev = _grabFirstElFloat(xml, "Elevation");
    if (isNumArg(elev)) {
      out.elevation = elev;
    }
    const depth = _grabFirstElFloat(xml, "Depth");
    if (isNumArg(depth)) {
      out.depth = depth;
    }
    const waterLevel = _grabFirstElFloat(xml, "WaterLevel");
    if (isNumArg(waterLevel)) {
      out.waterLevel = waterLevel;
    }
    const azimuth = _grabFirstElFloat(xml, "Azimuth");
    if (isNumArg(azimuth)) {
      out.azimuth = azimuth;
    }
    const dip = _grabFirstElFloat(xml, "Dip");
    if (isNumArg(dip)) {
      out.dip = dip;
    }
    const desc = _grabFirstElText(xml, "Description");
    if (desc) {
      out.description = desc;
    }
    const sampleRate = _grabFirstElFloat(xml, "SampleRate");
    if (isNumArg(sampleRate)) {
      out.sampleRate = sampleRate;
    }
    if (_grabAttribute(xml, "endDate")) {
      out.endDate = _grabAttribute(xml, "endDate");
    }
    const sensor = xml.getElementsByTagNameNS(STAML_NS, "Sensor");
    if (sensor && sensor.length > 0) {
      const sensorTmp = sensor.item(0);
      if (isDef(sensorTmp)) {
        out.sensor = convertToEquipment(sensorTmp);
      }
    }
    const preamp = xml.getElementsByTagNameNS(STAML_NS, "PreAmplifier");
    if (preamp && preamp.length > 0) {
      const preampTmp = sensor.item(0);
      if (isDef(preampTmp)) {
        out.preamplifier = convertToEquipment(preampTmp);
      }
    }
    const datalogger = xml.getElementsByTagNameNS(STAML_NS, "DataLogger");
    if (datalogger && datalogger.length > 0) {
      const dataloggerTmp = sensor.item(0);
      if (isDef(dataloggerTmp)) {
        out.datalogger = convertToEquipment(dataloggerTmp);
      }
    }
    const description = _grabFirstElText(xml, "Description");
    if (isDef(description)) {
      out.description = description;
    }
    const identifierList = Array.from(
      xml.getElementsByTagNameNS(STAML_NS, "Identifier")
    );
    out.identifierList = identifierList.map((el) => {
      return el.textContent ? el.textContent : "";
    });
    const dataAvailEl = _grabFirstEl(xml, "DataAvailability");
    if (isDef(dataAvailEl)) {
      out.dataAvailability = convertToDataAvailability(dataAvailEl);
    }
    const commentArray = Array.from(
      xml.getElementsByTagNameNS(STAML_NS, "Comment")
    );
    const comments = [];
    for (const c of commentArray) {
      comments.push(convertToComment(c));
    }
    out.comments = comments;
    const equipmentArray = Array.from(
      xml.getElementsByTagNameNS(STAML_NS, "Equipment")
    );
    const equipmentList = [];
    for (const c of equipmentArray) {
      equipmentList.push(convertToEquipment(c));
    }
    out.equipmentList = equipmentList;
    const responseXml = xml.getElementsByTagNameNS(STAML_NS, "Response");
    if (responseXml && responseXml.length > 0) {
      const r = responseXml.item(0);
      if (r) {
        out.response = convertToResponse(r);
      }
    }
    return out;
  } catch (err) {
    throw reErrorWithMessage(err, `${locCode}.${chanCode}`);
  }
}
function convertToDataAvailability(xml) {
  const out = new DataAvailability();
  const extent = _grabFirstEl(xml, "Extent");
  if (extent && "start" in extent && "end" in extent) {
    const s = _grabAttribute(extent, "start");
    const e = _grabAttribute(extent, "end");
    if (s && e) {
      out.extent = Interval3.fromDateTimes(
        DateTime3.fromISO(s),
        DateTime3.fromISO(e)
      );
    }
  }
  const spanArray = Array.from(xml.getElementsByTagNameNS(STAML_NS, "Span"));
  const spanList = [];
  for (const c of spanArray) {
    const s = _grabAttribute(c, "start");
    const e = _grabAttribute(c, "end");
    if (s && e) {
      const span = new Span(
        Interval3.fromDateTimes(DateTime3.fromISO(s), DateTime3.fromISO(e))
      );
      const numSeg = _grabAttribute(c, "numberSegments");
      if (numSeg) {
        span.numberSegments = parseInt(numSeg);
      }
      const maxTear = _grabAttribute(c, "maximumTimeTear");
      if (maxTear) {
        span.maximumTimeTear = parseFloat(maxTear);
      }
      spanList.push(span);
    }
  }
  out.spanList = spanList;
  return out;
}
function convertToComment(xml) {
  let val = _grabFirstElText(xml, "Value");
  if (!val) {
    val = "";
  }
  const out = new Comment(val);
  const id = _grabAttribute(xml, "id");
  if (id) {
    out.id = id;
  }
  const subject = _grabAttribute(xml, "subject");
  if (subject) {
    out.subject = subject;
  }
  const b = _grabFirstElText(xml, "BeginEffectiveTime");
  if (b) {
    out.beginEffectiveTime = DateTime3.fromISO(b);
  }
  const e = _grabFirstElText(xml, "EndEffectiveTime");
  if (e) {
    out.endEffectiveTime = DateTime3.fromISO(e);
  }
  const authList = Array.from(xml.getElementsByTagNameNS(STAML_NS, "Author"));
  out.authorList = authList.map((aEl) => convertToAuthor(aEl));
  return out;
}
function convertToAuthor(xml) {
  const out = new Author();
  const name = _grabFirstElText(xml, "Name");
  if (name) {
    out.name = name;
  }
  const agency = _grabFirstElText(xml, "Agency");
  if (agency) {
    out.agency = agency;
  }
  const phEl = _grabFirstEl(xml, "Phone");
  if (phEl) {
    out.phone = `${_grabFirstElText(phEl, "CountryCode")}-${_grabFirstElText(phEl, "AreaCode")}-${_grabFirstElText(phEl, "PhoneNumber")}`;
  }
  return out;
}
function convertToEquipment(xml) {
  const out = new Equipment();
  let val;
  val = _grabFirstElText(xml, "Type");
  if (isNonEmptyStringArg(val)) {
    out.type = val;
  }
  val = _grabFirstElText(xml, "Description");
  if (isNonEmptyStringArg(val)) {
    out.description = val;
  }
  val = _grabFirstElText(xml, "Manufacturer");
  if (isNonEmptyStringArg(val)) {
    out.manufacturer = val;
  }
  val = _grabFirstElText(xml, "Vendor");
  if (isNonEmptyStringArg(val)) {
    out.vendor = val;
  }
  val = _grabFirstElText(xml, "Model");
  if (isNonEmptyStringArg(val)) {
    out.model = val;
  }
  val = _grabFirstElText(xml, "SerialNumber");
  if (isNonEmptyStringArg(val)) {
    out.serialNumber = val;
  }
  val = _grabFirstElText(xml, "InstallationDate");
  if (isNonEmptyStringArg(val)) {
    out.installationDate = checkStringOrDate(val);
  }
  val = _grabFirstElText(xml, "RemovalDate");
  if (isNonEmptyStringArg(val)) {
    out.removalDate = checkStringOrDate(val);
  }
  const calibXml = Array.from(
    xml.getElementsByTagNameNS(STAML_NS, "CalibrationDate")
  );
  out.calibrationDateList = [];
  for (const cal of calibXml) {
    if (isDef(cal.textContent)) {
      const d = checkStringOrDate(cal.textContent);
      if (isDef(d)) {
        out.calibrationDateList.push(d);
      }
    }
  }
  return out;
}
function convertToResponse(responseXml) {
  let out = new Response2();
  const inst = responseXml.getElementsByTagNameNS(
    STAML_NS,
    "InstrumentSensitivity"
  );
  if (inst && inst.item(0)) {
    const i = inst.item(0);
    if (i) {
      out = new Response2(convertToInstrumentSensitivity(i));
    }
  }
  if (!isDef(out)) {
    out = new Response2();
  }
  const xmlStages = responseXml.getElementsByTagNameNS(STAML_NS, "Stage");
  if (xmlStages && xmlStages.length > 0) {
    const jsStages = Array.from(xmlStages).map(function(stageXml) {
      return convertToStage(stageXml);
    });
    out.stages = jsStages;
  }
  return out;
}
function convertToInstrumentSensitivity(xml) {
  const sensitivity = _grabFirstElFloat(xml, "Value");
  const frequency = _grabFirstElFloat(xml, "Frequency");
  const inputUnits = _grabFirstElText(_grabFirstEl(xml, "InputUnits"), "Name");
  let outputUnits = _grabFirstElText(_grabFirstEl(xml, "OutputUnits"), "Name");
  if (FIX_INVALID_STAXML && !isDef(outputUnits)) {
    outputUnits = COUNT_UNIT_NAME;
  }
  if (!(isDef(sensitivity) && isDef(frequency) && isDef(inputUnits) && isDef(outputUnits))) {
    throw new Error(
      `Not all elements of Sensitivity exist: ${sensitivity} ${frequency} ${inputUnits} ${outputUnits}`
    );
  }
  return new InstrumentSensitivity(
    sensitivity,
    frequency,
    inputUnits,
    outputUnits
  );
}
function convertToStage(stageXml) {
  const subEl = stageXml.firstElementChild;
  let filter = null;
  if (!subEl) {
    throw new Error("Stage element has no child elements");
  } else if (stageXml.childElementCount === 1 && subEl.localName === "StageGain") {
  } else {
    const inputUnits = _grabFirstElText(
      _grabFirstEl(stageXml, "InputUnits"),
      "Name"
    );
    const outputUnits = _grabFirstElText(
      _grabFirstEl(stageXml, "OutputUnits"),
      "Name"
    );
    if (!isNonEmptyStringArg(inputUnits)) {
      throw new Error("Stage inputUnits required");
    }
    if (!isNonEmptyStringArg(outputUnits)) {
      throw new Error("Stage outputUnits required");
    }
    if (subEl.localName === "PolesZeros") {
      const pzFilter = new PolesZeros(inputUnits, outputUnits);
      const pzt = _grabFirstElText(stageXml, "PzTransferFunctionType");
      if (isNonEmptyStringArg(pzt)) {
        pzFilter.pzTransferFunctionType = pzt;
      }
      const nfa = _grabFirstElFloat(stageXml, "NormalizationFactor");
      if (isNumArg(nfa)) {
        pzFilter.normalizationFactor = nfa;
      }
      const nfr = _grabFirstElFloat(stageXml, "NormalizationFrequency");
      if (isNumArg(nfr)) {
        pzFilter.normalizationFrequency = nfr;
      }
      const zeros = Array.from(
        stageXml.getElementsByTagNameNS(STAML_NS, "Zero")
      ).map(function(zeroEl) {
        return extractComplex(zeroEl);
      });
      const poles = Array.from(
        stageXml.getElementsByTagNameNS(STAML_NS, "Pole")
      ).map(function(poleEl) {
        return extractComplex(poleEl);
      });
      pzFilter.zeros = zeros;
      pzFilter.poles = poles;
      filter = pzFilter;
    } else if (subEl.localName === "Coefficients") {
      const coeffXml = subEl;
      const cFilter = new CoefficientsFilter(inputUnits, outputUnits);
      const cft = _grabFirstElText(coeffXml, "CfTransferFunctionType");
      if (isNonEmptyStringArg(cft)) {
        cFilter.cfTransferFunction = cft;
      }
      cFilter.numerator = Array.from(
        coeffXml.getElementsByTagNameNS(STAML_NS, "Numerator")
      ).map(function(numerEl) {
        return isNonEmptyStringArg(numerEl.textContent) ? parseFloat(numerEl.textContent) : null;
      }).filter(isDef);
      cFilter.denominator = Array.from(
        coeffXml.getElementsByTagNameNS(STAML_NS, "Denominator")
      ).map(function(denomEl) {
        return isNonEmptyStringArg(denomEl.textContent) ? parseFloat(denomEl.textContent) : null;
      }).filter(isDef);
      filter = cFilter;
    } else if (subEl.localName === "ResponseList") {
      throw new Error("ResponseList not supported: ");
    } else if (subEl.localName === "FIR") {
      const firXml = subEl;
      const firFilter = new FIR(inputUnits, outputUnits);
      const s = _grabFirstElText(firXml, "Symmetry");
      if (isNonEmptyStringArg(s)) {
        firFilter.symmetry = s;
      }
      firFilter.numerator = Array.from(
        firXml.getElementsByTagNameNS(STAML_NS, "NumeratorCoefficient")
      ).map(function(numerEl) {
        return isNonEmptyStringArg(numerEl.textContent) ? parseFloat(numerEl.textContent) : null;
      }).filter(isDef);
      filter = firFilter;
    } else if (subEl.localName === "Polynomial") {
      throw new Error("Polynomial not supported: ");
    } else if (subEl.localName === "StageGain") {
    } else {
      throw new Error("Unknown Stage type: " + subEl.localName);
    }
    if (filter) {
      const description = _grabFirstElText(subEl, "Description");
      if (isNonEmptyStringArg(description)) {
        filter.description = description;
      }
      if (subEl.hasAttribute("name")) {
        const n = _grabAttribute(subEl, "name");
        if (isNonEmptyStringArg(n)) {
          filter.name = n;
        }
      }
    }
  }
  const decimationXml = _grabFirstEl(stageXml, "Decimation");
  let decimation = null;
  if (decimationXml) {
    decimation = convertToDecimation(decimationXml);
  }
  const gainXml = _grabFirstEl(stageXml, "StageGain");
  let gain = null;
  if (gainXml) {
    gain = convertToGain(gainXml);
  } else {
    throw new Error(
      "Did not find Gain in stage number " + stringify(_grabAttribute(stageXml, "number"))
    );
  }
  const out = new Stage(filter, decimation, gain);
  return out;
}
function convertToDecimation(decXml) {
  let out;
  const insr = _grabFirstElFloat(decXml, "InputSampleRate");
  const fac = _grabFirstElInt(decXml, "Factor");
  if (isNumArg(insr) && isNumArg(fac)) {
    out = new Decimation(insr, fac);
  } else {
    throw new Error(
      `Decimation without InputSampleRate and Factor: ${insr} ${fac}`
    );
  }
  out.offset = _grabFirstElInt(decXml, "Offset");
  out.delay = _grabFirstElFloat(decXml, "Delay");
  out.correction = _grabFirstElFloat(decXml, "Correction");
  return out;
}
function convertToGain(gainXml) {
  let out;
  const v = _grabFirstElFloat(gainXml, "Value");
  const f = _grabFirstElFloat(gainXml, "Frequency");
  if (isNumArg(v) && isNumArg(f)) {
    out = new Gain(v, f);
  } else {
    throw new Error(`Gain does not have value and frequency: ${v} ${f}`);
  }
  return out;
}
function createInterval(start, end) {
  if (end) {
    return Interval3.fromDateTimes(start, end);
  } else {
    return Interval3.fromDateTimes(start, WAY_FUTURE);
  }
}
function extractComplex(el) {
  const re = _grabFirstElFloat(el, "Real");
  const im = _grabFirstElFloat(el, "Imaginary");
  if (isNumArg(re) && isNumArg(im)) {
    return new Complex(re, im);
  } else {
    throw new Error(`Both Real and Imaginary required: ${re} ${im}`);
  }
}
function* allStations(networks) {
  for (const n of networks) {
    for (const s of n.stations) {
      yield s;
    }
  }
}
function* allChannels(networks) {
  for (const s of allStations(networks)) {
    for (const c of s.channels) {
      yield c;
    }
  }
}
function* findChannels(networks, netCode, staCode, locCode, chanCode) {
  const netRE = new RegExp(`^${netCode}$`);
  const staRE = new RegExp(`^${staCode}$`);
  const locRE = new RegExp(`^${locCode}$`);
  const chanRE = new RegExp(`^${chanCode}$`);
  for (const n of networks.filter((n2) => netRE.test(n2.networkCode))) {
    for (const s of n.stations.filter((s2) => staRE.test(s2.stationCode))) {
      for (const c of s.channels.filter(
        (c2) => locRE.test(c2.locationCode) && chanRE.test(c2.channelCode)
      )) {
        yield c;
      }
    }
  }
}
function uniqueSourceIds(channelList) {
  const out = /* @__PURE__ */ new Map();
  for (const c of channelList) {
    if (c) {
      out.set(c.sourceId.toString(), c.sourceId);
    }
  }
  return Array.from(out.values()).sort(SourceIdSorter);
}
function uniqueStations(channelList) {
  const out = /* @__PURE__ */ new Set();
  for (const c of channelList) {
    if (c) {
      out.add(c.station);
    }
  }
  return Array.from(out.values());
}
function uniqueNetworks(channelList) {
  const out = /* @__PURE__ */ new Set();
  for (const c of channelList) {
    if (c) {
      out.add(c.station.network);
    }
  }
  return Array.from(out.values());
}
function fetchStationXml(url, timeoutSec2 = 10, nodata = 204) {
  const fetchInit = defaultFetchInitObj(XML_MIME);
  return doFetchWithTimeout(url, fetchInit, timeoutSec2 * 1e3).then((response) => {
    if (response.status === 200) {
      return response.text();
    } else if (response.status === 204 || isDef(nodata) && response.status === nodata) {
      return FAKE_EMPTY_XML;
    } else {
      throw new Error(`Status not successful: ${response.status}`);
    }
  }).then(function(rawXmlText) {
    return new DOMParser().parseFromString(rawXmlText, XML_MIME);
  }).then((rawXml) => {
    return parseStationXml(rawXml);
  });
}
var _grabFirstEl = function(xml, tagName) {
  let out = null;
  if (xml instanceof Element) {
    const el = xml.getElementsByTagName(tagName);
    if (isObject(el) && el.length > 0) {
      const e = el.item(0);
      if (e) {
        out = e;
      }
    }
  }
  return out;
};
var _grabFirstElText = function _grabFirstElText2(xml, tagName) {
  let out = null;
  const el = _grabFirstEl(xml, tagName);
  if (el instanceof Element) {
    out = el.textContent;
  }
  return out;
};
var _grabFirstElFloat = function _grabFirstElFloat2(xml, tagName) {
  let out = null;
  const elText = _grabFirstElText(xml, tagName);
  if (isStringArg(elText)) {
    out = parseFloat(elText);
  }
  return out;
};
var _grabFirstElInt = function _grabFirstElInt2(xml, tagName) {
  let out = null;
  const elText = _grabFirstElText(xml, tagName);
  if (isStringArg(elText)) {
    out = parseInt(elText);
  }
  return out;
};
var _grabAttribute = function _grabAttribute2(xml, tagName) {
  let out = null;
  if (xml instanceof Element) {
    const a = xml.getAttribute(tagName);
    if (isStringArg(a)) {
      out = a;
    }
  }
  return out;
};
var _requireAttribute = function _requireAttribute2(xml, tagName) {
  const out = _grabAttribute(xml, tagName);
  if (typeof out !== "string") {
    throw new Error(`Attribute ${tagName} not found.`);
  }
  return out;
};
var _grabAttributeNS = function(xml, namespace, tagName) {
  let out = null;
  if (xml instanceof Element) {
    const a = xml.getAttributeNS(namespace, tagName);
    if (isStringArg(a)) {
      out = a;
    }
  }
  return out;
};
var parseUtil = {
  _grabFirstEl,
  _grabFirstElText,
  _grabFirstElFloat,
  _grabFirstElInt,
  _grabAttribute,
  _requireAttribute,
  _grabAttributeNS
};

// src/seismogram.ts
var COUNT_UNIT2 = "count";
var Seismogram = class _Seismogram {
  _segmentArray;
  _interval;
  _y;
  constructor(segmentArray) {
    this._y = null;
    if (Array.isArray(segmentArray) && segmentArray[0] instanceof SeismogramSegment) {
      this._segmentArray = segmentArray;
    } else if (segmentArray instanceof SeismogramSegment) {
      this._segmentArray = [segmentArray];
    } else {
      throw new Error(
        `segmentArray is not Array<SeismogramSegment> or SeismogramSegment: ${stringify(
          segmentArray
        )}`
      );
    }
    this.checkAllSimilar();
    this._interval = this.findStartEnd();
    checkLuxonValid(this._interval, "seis const");
  }
  checkAllSimilar() {
    if (this._segmentArray.length === 0) {
      throw new Error("Seismogram is empty");
    }
    const f = this._segmentArray[0];
    this._segmentArray.forEach((s, i) => {
      if (!s) {
        throw new Error(`index ${i} is null in trace`);
      }
      this.checkSimilar(f, s);
    });
  }
  checkSimilar(f, s) {
    if (!s.sourceId.equals(f.sourceId)) {
      throw new Error(
        `SourceId not same: ${s.sourceId.toString()} !== ${f.sourceId.toString()}`
      );
    }
    if (s.yUnit !== f.yUnit) {
      throw new Error("yUnit not same: " + s.yUnit + " !== " + f.yUnit);
    }
  }
  findStartEnd() {
    if (this._segmentArray.length === 0) {
      throw new Error("Seismogram is empty");
    }
    return this._segmentArray.reduce(
      (acc, cur) => acc.union(cur.timeRange),
      this._segmentArray[0].timeRange
    );
  }
  findMinMax(minMaxAccumulator) {
    if (this._segmentArray.length === 0) {
      throw new Error("No data");
    }
    for (const s of this._segmentArray) {
      minMaxAccumulator = s.findMinMax(minMaxAccumulator);
    }
    if (minMaxAccumulator) {
      return minMaxAccumulator;
    } else {
      throw new Error("No data to calc minmax");
    }
  }
  /**
   * calculates the mean of a seismogrma.
   *
   * @returns       mean value
   */
  mean() {
    let meanVal = 0;
    const npts = this.numPoints;
    for (const s of this.segments) {
      meanVal += meanOfSlice(s.y, s.y.length) * s.numPoints;
    }
    meanVal = meanVal / npts;
    return meanVal;
  }
  get start() {
    return this.startTime;
  }
  get startTime() {
    return validStartTime(this._interval);
  }
  get end() {
    return this.endTime;
  }
  get endTime() {
    return validEndTime(this._interval);
  }
  get timeRange() {
    return this._interval;
  }
  get networkCode() {
    return this.sourceId.networkCode;
  }
  get stationCode() {
    return this.sourceId.stationCode;
  }
  get locationCode() {
    return this.sourceId.locationCode;
  }
  get channelCode() {
    return this.sourceId.formChannelCode();
  }
  /**
   * return FDSN source id as a string.
   *
   * @returns FDSN source id
   */
  get sourceId() {
    return this._segmentArray[0].sourceId;
  }
  set sourceId(sid) {
    this._segmentArray.forEach((s) => s.sourceId = sid);
  }
  get sampleRate() {
    return this._segmentArray[0].sampleRate;
  }
  get samplePeriod() {
    return 1 / this.sampleRate;
  }
  get yUnit() {
    return this._segmentArray[0].yUnit;
  }
  isYUnitCount() {
    return this.yUnit?.toLowerCase() === COUNT_UNIT2;
  }
  get numPoints() {
    return this._segmentArray.reduce(
      (accumulator, seis) => accumulator + seis.numPoints,
      0
    );
  }
  hasCodes() {
    return this._segmentArray[0].hasCodes();
  }
  /**
   * return network, station, location and channels codes as one string.
   * Uses this.channel if it exists, this.seismogram if not.
   *
   * @returns net.sta.loc.chan
   */
  get nslc() {
    return this.codes();
  }
  get nslcId() {
    return this._segmentArray[0].nslcId;
  }
  codes() {
    return this._segmentArray[0].codes();
  }
  get segments() {
    return this._segmentArray;
  }
  append(seismogram) {
    if (seismogram instanceof _Seismogram) {
      seismogram._segmentArray.forEach((s) => this.append(s));
    } else {
      this.checkSimilar(this._segmentArray[0], seismogram);
      this._interval = this._interval.union(seismogram.timeRange);
      this._segmentArray.push(seismogram);
    }
  }
  /**
   * Cut the seismogram. Creates a new seismogram with all datapoints
   * contained in the time window.
   *
   * @param  timeRange start and end of cut
   * @returns            new seismogram
   */
  cut(timeRange) {
    let out = this.trim(timeRange);
    if (out && out._segmentArray) {
      const cutSeisArray = this._segmentArray.map((seg) => seg.cut(timeRange)).filter(isDef);
      if (cutSeisArray.length > 0) {
        out = new _Seismogram(cutSeisArray);
      } else {
        out = null;
      }
    } else {
      out = null;
    }
    return out;
  }
  /**
   * Creates a new Seismogram composed of all seismogram segments that overlap the
   * given time window. If none do, this returns null. This is a faster but coarser
   * version of cut as it only removes whole segments that do not overlap the
   * time window. For most seismograms that consist of a single contiguous
   * data segment, this will do nothing.
   *
   * @param timeRange time range to trim to
   * @returns new seismogram if data in the window, null otherwise
   * @see cut
   */
  trim(timeRange) {
    let out = null;
    checkLuxonValid(timeRange);
    const timeRange_start = validStartTime(timeRange);
    const timeRange_end = validEndTime(timeRange);
    if (this._segmentArray) {
      const trimSeisArray = this._segmentArray.filter(function(d) {
        return d.endTime >= timeRange_start;
      }).filter(function(d) {
        return d.startTime <= timeRange_end;
      });
      if (trimSeisArray.length > 0) {
        out = new _Seismogram(trimSeisArray);
      }
    }
    return out;
  }
  break(duration) {
    if (duration.valueOf() < 0) {
      throw new Error(`Negative duration not allowed: ${duration.toString()}`);
    }
    if (this._segmentArray) {
      let breakStart = this.startTime;
      let out = [];
      while (breakStart < this.endTime) {
        const breakWindow = Interval4.after(breakStart, duration);
        const cutSeisArray = this._segmentArray.map((seg) => seg.cut(breakWindow)).filter(isDef);
        out = out.concat(cutSeisArray);
        breakStart = breakStart.plus(duration);
      }
      out = out.filter(isDef);
      this._segmentArray = out;
    }
  }
  isContiguous() {
    if (this._segmentArray.length === 1) {
      return true;
    }
    let prev = null;
    for (const s of this._segmentArray) {
      if (prev && !(prev.endTime < s.startTime && prev.endTime.plus(
        Duration4.fromMillis(1e3 * 1.5 / prev.sampleRate)
      ) > s.startTime)) {
        return false;
      }
      prev = s;
    }
    return true;
  }
  /**
   * Merges all segments into a single array of the same type as the first
   * segment. No checking is done for gaps or overlaps, this is a simple
   * congatination. Be careful!
   *
   * @returns contatenated data
   */
  merge() {
    let outArray;
    let idx = 0;
    if (this._segmentArray.every((seg) => seg.y instanceof Int32Array)) {
      outArray = new Int32Array(this.numPoints);
      this._segmentArray.forEach((seg) => {
        outArray.set(seg.y, idx);
        idx += seg.y.length;
      });
    } else if (this._segmentArray.every((seg) => seg.y instanceof Float32Array)) {
      outArray = new Float32Array(this.numPoints);
      this._segmentArray.forEach((seg) => {
        outArray.set(seg.y, idx);
        idx += seg.y.length;
      });
    } else if (this._segmentArray.every((seg) => seg.y instanceof Float64Array)) {
      outArray = new Float64Array(this.numPoints);
      this._segmentArray.forEach((seg) => {
        outArray.set(seg.y, idx);
        idx += seg.y.length;
      });
    } else {
      throw new Error(
        `data not all same one of Int32Array, Float32Array or Float64Array`
      );
    }
    return outArray;
  }
  /**
   * Gets the timeseries as an typed array if it is contiguous.
   *
   * @throws {NonContiguousData} if data is not contiguous.
   * @returns  timeseries as array of number
   */
  get y() {
    if (!this._y) {
      if (this.isContiguous()) {
        this._y = this.merge();
      }
    }
    if (this._y) {
      return this._y;
    } else {
      throw new Error(
        "Seismogram is not contiguous, access each SeismogramSegment idividually."
      );
    }
  }
  set y(val) {
    throw new Error("seismogram y setter not impl, see cloneWithNewData()");
  }
  clone() {
    const cloned = this._segmentArray.map((s) => s.clone());
    return new _Seismogram(cloned);
  }
  cloneWithNewData(newY) {
    if (newY && newY.length > 0) {
      const seg = this._segmentArray[0].cloneWithNewData(newY);
      return new _Seismogram([seg]);
    } else {
      throw new Error("Y value is empty");
    }
  }
  /**
   * factory method to create a single segment Seismogram from either encoded data
   *  or a TypedArray, along with sample rate and start time.
   *
   * @param yArray array of encoded data or typed array
   * @param sampleRate sample rate, samples per second of the data
   * @param startTime time of first sample
   * @param sourceId optional source id
   * @returns seismogram initialized with the data
   */
  static fromContiguousData(yArray, sampleRate, startTime, sourceId) {
    const seg = new SeismogramSegment(yArray, sampleRate, startTime, sourceId);
    return new _Seismogram([seg]);
  }
};
var NonContiguousData = class extends Error {
  constructor(message) {
    super(message);
    this.name = this.constructor.name;
  }
};
function ensureIsSeismogram(seisSeismogram) {
  if (typeof seisSeismogram === "object") {
    if (seisSeismogram instanceof Seismogram) {
      return seisSeismogram;
    } else if (seisSeismogram instanceof SeismogramSegment) {
      return new Seismogram([seisSeismogram]);
    } else {
      throw new Error(
        "must be Seismogram or SeismogramSegment but " + stringify(seisSeismogram)
      );
    }
  } else {
    throw new Error(
      "must be Seismogram or SeismogramSegment but not an object: " + stringify(seisSeismogram)
    );
  }
}
var SeismogramDisplayData = class _SeismogramDisplayData {
  /** @private */
  _seismogram;
  _id;
  _sourceId;
  label;
  markerList;
  traveltimeList;
  channel;
  _instrumentSensitivity;
  quakeList;
  quakeReferenceList = [];
  timeRange;
  alignmentTime;
  doShow;
  _statsCache;
  constructor(timeRange) {
    if (!timeRange) {
      throw new Error("timeRange must not be missing.");
    }
    checkLuxonValid(timeRange);
    this._id = null;
    this._sourceId = null;
    this._seismogram = null;
    this.label = null;
    this.markerList = [];
    this.traveltimeList = [];
    this.channel = null;
    this._instrumentSensitivity = null;
    this.quakeList = [];
    this.timeRange = timeRange;
    this.alignmentTime = validStartTime(timeRange);
    this.doShow = true;
    this._statsCache = null;
  }
  static fromSeismogram(seismogram) {
    const out = new _SeismogramDisplayData(
      Interval4.fromDateTimes(seismogram.startTime, seismogram.endTime)
    );
    out.seismogram = seismogram;
    return out;
  }
  /**
   * Create a Seismogram from the segment, then call fromSeismogram to create
   * the SeismogramDisplayData;
   * @param  seisSegment segment of contiguous data
   * @returns             new SeismogramDisplayData
   */
  static fromSeismogramSegment(seisSegment) {
    return _SeismogramDisplayData.fromSeismogram(new Seismogram([seisSegment]));
  }
  /**
   * Useful for creating fake data from an array, sample rate and start time
   *
   * @param yArray fake data
   * @param sampleRate samples per second
   * @param startTime  start of data, time of first point
   * @param sourceId  optional source id
   * @returns seismogramdisplaydata
   */
  static fromContiguousData(yArray, sampleRate, startTime, sourceId) {
    return _SeismogramDisplayData.fromSeismogram(
      Seismogram.fromContiguousData(yArray, sampleRate, startTime, sourceId)
    );
  }
  static fromChannelAndTimeWindow(channel, timeRange) {
    if (!channel) {
      throw new Error("fromChannelAndTimeWindow, channel is undef");
    }
    const out = new _SeismogramDisplayData(timeRange);
    out.channel = channel;
    return out;
  }
  static fromChannelAndTimes(channel, startTime, endTime) {
    const out = new _SeismogramDisplayData(
      Interval4.fromDateTimes(startTime, endTime)
    );
    out.channel = channel;
    return out;
  }
  static fromSourceIdAndTimes(sourceId, startTime, endTime) {
    const out = new _SeismogramDisplayData(
      Interval4.fromDateTimes(startTime, endTime)
    );
    out._sourceId = sourceId;
    return out;
  }
  static fromCodesAndTimes(networkCode, stationCode, locationCode, channelCode, startTime, endTime) {
    const out = new _SeismogramDisplayData(
      Interval4.fromDateTimes(startTime, endTime)
    );
    out._sourceId = FDSNSourceId.fromNslc(
      networkCode,
      stationCode,
      locationCode,
      channelCode
    );
    return out;
  }
  addQuake(quake) {
    if (Array.isArray(quake)) {
      quake.forEach((q) => this.quakeList.push(q));
    } else {
      this.quakeList.push(quake);
    }
  }
  /**
   * Adds a public id for a Quake to the seismogram. For use in case where
   * the quake is not yet available, but wish to retain the connection.
   * @param  publicId  id of the earthquake assocated with this seismogram
   */
  addQuakeId(publicId) {
    this.quakeReferenceList.push(publicId);
  }
  addMarker(marker) {
    this.addMarkers([marker]);
  }
  addMarkers(markers) {
    if (Array.isArray(markers)) {
      markers.forEach((m) => this.markerList.push(m));
    } else {
      this.markerList.push(markers);
    }
  }
  getMarkers() {
    return this.markerList;
  }
  addTravelTimes(ttimes) {
    if (Array.isArray(ttimes)) {
      ttimes.forEach((m) => this.traveltimeList.push(m));
    } else if ("arrivals" in ttimes) {
      ttimes.arrivals.forEach((m) => this.traveltimeList.push(m));
    } else {
      this.traveltimeList.push(ttimes);
    }
  }
  hasQuake() {
    return this.quakeList.length > 0;
  }
  get quake() {
    if (this.hasQuake()) {
      return this.quakeList[0];
    }
    return null;
  }
  hasSeismogram() {
    return isDef(this._seismogram);
  }
  append(seismogram) {
    if (isDef(this._seismogram)) {
      this._seismogram.append(seismogram);
      if (this.startTime > seismogram.startTime || this.endTime < seismogram.endTime) {
        const startTime = this.startTime < seismogram.startTime ? this.startTime : seismogram.startTime;
        const endTime = this.endTime > seismogram.endTime ? this.endTime : seismogram.endTime;
        this.timeRange = Interval4.fromDateTimes(startTime, endTime);
      }
    } else {
      if (seismogram instanceof SeismogramSegment) {
        this.seismogram = new Seismogram(seismogram);
      } else {
        this.seismogram = seismogram;
      }
    }
    this._statsCache = null;
  }
  hasChannel() {
    return isDef(this.channel);
  }
  hasSensitivity() {
    return this._instrumentSensitivity !== null || isDef(this.channel) && this.channel.hasInstrumentSensitivity();
  }
  /**
   * Allows id-ing a seismogram. Optional.
   *
   * @returns         string id
   */
  get id() {
    return this._id;
  }
  /**
   * Allows iding a seismogram. Optional.
   *
   * @param   value string id
   */
  set id(value) {
    this._id = value;
  }
  /**
   * return network code as a string.
   * Uses this.channel if it exists, this.seismogram if not.
   *
   * @returns network code
   */
  get networkCode() {
    let out = this.sourceId.networkCode;
    if (!isDef(out)) {
      out = "unknown";
    }
    return out;
  }
  /**
   * return station code as a string.
   * Uses this.channel if it exists, this.seismogram if not.
   *
   * @returns station code
   */
  get stationCode() {
    let out = this.sourceId.stationCode;
    if (!isDef(out)) {
      out = "unknown";
    }
    return out;
  }
  /**
   * return location code a a string.
   * Uses this.channel if it exists, this.seismogram if not.
   *
   * @returns location code
   */
  get locationCode() {
    let out = this.sourceId.locationCode;
    if (!isDef(out)) {
      out = "unknown";
    }
    return out;
  }
  /**
   * return channels code as a string.
   * Uses this.channel if it exists, this.seismogram if not.
   *
   * @returns channel code
   */
  get channelCode() {
    let out = this.sourceId.formChannelCode();
    if (!isDef(out)) {
      out = "unknown";
    }
    return out;
  }
  /**
   * return FDSN source id as a string.
   * Uses this.channel if it exists, this.seismogram if not.
   *
   * @returns FDSN source id
   */
  get sourceId() {
    if (isDef(this.channel)) {
      return this.channel.sourceId;
    } else if (isDef(this._seismogram)) {
      return this._seismogram.sourceId;
    } else if (isDef(this._sourceId)) {
      return this._sourceId;
    } else {
      return FDSNSourceId.createUnknown();
    }
  }
  /**
   * return network, station, location and channels codes as one string.
   * Uses this.channel if it exists, this.seismogram if not
   *
   * @returns net.sta.loc.chan
   */
  get nslc() {
    return this.codes();
  }
  get nslcId() {
    if (this.channel !== null) {
      return this.channel.nslcId;
    } else {
      return new NslcId(
        this.networkCode ? this.networkCode : "",
        this.stationCode ? this.stationCode : "",
        this.locationCode && this.locationCode !== "--" ? this.locationCode : "",
        this.channelCode ? this.channelCode : ""
      );
    }
  }
  /**
   * return network, station, location and channels codes as one string.
   * Uses this.channel if it exists, this.seismogram if not.
   *
   * @param sep separator, defaults to '.'
   * @returns nslc codes separated by sep
   */
  codes(sep = ".") {
    if (this.channel !== null) {
      return this.channel.codes();
    } else {
      return (this.networkCode ? this.networkCode : "") + sep + (this.stationCode ? this.stationCode : "") + sep + (this.locationCode ? this.locationCode : "") + sep + (this.channelCode ? this.channelCode : "");
    }
  }
  get startTime() {
    return validStartTime(this.timeRange);
  }
  get start() {
    return this.startTime;
  }
  get endTime() {
    return validEndTime(this.timeRange);
  }
  get end() {
    return this.endTime;
  }
  get numPoints() {
    if (this._seismogram) {
      return this._seismogram.numPoints;
    }
    return 0;
  }
  associateChannel(nets) {
    const matchChans = findChannels(
      nets,
      this.networkCode,
      this.stationCode,
      this.locationCode,
      this.channelCode
    );
    for (const c of matchChans) {
      if (c.timeRange.overlaps(this.timeRange)) {
        this.channel = c;
        return;
      }
    }
  }
  alignStartTime() {
    this.alignmentTime = this.start;
  }
  alignOriginTime() {
    if (this.quake) {
      this.alignmentTime = this.quake.time;
    } else {
      this.alignmentTime = this.start;
    }
  }
  alignPhaseTime(phaseRegExp) {
    let intPhaseRegExp;
    if (typeof phaseRegExp === "string") {
      intPhaseRegExp = new RegExp(phaseRegExp);
    } else {
      intPhaseRegExp = phaseRegExp;
    }
    if (this.quake && this.traveltimeList) {
      const q = this.quake;
      const matchArrival = this.traveltimeList.find((ttArrival) => {
        const match = intPhaseRegExp.exec(ttArrival.phase);
        return match !== null && match[0] === ttArrival.phase;
      });
      if (matchArrival) {
        this.alignmentTime = q.time.plus(
          Duration4.fromMillis(matchArrival.time * 1e3)
        );
      } else {
        this.alignmentTime = this.start;
      }
    }
  }
  /**
   * Create a time window relative to the alignmentTime if set, or the start time if not.
   * Negative durations are allowed.
   * @param alignmentOffset offset duration from the alignment time
   * @param duration duration from the offset for the window
   * @returns time window as an Interval
   */
  relativeTimeWindow(alignmentOffset, duration) {
    const atime = this.alignmentTime ? this.alignmentTime.plus(alignmentOffset) : this.startTime.plus(alignmentOffset);
    return startDuration(atime, duration);
  }
  get sensitivity() {
    const channel = this.channel;
    if (this._instrumentSensitivity) {
      return this._instrumentSensitivity;
    } else if (isDef(channel) && channel.hasInstrumentSensitivity()) {
      return channel.instrumentSensitivity;
    } else {
      return null;
    }
  }
  set sensitivity(value) {
    this._instrumentSensitivity = value;
  }
  get min() {
    if (!this._statsCache) {
      this._statsCache = this.calcStats();
    }
    return this._statsCache.min;
  }
  get max() {
    if (!this._statsCache) {
      this._statsCache = this.calcStats();
    }
    return this._statsCache.max;
  }
  get mean() {
    if (!this._statsCache) {
      this._statsCache = this.calcStats();
    }
    return this._statsCache.mean;
  }
  get middle() {
    if (!this._statsCache) {
      this._statsCache = this.calcStats();
    }
    return this._statsCache.middle;
  }
  get seismogram() {
    return this._seismogram;
  }
  set seismogram(value) {
    this._seismogram = value;
    this._statsCache = null;
  }
  get segments() {
    if (this._seismogram) {
      return this._seismogram.segments;
    } else {
      return [];
    }
  }
  calcStats() {
    const stats = new SeismogramDisplayStats();
    if (this.seismogram) {
      const minMax = this.seismogram.findMinMax();
      stats.min = minMax.min;
      stats.max = minMax.max;
      stats.mean = this.seismogram.mean();
    }
    this._statsCache = stats;
    return stats;
  }
  /**
   * Calculates distance and azimuth for each event in quakeList.
   *
   * @returns Array of DistAzOutput, empty array if no quakes.
   */
  get distazList() {
    if (this.quakeList.length > 0 && isDef(this.channel)) {
      const c = this.channel;
      return this.quakeList.map(
        (q) => distaz(c.latitude, c.longitude, q.latitude, q.longitude)
      );
    }
    return [];
  }
  /**
   * Calculates distance and azimuth for the first event in quakeList. This is
   * a convienence method as usually there will only be one quake.
   *
   * @returns DistAzOutput, null if no quakes.
   */
  get distaz() {
    let out = null;
    if (this.quakeList.length > 0 && this.channel !== null) {
      out = distaz(
        this.channel.latitude,
        this.channel.longitude,
        this.quakeList[0].latitude,
        this.quakeList[0].longitude
      );
    }
    return out;
  }
  clone() {
    return this.cloneWithNewSeismogram(
      this.seismogram ? this.seismogram.clone() : null
    );
  }
  cloneWithNewSeismogram(seis) {
    const out = new _SeismogramDisplayData(this.timeRange);
    const handled = ["_seismogram", "_statsCache", "_sourceId"];
    Object.assign(out, this);
    Object.getOwnPropertyNames(this).forEach((name) => {
      const v = this[name];
      if (handled.find((n) => name === n)) {
      } else if (Array.isArray(v)) {
        out[name] = v.slice();
      }
    });
    out.seismogram = seis;
    out._statsCache = null;
    if (!isDef(out._seismogram) && !isDef(out.channel)) {
      if (this.sourceId) {
        out._sourceId = this.sourceId.clone();
      }
    }
    return out;
  }
  /**
   * Cut the seismogram. Creates a new seismogramDisplayData with the cut
   * seismogram and the timeRange set to the new time window.
   *
   * @param  timeRange start and end of cut
   * @returns           new seismogramDisplayData
   */
  cut(timeRange) {
    let cutSeis = this.seismogram;
    let out;
    if (cutSeis) {
      cutSeis = cutSeis.cut(timeRange);
      out = this.cloneWithNewSeismogram(cutSeis);
      if (!isDef(out._seismogram) && !isDef(out.channel)) {
        out._sourceId = this.sourceId;
      }
    } else {
      out = this.clone();
    }
    out.timeRange = timeRange;
    return out;
  }
  /**
   * Coarse trim the seismogram. Creates a new seismogramDisplayData with the
   * trimmed seismogram and the timeRange set to the new time window.
   * If timeRange is not given, the current time range of the
   * SeismogramDisplayData is used, effectively trimming data to the current
   * window.
   *
   * @param  timeRange start and end of cut
   * @returns           new seismogramDisplayData
   */
  trim(timeRange) {
    if (!timeRange) {
      timeRange = this.timeRange;
    }
    let cutSeis = this.seismogram;
    let out;
    if (cutSeis) {
      cutSeis = cutSeis.trim(timeRange);
      out = this.cloneWithNewSeismogram(cutSeis);
      if (!isDef(out._seismogram) && !isDef(out.channel)) {
        out._sourceId = this.sourceId;
      }
    } else {
      out = this.clone();
    }
    out.timeRange = timeRange;
    return out;
  }
  /**
   * Coarse trim the seismogram in place. The seismogram is
   * trimmed to the given time window.
   * If timeRange is not given, the current time range of the
   * SeismogramDisplayData is used, effectively trimming data to the current
   * window.
   *
   * @param  timeRange start and end of cut
   */
  trimInPlace(timeRange) {
    if (!timeRange) {
      timeRange = this.timeRange;
    }
    const cutSeis = this.seismogram;
    if (cutSeis) {
      this.seismogram = cutSeis.trim(timeRange);
    }
  }
  toString() {
    return `${this.sourceId.toString()} ${this.timeRange.toString()}`;
  }
};
var SeismogramDisplayStats = class {
  min;
  max;
  mean;
  trendSlope;
  constructor() {
    this.min = 0;
    this.max = 0;
    this.mean = 0;
    this.trendSlope = 0;
  }
  get middle() {
    return (this.min + this.max) / 2;
  }
};
function findStartEnd(sddList) {
  if (sddList.length === 0) {
    return Interval4.before(DateTime4.utc(), 0);
  }
  return sddList.reduce(
    (acc, sdd) => acc.union(sdd.timeRange),
    sddList[0].timeRange
  );
}
function findMaxDuration(sddList) {
  return findMaxDurationOfType("start", sddList);
}
function findMaxDurationOfType(type, sddList) {
  return sddList.reduce((acc, sdd) => {
    let timeRange;
    if (type === "start") {
      timeRange = sdd.timeRange;
    } else if (type === "origin" && sdd.hasQuake()) {
      timeRange = Interval4.fromDateTimes(
        sdd.quakeList[0].time,
        validEndTime(sdd.timeRange)
      );
    } else if (type === "align" && sdd.alignmentTime) {
      timeRange = Interval4.fromDateTimes(
        sdd.alignmentTime,
        validEndTime(sdd.timeRange)
      );
    } else {
      timeRange = sdd.timeRange;
    }
    if (timeRange.toDuration().toMillis() > acc.toMillis()) {
      return timeRange.toDuration();
    } else {
      return acc;
    }
  }, Duration4.fromMillis(0));
}
function findMinMax(sddList, doGain = false, amplitudeMode = "minmax" /* MinMax */) {
  return findMinMaxOverTimeRange(sddList, null, doGain, amplitudeMode);
}
function findMinMaxOverTimeRange(sddList, timeRange = null, doGain = false, amplitudeMode = "minmax" /* MinMax */) {
  if (sddList.length === 0) {
    return new MinMaxable(-1, 1);
  }
  const minMaxArr = sddList.map((sdd) => {
    return calcMinMax(sdd, timeRange, doGain, amplitudeMode);
  }).filter((x) => x).reduce(function(p, v) {
    if (amplitudeMode === "raw" /* Raw */ || amplitudeMode === "zero" /* Zero */) {
      return p ? v ? p.union(v) : p : v;
    } else {
      let hw = 0;
      if (p && v) {
        hw = Math.max(p.halfWidth, v.halfWidth);
      } else if (p) {
        hw = p.halfWidth;
      } else if (v) {
        hw = v.halfWidth;
      } else {
        hw = 0;
      }
      return MinMaxable.fromMiddleHalfWidth(0, hw);
    }
  }, null);
  if (minMaxArr) {
    return minMaxArr;
  }
  return new MinMaxable(-1, 1);
}
function findMinMaxOverRelativeTimeRange(sddList, alignmentOffset, duration, doGain = false, amplitudeMode = "minmax" /* MinMax */) {
  if (sddList.length === 0) {
    return new MinMaxable(0, 0);
  }
  const minMaxArr = sddList.map((sdd) => {
    const timeRange = sdd.relativeTimeWindow(alignmentOffset, duration);
    return calcMinMax(sdd, timeRange, doGain, amplitudeMode);
  }).filter((x) => x).reduce(function(p, v) {
    if (amplitudeMode === "raw" /* Raw */ || amplitudeMode === "zero" /* Zero */) {
      return p ? v ? p.union(v) : p : v;
    } else {
      let hw = 0;
      if (p && v) {
        hw = Math.max(p.halfWidth, v.halfWidth);
      } else if (p) {
        hw = p.halfWidth;
      } else if (v) {
        hw = v.halfWidth;
      } else {
        hw = 0;
      }
      return MinMaxable.fromMiddleHalfWidth(0, hw);
    }
  }, null);
  if (minMaxArr) {
    return minMaxArr;
  }
  return new MinMaxable(-1, 1);
}
function calcMinMax(sdd, timeRange = null, doGain = false, amplitudeMode = "minmax" /* MinMax */) {
  if (sdd.seismogram) {
    let cutSDD;
    if (timeRange) {
      cutSDD = sdd.cut(timeRange);
    } else {
      cutSDD = sdd;
    }
    if (cutSDD) {
      let sens = 1;
      if (doGain && sdd.sensitivity) {
        sens = sdd.sensitivity.sensitivity;
      }
      let middle = 0;
      let halfWidth = 0;
      if (amplitudeMode === "minmax" /* MinMax */ || amplitudeMode === "raw" /* Raw */) {
        middle = cutSDD.middle;
        halfWidth = Math.max(
          (middle - cutSDD.min) / sens,
          (cutSDD.max - middle) / sens
        );
      } else if (amplitudeMode === "mean" /* Mean */) {
        middle = sdd.mean;
        halfWidth = Math.max(
          (middle - cutSDD.min) / sens,
          (cutSDD.max - middle) / sens
        );
      } else if (amplitudeMode === "zero" /* Zero */) {
        const minwz = Math.min(0, cutSDD.min);
        const maxwz = Math.max(0, cutSDD.max);
        middle = (minwz + maxwz) / 2;
        halfWidth = (maxwz - minwz) / 2 / sens;
      } else {
        throw new Error(
          `Unknown amplitudeMode: ${stringify(amplitudeMode)}. Must be one of raw, zero, minmax, mean`
        );
      }
      return MinMaxable.fromMiddleHalfWidth(middle, halfWidth);
    }
  }
  return null;
}
function findStartEndOfSeismograms(data, accumulator) {
  let out;
  if (!accumulator && !data) {
    throw new Error("data and accumulator are not defined");
  } else if (!accumulator) {
    if (data.length !== 0) {
      out = data[0].timeRange;
    } else {
      throw new Error("data.length == 0 and accumulator is not defined");
    }
  } else {
    out = accumulator;
  }
  if (Array.isArray(data)) {
    return data.reduce(
      (acc, cur) => acc.union(cur.timeRange),
      data[0].timeRange
    );
  } else {
    throw new Error(`Expected Array as first arg but was: ${typeof data}`);
  }
  return out;
}
function findMinMaxOfSeismograms(data, minMaxAccumulator) {
  for (const s of data) {
    minMaxAccumulator = s.findMinMax(minMaxAccumulator);
  }
  if (minMaxAccumulator) {
    return minMaxAccumulator;
  } else {
    return new MinMaxable(-1, 1);
  }
}
function findMinMaxOfSDD(data, minMaxAccumulator) {
  const seisData = [];
  data.forEach((sdd) => {
    if (!!sdd && !!sdd.seismogram) {
      seisData.push(sdd.seismogram);
    }
  });
  return findMinMaxOfSeismograms(seisData, minMaxAccumulator);
}
function uniqueStations2(seisData) {
  const out = /* @__PURE__ */ new Set();
  seisData.forEach((sdd) => {
    if (sdd.channel) {
      out.add(sdd.channel.station);
    }
  });
  return Array.from(out.values());
}
function uniqueChannels(seisData) {
  const out = /* @__PURE__ */ new Set();
  seisData.forEach((sdd) => {
    if (sdd.channel) {
      out.add(sdd.channel);
    }
  });
  return Array.from(out.values());
}
function uniqueQuakes(seisData) {
  const out = /* @__PURE__ */ new Set();
  seisData.forEach((sdd) => {
    sdd.quakeList.forEach((q) => out.add(q));
  });
  return Array.from(out.values());
}

// src/miniseed.ts
var MINISEED_MIME = "application/vnd.fdsn.mseed";
var R_TYPECODE = "R".charCodeAt(0);
var D_TYPECODE = "D".charCodeAt(0);
var Q_TYPECODE = "Q".charCodeAt(0);
var M_TYPECODE = "M".charCodeAt(0);
function parseDataRecords(arrayBuffer) {
  const dataRecords = [];
  let offset = 0;
  while (offset < arrayBuffer.byteLength) {
    const dataView = new DataView(arrayBuffer, offset);
    const dr = parseSingleDataRecord(dataView);
    dataRecords.push(dr);
    offset += dr.header.recordSize;
  }
  return dataRecords;
}
function parseSingleDataRecord(dataView) {
  const header = parseSingleDataRecordHeader(dataView);
  const data = new DataView(
    dataView.buffer,
    dataView.byteOffset + header.dataOffset,
    header.recordSize - header.dataOffset
  );
  return new DataRecord(header, data);
}
function parseSingleDataRecordHeader(dataView) {
  if (dataView.byteLength < 47) {
    throw new Error(
      `Not enought bytes for header, need 47, found ${dataView.byteLength}`
    );
  }
  const out = new DataHeader();
  out.seq = makeString(dataView, 0, 6);
  out.typeCode = dataView.getUint8(6);
  out.continuationCode = dataView.getUint8(7);
  out.staCode = makeString(dataView, 8, 5);
  out.locCode = makeString(dataView, 13, 2);
  out.chanCode = makeString(dataView, 15, 3);
  out.netCode = makeString(dataView, 18, 2);
  out.startBTime = parseBTime(dataView, 20);
  const headerByteSwap = checkByteSwap(out.startBTime);
  if (headerByteSwap) {
    out.startBTime = parseBTime(dataView, 20, headerByteSwap);
  }
  out.numSamples = dataView.getInt16(30, headerByteSwap);
  out.sampRateFac = dataView.getInt16(32, headerByteSwap);
  out.sampRateMul = dataView.getInt16(34, headerByteSwap);
  out.activityFlags = dataView.getUint8(36);
  out.ioClockFlags = dataView.getUint8(37);
  out.dataQualityFlags = dataView.getUint8(38);
  out.numBlockettes = dataView.getUint8(39);
  out.timeCorrection = dataView.getInt32(40, headerByteSwap);
  out.dataOffset = dataView.getUint16(44, headerByteSwap);
  out.blocketteOffset = dataView.getUint16(46, headerByteSwap);
  let offset = out.blocketteOffset;
  out.blocketteList = [];
  out.recordSize = 4096;
  out.sampleRate = out.calcSampleRate();
  out.startTime = out.startBTime.toDateTime();
  for (let i = 0; i < out.numBlockettes; i++) {
    let nextOffset = dataView.getUint16(offset + 2, headerByteSwap);
    if (nextOffset === 0) {
      nextOffset = out.dataOffset;
    }
    if (nextOffset === 0) {
      nextOffset = offset;
    }
    const blockette = parseBlockette(
      dataView,
      offset,
      nextOffset - offset,
      headerByteSwap
    );
    out.blocketteList.push(blockette);
    offset = nextOffset;
    if (blockette instanceof Blockette1000) {
      out.recordSize = 1 << blockette.dataRecordLengthByte;
      out.encoding = blockette.encoding;
      out.littleEndian = blockette.wordOrder === 0;
    } else if (blockette instanceof Blockette1001) {
      out.startBTime.microsecond = blockette.microsecond;
    } else if (blockette instanceof Blockette100) {
      out.sampleRate = blockette.sampleRate;
    }
  }
  out.endTime = out.timeOfSample(out.numSamples - 1);
  return out;
}
function parseBlockette(dataView, offset, length, headerByteSwap) {
  const type = dataView.getUint16(offset, headerByteSwap);
  const body = new DataView(
    dataView.buffer,
    dataView.byteOffset + offset,
    length
  );
  if (type === 1e3) {
    const encoding = body.getUint8(4);
    const wordOrder = body.getUint8(5);
    const dataRecordLengthByte = body.getUint8(6);
    return new Blockette1000(
      type,
      body,
      encoding,
      dataRecordLengthByte,
      wordOrder
    );
  } else if (type === 1001) {
    const timeQual = body.getUint8(4);
    const microsecond = body.getUint8(5);
    const frameCount = body.getUint8(7);
    return new Blockette1001(type, body, timeQual, microsecond, frameCount);
  } else if (type === 100) {
    const sampleRate = body.getFloat32(4);
    const flags = body.getUint8(8);
    return new Blockette100(type, body, sampleRate, flags);
  } else {
    return new Blockette(type, body);
  }
}
var DataRecord = class {
  header;
  data;
  constructor(header, data) {
    this.header = header;
    this.data = data;
  }
  /**
   * Decompresses the data , if the compression type is known.
   *
   * @returns decompressed data
   */
  decompress() {
    return this.asEncodedDataSegment().decode();
  }
  asEncodedDataSegment() {
    return new EncodedDataSegment(
      this.header.encoding,
      this.data,
      this.header.numSamples,
      this.header.littleEndian
    );
  }
  /**
   * Concatenates the net, station, loc and channel codes,
   * separated by the given seperator, or periods if not given.
   *
   * @param sep optional separater, defaults to .
   * @returns string of codes
   */
  codes(sep) {
    if (!isNonEmptyStringArg(sep)) {
      sep = ".";
    }
    return this.header.netCode + sep + this.header.staCode + sep + this.header.locCode + sep + this.header.chanCode;
  }
};
var DataHeader = class {
  seq;
  typeCode;
  continuationCode;
  staCode;
  locCode;
  chanCode;
  netCode;
  startBTime;
  numSamples;
  encoding;
  littleEndian;
  sampRateFac;
  sampRateMul;
  sampleRate;
  activityFlags;
  ioClockFlags;
  dataQualityFlags;
  numBlockettes;
  timeCorrection;
  dataOffset;
  blocketteOffset;
  recordSize;
  blocketteList;
  startTime;
  endTime;
  constructor() {
    this.seq = "      ";
    this.typeCode = 68;
    this.continuationCode = 32;
    this.staCode = "";
    this.locCode = "";
    this.chanCode = "";
    this.netCode = "";
    this.startBTime = new BTime(1900, 1, 0, 0, 0, 0);
    this.numSamples = 0;
    this.sampRateFac = 0;
    this.sampRateMul = 0;
    this.activityFlags = 0;
    this.ioClockFlags = 0;
    this.dataQualityFlags = 0;
    this.numBlockettes = 0;
    this.timeCorrection = 0;
    this.dataOffset = 0;
    this.blocketteOffset = 0;
    this.blocketteList = [];
    this.recordSize = 4096;
    this.encoding = 0;
    this.littleEndian = false;
    this.sampleRate = 0;
    this.startTime = this.startBTime.toDateTime();
    this.endTime = this.startTime;
  }
  toString() {
    return `${this.netCode}.${this.staCode}.${this.locCode}.${this.chanCode} ${this.startTime.toISO()} ${String(this.encoding)}`;
  }
  /**
   * Calculates the sample rate in hertz from the sampRateFac and sampRateMul
   * parameters. This.sampleRate value is set to this value at construction.
   *
   * @returns sample rate
   */
  calcSampleRate() {
    const factor = this.sampRateFac;
    const multiplier = this.sampRateMul;
    let sampleRate = 1e4;
    if (factor * multiplier !== 0) {
      sampleRate = Math.pow(Math.abs(factor), factor / Math.abs(factor)) * Math.pow(Math.abs(multiplier), multiplier / Math.abs(multiplier));
    }
    return sampleRate;
  }
  /**
   * Calculates the time of the i-th sample in the record, zero based,
   *  so timeOfSample(0) is the start and timeOfSample(this.numSamples-1) is end.
   *
   * @param i sample index
   * @returns time at i-th sample as DateTime
   */
  timeOfSample(i) {
    return this.startTime.plus(
      Duration5.fromMillis(1e3 * i / this.sampleRate)
    );
  }
};
var Blockette = class {
  type;
  body;
  constructor(type, body) {
    this.type = type;
    this.body = body;
  }
};
var Blockette1000 = class extends Blockette {
  encoding;
  dataRecordLengthByte;
  wordOrder;
  constructor(type, body, encoding, dataRecordLengthByte, wordOrder) {
    super(type, body);
    if (type !== 1e3) {
      throw new Error("Not a blockette1000: " + this.type);
    }
    this.encoding = encoding;
    this.dataRecordLengthByte = dataRecordLengthByte;
    this.wordOrder = wordOrder;
  }
};
var Blockette1001 = class extends Blockette {
  timeQual;
  microsecond;
  frameCount;
  constructor(type, body, timeQual, microsecond, frameCount) {
    super(type, body);
    if (type !== 1001) {
      throw new Error("Not a blockette1001: " + this.type);
    }
    this.timeQual = timeQual;
    this.microsecond = microsecond;
    this.frameCount = frameCount;
  }
};
var Blockette100 = class extends Blockette {
  sampleRate;
  flags;
  constructor(type, body, sampleRate, flags) {
    super(type, body);
    if (type !== 100) {
      throw new Error("Not a blockette100: " + this.type);
    }
    this.sampleRate = sampleRate;
    this.flags = flags;
  }
};
function makeString(dataView, offset, length) {
  let out = "";
  for (let i = offset; i < offset + length; i++) {
    const charCode = dataView.getUint8(i);
    if (charCode > 31) {
      out += String.fromCharCode(charCode);
    }
  }
  return out.trim();
}
function parseBTime(dataView, offset, byteSwap) {
  if (!isDef(byteSwap)) {
    byteSwap = false;
  }
  const year = dataView.getInt16(offset, byteSwap);
  const jday = dataView.getInt16(offset + 2, byteSwap);
  const hour = dataView.getInt8(offset + 4);
  const min = dataView.getInt8(offset + 5);
  const sec = dataView.getInt8(offset + 6);
  const tenthMilli = dataView.getInt16(offset + 8, byteSwap);
  return new BTime(year, jday, hour, min, sec, tenthMilli);
}
var BTime = class {
  year;
  jday;
  hour;
  min;
  sec;
  tenthMilli;
  microsecond;
  // -50 to 49, not part of BTime proper, but added in case of B1001
  length;
  constructor(year, jday, hour, min, sec, tenthMilli) {
    this.length = 10;
    this.year = year;
    this.jday = jday;
    this.hour = hour;
    this.min = min;
    this.sec = sec;
    this.tenthMilli = tenthMilli;
    this.microsecond = 0;
  }
  toString() {
    return this.year + "-" + this.jday + " " + this.hour + ":" + this.min + ":" + this.sec + "." + this.tenthMilli.toFixed().padStart(4, "0") + " " + (this.microsecond !== 0 ? `usec: ${this.microsecond} ` : "") + "iso: " + this.toDateTime().toISO();
  }
  /**
   * Converts this BTime to a luxon utc DateTime. Note DateTime's precision
   * is limited to milliseconds and leap seconds are not supported,
   * ie 60 seconds returns DateTime.invalid.
   *
   * @returns         BTime as a DateTime
   */
  toDateTime() {
    const millis = Math.round(this.tenthMilli / 10);
    if (this.sec === 60) {
      return DateTime5.invalid(
        "Leap seconds not supported",
        `seconds value ${this.sec} is a leap second, but luxon does not support`
      );
    }
    const d = DateTime5.fromObject(
      {
        year: this.year,
        ordinal: this.jday,
        hour: this.hour,
        minute: this.min,
        second: this.sec,
        millisecond: 0
      },
      UTC_OPTIONS
    );
    return d.plus(millis);
  }
};
function checkByteSwap(bTime) {
  if (bTime.year === 1900 && bTime.jday === 1) {
    return false;
  }
  return bTime.year < 1960 || bTime.year > 2055;
}
function areContiguous(dr1, dr2) {
  const h1 = dr1.header;
  const h2 = dr2.header;
  return h1.endTime < h2.startTime && h1.endTime.valueOf() + 1e3 * 1.5 / h1.sampleRate > h2.startTime.valueOf();
}
function createSeismogramSegment(contig) {
  if (!Array.isArray(contig)) {
    contig = [contig];
  }
  const contigData = contig.map((dr) => dr.asEncodedDataSegment());
  const out = new SeismogramSegment(
    contigData,
    contig[0].header.sampleRate,
    contig[0].header.startTime,
    FDSNSourceId.fromNslc(
      contig[0].header.netCode,
      contig[0].header.staCode,
      contig[0].header.locCode,
      contig[0].header.chanCode
    )
  );
  return out;
}
function merge(drList) {
  return new Seismogram(mergeSegments(drList));
}
function mergeSegments(drList) {
  const out = [];
  let currDR;
  drList.sort(function(a, b) {
    return a.header.startTime.toMillis() - b.header.startTime.toMillis();
  });
  let contig = [];
  for (let i = 0; i < drList.length; i++) {
    currDR = drList[i];
    if (contig.length === 0) {
      contig.push(currDR);
    } else if (areContiguous(contig[contig.length - 1], currDR)) {
      contig.push(currDR);
    } else {
      out.push(createSeismogramSegment(contig));
      contig = [currDR];
    }
  }
  if (contig.length > 0) {
    out.push(createSeismogramSegment(contig));
    contig = [];
  }
  return out;
}
function byChannel(drList) {
  const out = /* @__PURE__ */ new Map();
  let key;
  for (let i = 0; i < drList.length; i++) {
    const currDR = drList[i];
    key = currDR.codes();
    let drArray = out.get(key);
    if (!drArray) {
      drArray = [currDR];
      out.set(key, drArray);
    } else {
      drArray.push(currDR);
    }
  }
  return out;
}
function seismogramSegmentPerChannel(drList) {
  let out = new Array(0);
  const byChannelMap = byChannel(drList);
  byChannelMap.forEach(
    (segments) => out = out.concat(mergeSegments(segments))
  );
  return out;
}
function seismogramPerChannel(drList) {
  const out = [];
  const byChannelMap = byChannel(drList);
  byChannelMap.forEach((segments) => out.push(merge(segments)));
  return out;
}

// src/mseed3.ts
var mseed3_exports = {};
__export(mseed3_exports, {
  BIG_ENDIAN: () => BIG_ENDIAN,
  CRC_OFFSET: () => CRC_OFFSET,
  FDSN_PREFIX: () => FDSN_PREFIX2,
  FIXED_HEADER_SIZE: () => FIXED_HEADER_SIZE,
  LITTLE_ENDIAN: () => LITTLE_ENDIAN,
  MINISEED_THREE_MIME: () => MINISEED_THREE_MIME,
  MSeed3Header: () => MSeed3Header,
  MSeed3Record: () => MSeed3Record,
  UNKNOWN_DATA_VERSION: () => UNKNOWN_DATA_VERSION,
  areContiguous: () => areContiguous2,
  byChannel: () => byChannel2,
  calculateCRC32C: () => calculateCRC32C,
  convertMS2Record: () => convertMS2Record,
  convertMS2toMSeed3: () => convertMS2toMSeed3,
  crcToHexString: () => crcToHexString,
  createSeismogramSegment: () => createSeismogramSegment2,
  makeString: () => makeString2,
  merge: () => merge2,
  mergeSegments: () => mergeSegments2,
  padZeros: () => padZeros,
  parseExtraHeaders: () => parseExtraHeaders,
  parseMSeed3Records: () => parseMSeed3Records,
  sddPerChannel: () => sddPerChannel,
  seismogramPerChannel: () => seismogramPerChannel2,
  seismogramSegmentPerChannel: () => seismogramSegmentPerChannel2,
  toMSeed3: () => toMSeed3
});

// src/quakeml.ts
var quakeml_exports = {};
__export(quakeml_exports, {
  ANSS_CATALOG_NS: () => ANSS_CATALOG_NS,
  ANSS_NS: () => ANSS_NS,
  Amplitude: () => Amplitude,
  Arrival: () => Arrival,
  Axis: () => Axis,
  BED_NS: () => BED_NS,
  Comment: () => Comment2,
  CompositeTime: () => CompositeTime,
  ConfidenceEllipsoid: () => ConfidenceEllipsoid,
  CreationInfo: () => CreationInfo,
  DataUsed: () => DataUsed,
  EventDescription: () => EventDescription,
  EventParameters: () => EventParameters,
  FAKE_EMPTY_XML: () => FAKE_EMPTY_XML2,
  FAKE_ORIGIN_TIME: () => FAKE_ORIGIN_TIME,
  FocalMechanism: () => FocalMechanism,
  IRIS_NS: () => IRIS_NS,
  Magnitude: () => Magnitude,
  MomentTensor: () => MomentTensor,
  NodalPlane: () => NodalPlane,
  NodalPlanes: () => NodalPlanes,
  Origin: () => Origin,
  OriginQuality: () => OriginQuality,
  OriginUncertainty: () => OriginUncertainty,
  Pick: () => Pick,
  PrincipalAxes: () => PrincipalAxes,
  QML_NS: () => QML_NS,
  QUAKE_CLICK_EVENT: () => QUAKE_CLICK_EVENT,
  Quake: () => Quake,
  Quantity: () => Quantity,
  SourceTimeFunction: () => SourceTimeFunction,
  StationMagnitude: () => StationMagnitude,
  StationMagnitudeContribution: () => StationMagnitudeContribution,
  Tensor: () => Tensor,
  TimeWindow: () => TimeWindow,
  UNKNOWN_MAG_TYPE: () => UNKNOWN_MAG_TYPE,
  UNKNOWN_PUBLIC_ID: () => UNKNOWN_PUBLIC_ID,
  USGS_HOST: () => USGS_HOST,
  WaveformID: () => WaveformID,
  createQuakeClickEvent: () => createQuakeClickEvent,
  createQuakeFromValues: () => createQuakeFromValues,
  fetchQuakeML: () => fetchQuakeML,
  parseQuakeML: () => parseQuakeML,
  parseUtil: () => parseUtil2
});

// src/textformat.ts
var lang = typeof navigator !== "undefined" && navigator?.language ? navigator?.language : "en-US";
var latlonFormat = new Intl.NumberFormat(lang, {
  style: "unit",
  unit: "degree",
  unitDisplay: "narrow",
  maximumFractionDigits: 2
});
var magFormat = new Intl.NumberFormat(lang, {
  style: "decimal",
  maximumFractionDigits: 2
});
var depthNoUnitFormat = new Intl.NumberFormat(lang, {
  style: "decimal",
  maximumFractionDigits: 2
});
var depthFormat = new Intl.NumberFormat(lang, {
  style: "unit",
  unit: "kilometer",
  unitDisplay: "narrow",
  maximumFractionDigits: 2,
  minimumFractionDigits: 2
});
var depthMeterFormat = new Intl.NumberFormat(lang, {
  style: "unit",
  unit: "meter",
  unitDisplay: "narrow",
  maximumFractionDigits: 1,
  minimumFractionDigits: 1
});

// src/quakeml.ts
import { DateTime as DateTime6 } from "luxon";
var QML_NS = "http://quakeml.org/xmlns/quakeml/1.2";
var BED_NS = "http://quakeml.org/xmlns/bed/1.2";
var IRIS_NS = "http://service.iris.edu/fdsnws/event/1/";
var ANSS_NS = "http://anss.org/xmlns/event/0.1";
var ANSS_CATALOG_NS = "http://anss.org/xmlns/catalog/0.1";
var USGS_HOST = "earthquake.usgs.gov";
var UNKNOWN_MAG_TYPE = "unknown";
var UNKNOWN_PUBLIC_ID = "unknownId";
var FAKE_ORIGIN_TIME = DateTime6.fromISO("1900-01-01T00:00:00Z");
var FAKE_EMPTY_XML2 = '<?xml version="1.0"?><q:quakeml xmlns="http://quakeml.org/xmlns/bed/1.2" xmlns:q="http://quakeml.org/xmlns/quakeml/1.2"><eventParameters publicID="quakeml:fake/empty"></eventParameters></q:quakeml>';
var QUAKE_CLICK_EVENT = "quakeclick";
function createQuakeClickEvent(q, mouseclick) {
  const detail = {
    mouseevent: mouseclick,
    quake: q
  };
  return new CustomEvent(QUAKE_CLICK_EVENT, { detail });
}
var BaseElement = class {
  publicId = UNKNOWN_PUBLIC_ID;
  comments = [];
  creationInfo;
  populate(qml) {
    const pid = _grabAttribute3(qml, "publicID");
    if (!isNonEmptyStringArg(pid)) {
      throw new Error("missing publicID");
    }
    this.publicId = pid;
    this.comments = _grabAllElComment(qml, "comment");
    this.creationInfo = _grabFirstElCreationInfo(qml, "creationInfo");
  }
};
var EventParameters = class _EventParameters extends BaseElement {
  eventList = [];
  description;
  /**
   * Parses a QuakeML event parameters xml element into an EventParameters object.
   *
   * @param eventParametersQML the event parameters xml Element
   * @param host optional source of the xml, helpful for parsing the eventid
   * @returns EventParameters instance
   */
  static createFromXml(eventParametersQML, host) {
    if (eventParametersQML.localName !== "eventParameters") {
      throw new Error(
        `Cannot extract, not a QuakeML event parameters: ${eventParametersQML.localName}`
      );
    }
    const eventEls = Array.from(
      eventParametersQML.getElementsByTagNameNS(BED_NS, "event")
    );
    const events = eventEls.map((e) => Quake.createFromXml(e, host));
    const description = _grabFirstElText3(eventParametersQML, "description");
    const out = new _EventParameters();
    out.populate(eventParametersQML);
    out.eventList = events;
    out.description = description;
    return out;
  }
};
var Quake = class _Quake extends BaseElement {
  eventId;
  descriptionList = [];
  amplitudeList = [];
  stationMagnitudeList = [];
  magnitudeList = [];
  originList = [];
  pickList = [];
  focalMechanismList = [];
  preferredOrigin;
  preferredMagnitude;
  preferredFocalMechanism;
  type;
  typeCertainty;
  /**
   * Parses a QuakeML event xml element into a Quake object. Pass in
   * host=seisplotjs.fdsnevent.USGS_HOST for xml from the USGS service
   * in order to parse the eventid, otherwise this can be left out
   *
   * @param qml the event xml Element
   * @param host optional source of the xml, helpful for parsing the eventid
   * @returns QuakeML Quake(Event) object
   */
  static createFromXml(qml, host) {
    if (qml.localName !== "event") {
      throw new Error(`Cannot extract, not a QuakeML Event: ${qml.localName}`);
    }
    const out = new _Quake();
    out.populate(qml);
    const descriptionEls = Array.from(qml.children).filter(
      (e) => e.tagName === "description"
    );
    out.descriptionList = descriptionEls.map(
      (d) => EventDescription.createFromXml(d)
    );
    const allPickEls = Array.from(qml.getElementsByTagNameNS(BED_NS, "pick"));
    const allPicks = [];
    for (const pickEl of allPickEls) {
      allPicks.push(Pick.createFromXml(pickEl));
    }
    const allAmplitudeEls = Array.from(
      qml.getElementsByTagNameNS(BED_NS, "amplitude")
    );
    const allAmplitudes = [];
    for (const amplitudeEl of allAmplitudeEls) {
      allAmplitudes.push(Amplitude.createFromXml(amplitudeEl, allPicks));
    }
    const allOriginEls = Array.from(
      qml.getElementsByTagNameNS(BED_NS, "origin")
    );
    const allOrigins = [];
    for (const originEl of allOriginEls) {
      allOrigins.push(Origin.createFromXml(originEl, allPicks));
    }
    const allStationMagEls = Array.from(
      qml.getElementsByTagNameNS(BED_NS, "stationMagnitude")
    );
    const allStationMags = [];
    for (const stationMagEl of allStationMagEls) {
      allStationMags.push(
        StationMagnitude.createFromXml(stationMagEl, allOrigins, allAmplitudes)
      );
    }
    const allMagEls = Array.from(
      qml.getElementsByTagNameNS(BED_NS, "magnitude")
    );
    const allMags = [];
    for (const magEl of allMagEls) {
      allMags.push(Magnitude.createFromXml(magEl, allOrigins, allStationMags));
    }
    const allFocalMechEls = Array.from(
      qml.getElementsByTagNameNS(BED_NS, "focalMechanism")
    );
    const allFocalMechs = [];
    for (const focalMechEl of allFocalMechEls) {
      allFocalMechs.push(
        FocalMechanism.createFromXml(focalMechEl, allOrigins, allMags)
      );
    }
    out.originList = allOrigins;
    out.magnitudeList = allMags;
    out.pickList = allPicks;
    out.amplitudeList = allAmplitudes;
    out.stationMagnitudeList = allStationMags;
    out.focalMechanismList = allFocalMechs;
    out.eventId = _Quake.extractEventId(qml, host);
    const preferredOriginId = _grabFirstElText3(qml, "preferredOriginID");
    const preferredMagnitudeId = _grabFirstElText3(qml, "preferredMagnitudeID");
    const preferredFocalMechId = _grabFirstElText3(
      qml,
      "preferredFocalMechanismID"
    );
    if (isNonEmptyStringArg(preferredOriginId)) {
      out.preferredOrigin = allOrigins.find(
        (o) => o.publicId === preferredOriginId
      );
      if (!out.preferredOrigin) {
        throw new Error(`no preferredOriginId match: ${preferredOriginId}`);
      }
    }
    if (isNonEmptyStringArg(preferredMagnitudeId)) {
      out.preferredMagnitude = allMags.find(
        (m) => m.publicId === preferredMagnitudeId
      );
      if (!out.preferredMagnitude) {
        throw new Error(`no match: ${preferredMagnitudeId}`);
      }
    }
    if (isNonEmptyStringArg(preferredFocalMechId)) {
      out.preferredFocalMechanism = allFocalMechs.find(
        (m) => m.publicId === preferredFocalMechId
      );
      if (!out.preferredFocalMechanism) {
        throw new Error(`no match: ${preferredFocalMechId}`);
      }
    }
    out.type = _grabFirstElText3(qml, "type");
    out.typeCertainty = _grabFirstElText3(qml, "typeCertainty");
    return out;
  }
  /**
   * Extracts the EventId from a QuakeML element, guessing from one of several
   * incompatible (grumble grumble) formats.
   *
   * @param   qml Quake(Event) to extract from
   * @param   host optional source of the xml to help determine the event id style
   * @returns     Extracted Id, or "unknownEventId" if we can't figure it out
   */
  static extractEventId(qml, host) {
    const eventId = _grabAttributeNS2(qml, ANSS_CATALOG_NS, "eventid");
    const catalogEventSource = _grabAttributeNS2(
      qml,
      ANSS_CATALOG_NS,
      "eventsource"
    );
    if (isNonEmptyStringArg(eventId)) {
      if (host === USGS_HOST && isNonEmptyStringArg(catalogEventSource)) {
        return catalogEventSource + eventId;
      } else {
        return eventId;
      }
    }
    const publicid = _grabAttribute3(qml, "publicID");
    if (isNonEmptyStringArg(publicid)) {
      let re = /eventid=([\w\d]+)/;
      let parsed = re.exec(publicid);
      if (parsed) {
        return parsed[1];
      }
      re = /evid=([\w\d]+)/;
      parsed = re.exec(publicid);
      if (parsed) {
        return parsed[1];
      }
    }
    return UNKNOWN_PUBLIC_ID;
  }
  hasPreferredOrigin() {
    return isDef(this.preferredOrigin);
  }
  hasOrigin() {
    return isDef(this.preferredOrigin) || this.originList.length > 1;
  }
  get origin() {
    if (isDef(this.preferredOrigin)) {
      return this.preferredOrigin;
    } else if (this.originList.length > 0) {
      return this.originList[0];
    } else {
      throw new Error("No origins in quake");
    }
  }
  hasPreferredMagnitude() {
    return isDef(this.preferredMagnitude);
  }
  hasMagnitude() {
    return isDef(this.preferredMagnitude) || this.magnitudeList.length > 1;
  }
  get magnitude() {
    if (isDef(this.preferredMagnitude)) {
      return this.preferredMagnitude;
    } else if (this.magnitudeList.length > 0) {
      return this.magnitudeList[0];
    } else {
      throw new Error("No magnitudes in quake");
    }
  }
  get time() {
    return this.origin.time;
  }
  get latitude() {
    return this.origin.latitude;
  }
  get longitude() {
    return this.origin.longitude;
  }
  get depth() {
    return this.origin.depth;
  }
  get depthKm() {
    return this.depth / 1e3;
  }
  get description() {
    return this.descriptionList.length > 0 ? this.descriptionList[0].text : "";
  }
  get arrivals() {
    return this.origin.arrivalList;
  }
  get picks() {
    return this.pickList;
  }
  toString() {
    if (this.hasOrigin()) {
      const magStr = this.hasMagnitude() ? this.magnitude.toString() : "";
      const latlon = `(${latlonFormat.format(this.latitude)}/${latlonFormat.format(this.longitude)})`;
      const depth = depthFormat.format(this.depth / 1e3);
      return `${this.time.toISO()} ${latlon} ${depth} ${magStr}`;
    } else if (this.eventId != null) {
      return `Event: ${this.eventId}`;
    } else {
      return `Event: unknown`;
    }
  }
};
var EventDescription = class _EventDescription {
  text;
  type;
  constructor(text) {
    this.text = text;
  }
  /**
   * Parses a QuakeML description xml element into a EventDescription object.
   *
   * @param descriptionQML the description xml Element
   * @returns EventDescription instance
   */
  static createFromXml(descriptionQML) {
    if (descriptionQML.localName !== "description") {
      throw new Error(
        `Cannot extract, not a QuakeML description ID: ${descriptionQML.localName}`
      );
    }
    const text = _grabFirstElText3(descriptionQML, "text");
    if (!isNonEmptyStringArg(text)) {
      throw new Error("description missing text");
    }
    const out = new _EventDescription(text);
    out.type = _grabFirstElText3(descriptionQML, "type");
    return out;
  }
  toString() {
    return this.text;
  }
};
var Amplitude = class _Amplitude extends BaseElement {
  genericAmplitude;
  type;
  category;
  unit;
  methodID;
  period;
  snr;
  timeWindow;
  pick;
  waveformID;
  filterID;
  scalingTime;
  magnitudeHint;
  evaluationMode;
  evaluationStatus;
  constructor(genericAmplitude) {
    super();
    this.genericAmplitude = genericAmplitude;
  }
  /**
   * Parses a QuakeML amplitude xml element into an Amplitude object.
   *
   * @param amplitudeQML the amplitude xml Element
   * @param allPicks picks already extracted from the xml for linking arrivals with picks
   * @returns Amplitude instance
   */
  static createFromXml(amplitudeQML, allPicks) {
    if (amplitudeQML.localName !== "amplitude") {
      throw new Error(
        `Cannot extract, not a QuakeML amplitude: ${amplitudeQML.localName}`
      );
    }
    const genericAmplitude = _grabFirstElRealQuantity(
      amplitudeQML,
      "genericAmplitude"
    );
    if (!isDef(genericAmplitude)) {
      throw new Error("amplitude missing genericAmplitude");
    }
    const out = new _Amplitude(genericAmplitude);
    out.populate(amplitudeQML);
    out.type = _grabFirstElText3(amplitudeQML, "type");
    out.category = _grabFirstElText3(amplitudeQML, "category");
    out.unit = _grabFirstElText3(amplitudeQML, "unit");
    out.methodID = _grabFirstElText3(amplitudeQML, "methodID");
    out.period = _grabFirstElRealQuantity(amplitudeQML, "period");
    out.snr = _grabFirstElFloat3(amplitudeQML, "snr");
    out.timeWindow = _grabFirstElType(
      TimeWindow.createFromXml.bind(TimeWindow)
    )(amplitudeQML, "timeWindow");
    const pickID = _grabFirstElText3(amplitudeQML, "pickID");
    out.pick = allPicks.find((p) => p.publicId === pickID);
    if (pickID && !out.pick) {
      throw new Error("No pick with ID " + pickID);
    }
    out.waveformID = _grabFirstElType(
      WaveformID.createFromXml.bind(WaveformID)
    )(amplitudeQML, "waveformID");
    out.filterID = _grabFirstElText3(amplitudeQML, "filterID");
    out.scalingTime = _grabFirstElTimeQuantity(amplitudeQML, "scalingTime");
    out.magnitudeHint = _grabFirstElText3(amplitudeQML, "magnitudeHint");
    out.evaluationMode = _grabFirstElText3(amplitudeQML, "evaluationMode");
    out.evaluationStatus = _grabFirstElText3(amplitudeQML, "evaluationStatus");
    return out;
  }
};
var StationMagnitude = class _StationMagnitude extends BaseElement {
  origin;
  mag;
  type;
  amplitude;
  methodID;
  waveformID;
  constructor(origin, mag) {
    super();
    this.origin = origin;
    this.mag = mag;
  }
  /**
   * Parses a QuakeML station magnitude xml element into a StationMagnitude object.
   *
   * @param stationMagnitudeQML the station magnitude xml Element
   * @param allOrigins origins already extracted from the xml for linking station magnitudes with origins
   * @param allAmplitudes amplitudes already extracted from the xml for linking station magnitudes with amplitudes
   * @returns StationMagnitude instance
   */
  static createFromXml(stationMagnitudeQML, allOrigins, allAmplitudes) {
    if (stationMagnitudeQML.localName !== "stationMagnitude") {
      throw new Error(
        `Cannot extract, not a QuakeML station magnitude: ${stationMagnitudeQML.localName}`
      );
    }
    const originID = _grabFirstElText3(stationMagnitudeQML, "originID");
    if (!isNonEmptyStringArg(originID)) {
      throw new Error("stationMagnitude missing origin ID");
    }
    const origin = allOrigins.find((o) => o.publicId === originID);
    if (!isDef(origin)) {
      throw new Error("No origin with ID " + originID);
    }
    const mag = _grabFirstElRealQuantity(stationMagnitudeQML, "mag");
    if (!isDef(mag)) {
      throw new Error("stationMagnitude missing mag");
    }
    const out = new _StationMagnitude(origin, mag);
    out.populate(stationMagnitudeQML);
    out.type = _grabFirstElText3(stationMagnitudeQML, "type");
    const amplitudeID = _grabFirstElText3(stationMagnitudeQML, "amplitudeID");
    out.amplitude = allAmplitudes.find((a) => a.publicId === amplitudeID);
    if (amplitudeID && !out.amplitude) {
      throw new Error("No amplitude with ID " + amplitudeID);
    }
    out.methodID = _grabFirstElText3(stationMagnitudeQML, "methodID");
    out.waveformID = _grabFirstElType(
      WaveformID.createFromXml.bind(WaveformID)
    )(stationMagnitudeQML, "waveformID");
    return out;
  }
};
var TimeWindow = class _TimeWindow {
  begin;
  end;
  reference;
  constructor(begin, end, reference) {
    this.begin = begin;
    this.end = end;
    this.reference = reference;
  }
  /**
   * Parses a QuakeML time window xml element into a TimeWindow object.
   *
   * @param timeWindowQML the time window xml Element
   * @returns TimeWindow instance
   */
  static createFromXml(timeWindowQML) {
    if (timeWindowQML.localName !== "timeWindow") {
      throw new Error(
        `Cannot extract, not a QuakeML time window: ${timeWindowQML.localName}`
      );
    }
    const begin = _grabFirstElFloat3(timeWindowQML, "begin");
    if (!isDef(begin)) {
      throw new Error("timeWindow missing begin");
    }
    const end = _grabFirstElFloat3(timeWindowQML, "end");
    if (!isDef(end)) {
      throw new Error("timeWindow missing end");
    }
    const reference = _grabFirstElDateTime(timeWindowQML, "reference");
    if (!isDef(reference)) {
      throw new Error("timeWindow missing reference");
    }
    const out = new _TimeWindow(begin, end, reference);
    return out;
  }
};
var Origin = class _Origin extends BaseElement {
  compositeTimes;
  originUncertainty;
  arrivalList;
  timeQuantity;
  latitudeQuantity;
  longitudeQuantity;
  depthQuantity;
  depthType;
  timeFixed;
  epicenterFixed;
  referenceSystemID;
  methodID;
  earthModelID;
  quality;
  type;
  region;
  evaluationMode;
  evaluationStatus;
  constructor(time, latitude, longitude) {
    super();
    this.compositeTimes = [];
    this.arrivalList = [];
    if (time instanceof DateTime6) {
      this.timeQuantity = new Quantity(time);
    } else {
      this.timeQuantity = time;
    }
    if (typeof latitude == "number") {
      this.latitudeQuantity = new Quantity(latitude);
    } else {
      this.latitudeQuantity = latitude;
    }
    if (typeof longitude == "number") {
      this.longitudeQuantity = new Quantity(longitude);
    } else {
      this.longitudeQuantity = longitude;
    }
  }
  /**
   * Parses a QuakeML origin xml element into a Origin object.
   *
   * @param qml the origin xml Element
   * @param allPicks picks already extracted from the xml for linking arrivals with picks
   * @returns Origin instance
   */
  static createFromXml(qml, allPicks) {
    if (qml.localName !== "origin") {
      throw new Error(`Cannot extract, not a QuakeML Origin: ${qml.localName}`);
    }
    const time = _grabFirstElTimeQuantity(qml, "time");
    if (!isObject(time)) {
      throw new Error("origin missing time");
    }
    const lat = _grabFirstElRealQuantity(qml, "latitude");
    if (!isObject(lat)) {
      throw new Error("origin missing latitude");
    }
    const lon = _grabFirstElRealQuantity(qml, "longitude");
    if (!isObject(lon)) {
      throw new Error("origin missing longitude");
    }
    const out = new _Origin(time, lat, lon);
    out.populate(qml);
    out.originUncertainty = _grabFirstElType(
      OriginUncertainty.createFromXml.bind(OriginUncertainty)
    )(qml, "originUncertainty");
    const allArrivalEls = Array.from(
      qml.getElementsByTagNameNS(BED_NS, "arrival")
    );
    out.arrivalList = allArrivalEls.map(
      (arrivalEl) => Arrival.createFromXml(arrivalEl, allPicks)
    );
    out.depthQuantity = _grabFirstElRealQuantity(qml, "depth");
    out.depthType = _grabFirstElText3(qml, "depthType");
    out.timeFixed = _grabFirstElBool(qml, "timeFixed");
    out.epicenterFixed = _grabFirstElBool(qml, "epicenterFixed");
    out.referenceSystemID = _grabFirstElText3(qml, "referenceSystemID");
    out.methodID = _grabFirstElText3(qml, "methodID");
    out.earthModelID = _grabFirstElText3(qml, "earthModelID");
    out.quality = _grabFirstElType(
      OriginQuality.createFromXml.bind(OriginQuality)
    )(qml, "quality");
    out.type = _grabFirstElText3(qml, "type");
    out.region = _grabFirstElText3(qml, "region");
    out.evaluationMode = _grabFirstElText3(qml, "evaluationMode");
    out.evaluationStatus = _grabFirstElText3(qml, "evaluationStatus");
    return out;
  }
  toString() {
    const latlon = `(${latlonFormat.format(this.latitude)}/${latlonFormat.format(this.longitude)})`;
    const depth = depthFormat.format(this.depth / 1e3);
    return `${this.time.toISO()} ${latlon} ${depth} km`;
  }
  get time() {
    return this.timeQuantity.value;
  }
  set time(t) {
    if (t instanceof DateTime6) {
      this.timeQuantity.value = t;
    } else {
      this.timeQuantity = t;
    }
  }
  get latitude() {
    return this.latitudeQuantity.value;
  }
  set latitude(lat) {
    if (typeof lat == "number") {
      this.latitudeQuantity.value = lat;
    } else {
      this.latitudeQuantity = lat;
    }
  }
  get longitude() {
    return this.longitudeQuantity.value;
  }
  set longitude(lon) {
    if (typeof lon == "number") {
      this.longitudeQuantity.value = lon;
    } else {
      this.longitudeQuantity = lon;
    }
  }
  get depthKm() {
    return this.depth / 1e3;
  }
  get depth() {
    return this.depthQuantity?.value ?? NaN;
  }
  set depth(depth) {
    if (typeof depth == "number") {
      if (!this.depthQuantity) {
        this.depthQuantity = new Quantity(depth);
      } else {
        this.depthQuantity.value = depth;
      }
    } else {
      this.depthQuantity = depth;
    }
  }
  get arrivals() {
    return this.arrivalList;
  }
};
var CompositeTime = class _CompositeTime {
  year;
  month;
  day;
  hour;
  minute;
  second;
  /**
   * Parses a QuakeML composite time xml element into an CompositeTime object.
   *
   * @param qml the composite time xml Element
   * @returns CompositeTime instance
   */
  static createFromXml(qml) {
    if (qml.localName !== "compositeTime") {
      throw new Error(
        `Cannot extract, not a QuakeML Composite Time: ${qml.localName}`
      );
    }
    const out = new _CompositeTime();
    out.year = _grabFirstElIntegerQuantity(qml, "year");
    out.month = _grabFirstElIntegerQuantity(qml, "month");
    out.day = _grabFirstElIntegerQuantity(qml, "day");
    out.hour = _grabFirstElIntegerQuantity(qml, "hour");
    out.minute = _grabFirstElIntegerQuantity(qml, "minute");
    out.second = _grabFirstElIntegerQuantity(qml, "second");
    return out;
  }
};
var OriginUncertainty = class _OriginUncertainty {
  horizontalUncertainty;
  minHorizontalUncertainty;
  maxHorizontalUncertainty;
  azimuthMaxHorizontalUncertainty;
  confidenceEllipsoid;
  preferredDescription;
  confidenceLevel;
  /**
   * Parses a QuakeML origin uncertainty xml element into an OriginUncertainty object.
   *
   * @param qml the origin uncertainty xml Element
   * @returns OriginUncertainty instance
   */
  static createFromXml(qml) {
    if (qml.localName !== "originUncertainty") {
      throw new Error(
        `Cannot extract, not a QuakeML Origin Uncertainty: ${qml.localName}`
      );
    }
    const out = new _OriginUncertainty();
    out.horizontalUncertainty = _grabFirstElFloat3(qml, "horizontalUncertainty");
    out.minHorizontalUncertainty = _grabFirstElFloat3(
      qml,
      "minHorizontalUncertainty"
    );
    out.maxHorizontalUncertainty = _grabFirstElFloat3(
      qml,
      "maxHorizontalUncertainty"
    );
    out.azimuthMaxHorizontalUncertainty = _grabFirstElFloat3(
      qml,
      "azimuthMaxHorizontalUncertainty"
    );
    out.confidenceEllipsoid = _grabFirstElType(
      ConfidenceEllipsoid.createFromXml.bind(ConfidenceEllipsoid)
    )(qml, "confidenceEllipsoid");
    out.preferredDescription = _grabFirstElText3(qml, "preferredDescription");
    out.confidenceLevel = _grabFirstElFloat3(qml, "confidenceLevel");
    return out;
  }
};
var ConfidenceEllipsoid = class _ConfidenceEllipsoid {
  semiMajorAxisLength;
  semiMinorAxisLength;
  semiIntermediateAxisLength;
  majorAxisPlunge;
  majorAxisAzimuth;
  majorAxisRotation;
  constructor(semiMajorAxisLength, semiMinorAxisLength, semiIntermediateAxisLength, majorAxisPlunge, majorAxisAzimuth, majorAxisRotation) {
    this.semiMajorAxisLength = semiMajorAxisLength;
    this.semiMinorAxisLength = semiMinorAxisLength;
    this.semiIntermediateAxisLength = semiIntermediateAxisLength;
    this.majorAxisPlunge = majorAxisPlunge;
    this.majorAxisAzimuth = majorAxisAzimuth;
    this.majorAxisRotation = majorAxisRotation;
  }
  /**
   * Parses a QuakeML confidence ellipsoid xml element into an ConfidenceEllipsoid object.
   *
   * @param qml the confidence ellipsoid xml Element
   * @returns ConfidenceEllipsoid instance
   */
  static createFromXml(qml) {
    if (qml.localName !== "confidenceEllipsoid") {
      throw new Error(
        `Cannot extract, not a QuakeML Confidence Ellipsoid: ${qml.localName}`
      );
    }
    const semiMajorAxisLength = _grabFirstElFloat3(qml, "semiMajorAxisLength");
    if (semiMajorAxisLength === void 0) {
      throw new Error("confidenceEllipsoid missing semiMajorAxisLength");
    }
    const semiMinorAxisLength = _grabFirstElFloat3(qml, "semiMinorAxisLength");
    if (semiMinorAxisLength === void 0) {
      throw new Error("confidenceEllipsoid missing semiMinorAxisLength");
    }
    const semiIntermediateAxisLength = _grabFirstElFloat3(
      qml,
      "semiIntermediateAxisLength"
    );
    if (semiIntermediateAxisLength === void 0) {
      throw new Error("confidenceEllipsoid missing semiIntermediateAxisLength");
    }
    const majorAxisPlunge = _grabFirstElFloat3(qml, "majorAxisPlunge");
    if (majorAxisPlunge === void 0) {
      throw new Error("confidenceEllipsoid missing majorAxisPlunge");
    }
    const majorAxisAzimuth = _grabFirstElFloat3(qml, "majorAxisAzimuth");
    if (majorAxisAzimuth === void 0) {
      throw new Error("confidenceEllipsoid missing majorAxisAzimuth");
    }
    const majorAxisRotation = _grabFirstElFloat3(qml, "majorAxisRotation");
    if (majorAxisRotation === void 0) {
      throw new Error("confidenceEllipsoid missing majorAxisRotation");
    }
    const out = new _ConfidenceEllipsoid(
      semiMajorAxisLength,
      semiMinorAxisLength,
      semiIntermediateAxisLength,
      majorAxisPlunge,
      majorAxisAzimuth,
      majorAxisRotation
    );
    return out;
  }
};
var OriginQuality = class _OriginQuality {
  associatedPhaseCount;
  usedPhaseCount;
  associatedStationCount;
  usedStationCount;
  depthPhaseCount;
  standardError;
  azimuthalGap;
  secondaryAzimuthalGap;
  groundTruthLevel;
  maximumDistance;
  minimumDistance;
  medianDistance;
  /**
   * Parses a QuakeML origin quality xml element into an OriginQuality object.
   *
   * @param qml the origin quality xml Element
   * @returns OriginQuality instance
   */
  static createFromXml(qml) {
    if (qml.localName !== "quality") {
      throw new Error(
        `Cannot extract, not a QuakeML Origin Quality: ${qml.localName}`
      );
    }
    const out = new _OriginQuality();
    out.associatedPhaseCount = _grabFirstElInt3(qml, "associatedPhaseCount");
    out.usedPhaseCount = _grabFirstElInt3(qml, "usedPhaseCount");
    out.associatedStationCount = _grabFirstElInt3(qml, "associatedStationCount");
    out.usedStationCount = _grabFirstElInt3(qml, "usedStationCount");
    out.standardError = _grabFirstElFloat3(qml, "standardError");
    out.azimuthalGap = _grabFirstElFloat3(qml, "azimuthalGap");
    out.secondaryAzimuthalGap = _grabFirstElFloat3(qml, "secondaryAzimuthalGap");
    out.groundTruthLevel = _grabFirstElText3(qml, "groundTruthLevel");
    out.maximumDistance = _grabFirstElFloat3(qml, "maximumDistance");
    out.minimumDistance = _grabFirstElFloat3(qml, "minimumDistance");
    out.medianDistance = _grabFirstElFloat3(qml, "medianDistance");
    return out;
  }
};
var Magnitude = class _Magnitude extends BaseElement {
  stationMagnitudeContributions = [];
  magQuantity;
  type;
  origin;
  methodID;
  stationCount;
  azimuthalGap;
  evaluationMode;
  evaluationStatus;
  constructor(mag, type) {
    super();
    if (typeof mag === "number") {
      this.magQuantity = new Quantity(mag);
    } else {
      this.magQuantity = mag;
    }
    if (type) {
      this.type = type;
    }
  }
  /**
   * Parses a QuakeML magnitude xml element into a Magnitude object.
   *
   * @param qml the magnitude xml Element
   * @param allOrigins origins already extracted from the xml for linking magnitudes with origins
   * @param allStationMagnitudes station magnitudes already extracted from the xml
   * @returns Magnitude instance
   */
  static createFromXml(qml, allOrigins, allStationMagnitudes) {
    if (qml.localName !== "magnitude") {
      throw new Error(
        `Cannot extract, not a QuakeML Magnitude: ${qml.localName}`
      );
    }
    const mag = _grabFirstElRealQuantity(qml, "mag");
    if (!mag) {
      throw new Error("magnitude missing mag");
    }
    const out = new _Magnitude(mag);
    out.populate(qml);
    const stationMagnitudeContributionEls = Array.from(
      qml.getElementsByTagNameNS(BED_NS, "stationMagnitudeContribution")
    );
    out.stationMagnitudeContributions = stationMagnitudeContributionEls.map(
      (smc) => StationMagnitudeContribution.createFromXml(smc, allStationMagnitudes)
    );
    out.type = _grabFirstElText3(qml, "type");
    const originID = _grabFirstElText3(qml, "originID");
    out.origin = allOrigins.find((o) => o.publicId === originID);
    if (originID && !out.origin) {
      throw new Error("No origin with ID " + originID);
    }
    out.methodID = _grabFirstElText3(qml, "methodID");
    out.stationCount = _grabFirstElInt3(qml, "stationCount");
    out.azimuthalGap = _grabFirstElFloat3(qml, "azimuthalGap");
    out.evaluationMode = _grabFirstElText3(qml, "evaluationMode");
    out.evaluationStatus = _grabFirstElText3(qml, "evaluationStatus");
    return out;
  }
  toString() {
    return `${magFormat.format(this.mag)} ${this.type ? this.type : ""}`;
  }
  get mag() {
    return this.magQuantity.value;
  }
  set mag(value) {
    if (typeof value === "number") {
      this.magQuantity.value = value;
    } else {
      this.magQuantity = value;
    }
  }
};
var StationMagnitudeContribution = class _StationMagnitudeContribution {
  stationMagnitude;
  residual;
  weight;
  constructor(stationMagnitude) {
    this.stationMagnitude = stationMagnitude;
  }
  /**
   * Parses a QuakeML station magnitude contribution xml element into a StationMagnitudeContribution object.
   *
   * @param qml the station magnitude contribution xml Element
   * @param allStationMagnitudes station magnitudes already extracted from the xml for linking station magnitudes with station magnitude contributions
   * @returns StationMagnitudeContribution instance
   */
  static createFromXml(qml, allStationMagnitudes) {
    if (qml.localName !== "stationMagnitudeContribution") {
      throw new Error(
        `Cannot extract, not a QuakeML StationMagnitudeContribution: ${qml.localName}`
      );
    }
    const stationMagnitudeID = _grabFirstElText3(qml, "stationMagnitudeID");
    if (!isNonEmptyStringArg(stationMagnitudeID)) {
      throw new Error("stationMagnitudeContribution missing stationMagnitude");
    }
    const stationMagnitude = allStationMagnitudes.find(
      (sm) => sm.publicId === stationMagnitudeID
    );
    if (!isDef(stationMagnitude)) {
      throw new Error("No stationMagnitude with ID " + stationMagnitudeID);
    }
    const out = new _StationMagnitudeContribution(stationMagnitude);
    out.residual = _grabFirstElFloat3(qml, "residual");
    out.weight = _grabFirstElFloat3(qml, "weight");
    return out;
  }
};
var Arrival = class _Arrival extends BaseElement {
  phase;
  pick;
  timeCorrection;
  azimuth;
  distance;
  takeoffAngle;
  timeResidual;
  horizontalSlownessResidual;
  backazimuthResidual;
  timeWeight;
  horizontalSlownessWeight;
  backazimuthWeight;
  earthModelID;
  constructor(phase, pick) {
    super();
    this.phase = phase;
    this.pick = pick;
  }
  /**
   * Parses a QuakeML arrival xml element into a Arrival object.
   *
   * @param arrivalQML the arrival xml Element
   * @param allPicks picks already extracted from the xml for linking arrivals with picks
   * @returns Arrival instance
   */
  static createFromXml(arrivalQML, allPicks) {
    if (arrivalQML.localName !== "arrival") {
      throw new Error(
        `Cannot extract, not a QuakeML Arrival: ${arrivalQML.localName}`
      );
    }
    const pickId = _grabFirstElText3(arrivalQML, "pickID");
    const phase = _grabFirstElText3(arrivalQML, "phase");
    if (isNonEmptyStringArg(phase) && isNonEmptyStringArg(pickId)) {
      const myPick = allPicks.find(function(p) {
        return p.publicId === pickId;
      });
      if (!myPick) {
        throw new Error("Can't find pick with Id=" + pickId + " for Arrival");
      }
      const out = new _Arrival(phase, myPick);
      out.populate(arrivalQML);
      out.timeCorrection = _grabFirstElFloat3(arrivalQML, "timeCorrection");
      out.azimuth = _grabFirstElFloat3(arrivalQML, "azimuth");
      out.distance = _grabFirstElFloat3(arrivalQML, "distance");
      out.takeoffAngle = _grabFirstElRealQuantity(arrivalQML, "takeoffAngle");
      out.timeResidual = _grabFirstElFloat3(arrivalQML, "timeResidual");
      out.horizontalSlownessResidual = _grabFirstElFloat3(
        arrivalQML,
        "horizontalSlownessResidual"
      );
      out.backazimuthResidual = _grabFirstElFloat3(
        arrivalQML,
        "backazimuthResidual"
      );
      out.timeWeight = _grabFirstElFloat3(arrivalQML, "timeWeight");
      out.horizontalSlownessWeight = _grabFirstElFloat3(
        arrivalQML,
        "horizontalSlownessWeight"
      );
      out.backazimuthWeight = _grabFirstElFloat3(
        arrivalQML,
        "backazimuthWeight"
      );
      out.earthModelID = _grabFirstElText3(arrivalQML, "earthModelID");
      return out;
    } else {
      throw new Error(
        "Arrival does not have phase or pickId: " + stringify(phase) + " " + stringify(pickId)
      );
    }
  }
};
var Pick = class _Pick extends BaseElement {
  timeQuantity;
  waveformID;
  filterID;
  methodID;
  horizontalSlowness;
  backazimuth;
  slownessMethodID;
  onset;
  phaseHint;
  polarity;
  evaluationMode;
  evaluationStatus;
  constructor(time, waveformID) {
    super();
    if (time instanceof DateTime6) {
      this.timeQuantity = new Quantity(time);
    } else {
      this.timeQuantity = time;
    }
    this.waveformID = waveformID;
  }
  get time() {
    return this.timeQuantity.value;
  }
  set time(t) {
    if (t instanceof DateTime6) {
      this.timeQuantity.value = t;
    } else {
      this.timeQuantity = t;
    }
  }
  /**
   * Parses a QuakeML pick xml element into a Pick object.
   *
   * @param pickQML the pick xml Element
   * @returns Pick instance
   */
  static createFromXml(pickQML) {
    if (pickQML.localName !== "pick") {
      throw new Error(
        `Cannot extract, not a QuakeML Pick: ${pickQML.localName}`
      );
    }
    const time = _grabFirstElTimeQuantity(pickQML, "time");
    if (!isDef(time)) {
      throw new Error("Missing time");
    }
    const waveformId = _grabFirstElType(
      WaveformID.createFromXml.bind(WaveformID)
    )(pickQML, "waveformID");
    if (!isObject(waveformId)) {
      throw new Error("pick missing waveformID");
    }
    const out = new _Pick(time, waveformId);
    out.populate(pickQML);
    out.filterID = _grabFirstElText3(pickQML, "filterID");
    out.methodID = _grabFirstElText3(pickQML, "methodID");
    out.horizontalSlowness = _grabFirstElRealQuantity(
      pickQML,
      "horizontalSlowness"
    );
    out.backazimuth = _grabFirstElRealQuantity(pickQML, "backazimuth");
    out.slownessMethodID = _grabFirstElText3(pickQML, "slownessMethodID");
    out.onset = _grabFirstElText3(pickQML, "onset");
    out.phaseHint = _grabFirstElText3(pickQML, "phaseHint");
    out.polarity = _grabFirstElText3(pickQML, "polarity");
    out.evaluationMode = _grabFirstElText3(pickQML, "evaluationMode");
    out.evaluationStatus = _grabFirstElText3(pickQML, "evaluationStatus");
    return out;
  }
  get networkCode() {
    return this.waveformID.networkCode;
  }
  get stationCode() {
    return this.waveformID.stationCode;
  }
  get locationCode() {
    return this.waveformID.locationCode || "--";
  }
  get channelCode() {
    return this.waveformID.channelCode || "---";
  }
  isAtStation(station) {
    return this.networkCode === station.networkCode && this.stationCode === station.stationCode;
  }
  isOnChannel(channel) {
    return this.networkCode === channel.station.networkCode && this.stationCode === channel.station.stationCode && this.locationCode === channel.locationCode && this.channelCode === channel.channelCode;
  }
  toString() {
    return stringify(this.time) + ` ${this.networkCode}.${this.stationCode}.${this.locationCode}.${this.channelCode}`;
  }
};
var FocalMechanism = class _FocalMechanism extends BaseElement {
  waveformIDList = [];
  momentTensorList = [];
  triggeringOrigin;
  nodalPlanes;
  principalAxes;
  azimuthalGap;
  stationPolarityCount;
  misfit;
  stationDistributionRatio;
  methodID;
  evaluationMode;
  evaluationStatus;
  /**
   * Parses a QuakeML focal mechanism xml element into a FocalMechanism object.
   *
   * @param focalMechQML the focal mechanism xml Element
   * @param allOrigins origins already extracted from the xml for linking focal mechanisms with origins
   * @param allMagnitudes magnitudes already extracted from the xml for linking moment tensors with magnitudes
   * @returns FocalMechanism instance
   */
  static createFromXml(focalMechQML, allOrigins, allMagnitudes) {
    if (focalMechQML.localName !== "focalMechanism") {
      throw new Error(
        `Cannot extract, not a QuakeML focalMechanism: ${focalMechQML.localName}`
      );
    }
    const out = new _FocalMechanism();
    out.populate(focalMechQML);
    const waveformIDEls = Array.from(
      focalMechQML.getElementsByTagNameNS(BED_NS, "waveformID")
    );
    out.waveformIDList = waveformIDEls.map(
      (wid) => WaveformID.createFromXml(wid)
    );
    const momentTensorEls = Array.from(
      focalMechQML.getElementsByTagNameNS(BED_NS, "momentTensor")
    );
    out.momentTensorList = momentTensorEls.map(
      (mt) => MomentTensor.createFromXml(mt, allOrigins, allMagnitudes)
    );
    const triggeringOriginID = _grabFirstElText3(
      focalMechQML,
      "triggeringOriginID"
    );
    out.triggeringOrigin = allOrigins.find(
      (o) => o.publicId === triggeringOriginID
    );
    if (triggeringOriginID && !out.triggeringOrigin) {
      throw new Error("No origin with ID " + triggeringOriginID);
    }
    out.nodalPlanes = _grabFirstElType(
      NodalPlanes.createFromXml.bind(NodalPlanes)
    )(focalMechQML, "nodalPlanes");
    out.principalAxes = _grabFirstElType(
      PrincipalAxes.createFromXml.bind(PrincipalAxes)
    )(focalMechQML, "principalAxes");
    out.azimuthalGap = _grabFirstElFloat3(focalMechQML, "azimuthalGap");
    out.stationPolarityCount = _grabFirstElInt3(
      focalMechQML,
      "stationPolarityCount"
    );
    out.misfit = _grabFirstElFloat3(focalMechQML, "misfit");
    out.stationDistributionRatio = _grabFirstElFloat3(
      focalMechQML,
      "stationDistributionRatio"
    );
    out.methodID = _grabFirstElText3(focalMechQML, "methodID");
    out.evaluationMode = _grabFirstElText3(focalMechQML, "evaluationMode");
    out.evaluationStatus = _grabFirstElText3(focalMechQML, "evaluationStatus");
    return out;
  }
};
var NodalPlanes = class _NodalPlanes {
  nodalPlane1;
  nodalPlane2;
  preferredPlane;
  /**
   * Parses a QuakeML nodal planes xml element into a NodalPlanes object.
   *
   * @param nodalPlanesQML the nodal planes xml Element
   * @returns NodalPlanes instance
   */
  static createFromXml(nodalPlanesQML) {
    const out = new _NodalPlanes();
    out.nodalPlane1 = _grabFirstElType(
      NodalPlane.createFromXml.bind(NodalPlane)
    )(nodalPlanesQML, "nodalPlane1");
    out.nodalPlane2 = _grabFirstElType(
      NodalPlane.createFromXml.bind(NodalPlane)
    )(nodalPlanesQML, "nodalPlane2");
    const preferredPlaneString = _grabAttribute3(
      nodalPlanesQML,
      "preferredPlane"
    );
    out.preferredPlane = isNonEmptyStringArg(preferredPlaneString) ? parseInt(preferredPlaneString) : void 0;
    return out;
  }
};
var NodalPlane = class _NodalPlane {
  strike;
  dip;
  rake;
  constructor(strike, dip, rake) {
    this.strike = strike;
    this.dip = dip;
    this.rake = rake;
  }
  /**
   * Parses a QuakeML nodal plane xml element into a NodalPlane object.
   *
   * @param nodalPlaneQML the nodal plane xml Element
   * @returns NodalPlane instance
   */
  static createFromXml(nodalPlaneQML) {
    const strike = _grabFirstElRealQuantity(nodalPlaneQML, "strike");
    if (!isObject(strike)) {
      throw new Error("nodal plane missing strike");
    }
    const dip = _grabFirstElRealQuantity(nodalPlaneQML, "dip");
    if (!isObject(dip)) {
      throw new Error("nodal plane missing dip");
    }
    const rake = _grabFirstElRealQuantity(nodalPlaneQML, "rake");
    if (!isObject(rake)) {
      throw new Error("nodal plane missing rake");
    }
    const out = new _NodalPlane(strike, dip, rake);
    return out;
  }
};
var PrincipalAxes = class _PrincipalAxes {
  tAxis;
  pAxis;
  nAxis;
  constructor(tAxis, pAxis) {
    this.tAxis = tAxis;
    this.pAxis = pAxis;
  }
  /**
   * Parses a QuakeML princpalAxes element into a PrincipalAxes object.
   *
   * @param princpalAxesQML the princpalAxes xml Element
   * @returns PrincipalAxes instance
   */
  static createFromXml(princpalAxesQML) {
    if (princpalAxesQML.localName !== "principalAxes") {
      throw new Error(
        `Cannot extract, not a QuakeML princpalAxes: ${princpalAxesQML.localName}`
      );
    }
    const tAxis = _grabFirstElType(Axis.createFromXml.bind(Axis))(
      princpalAxesQML,
      "tAxis"
    );
    if (!isObject(tAxis)) {
      throw new Error("nodal plane missing tAxis");
    }
    const pAxis = _grabFirstElType(Axis.createFromXml.bind(Axis))(
      princpalAxesQML,
      "pAxis"
    );
    if (!isObject(pAxis)) {
      throw new Error("nodal plane missing pAxis");
    }
    const out = new _PrincipalAxes(tAxis, pAxis);
    out.nAxis = _grabFirstElType(Axis.createFromXml.bind(Axis))(
      princpalAxesQML,
      "nAxis"
    );
    return out;
  }
};
var Axis = class _Axis {
  azimuth;
  plunge;
  length;
  constructor(azimuth, plunge, length) {
    this.azimuth = azimuth;
    this.plunge = plunge;
    this.length = length;
  }
  /**
   * Parses a QuakeML axis xml element into a Axis object.
   *
   * @param axisQML the axis xml Element
   * @returns Axis instance
   */
  static createFromXml(axisQML) {
    const azimuth = _grabFirstElRealQuantity(axisQML, "azimuth");
    if (!isObject(azimuth)) {
      throw new Error("nodal plane missing azimuth");
    }
    const plunge = _grabFirstElRealQuantity(axisQML, "plunge");
    if (!isObject(plunge)) {
      throw new Error("nodal plane missing plunge");
    }
    const length = _grabFirstElRealQuantity(axisQML, "length");
    if (!isObject(length)) {
      throw new Error("nodal plane missing length");
    }
    const out = new _Axis(azimuth, plunge, length);
    return out;
  }
};
var MomentTensor = class _MomentTensor extends BaseElement {
  dataUsedList = [];
  derivedOrigin;
  momentMagnitude;
  scalarMoment;
  tensor;
  variance;
  varianceReduction;
  doubleCouple;
  clvd;
  iso;
  greensFunctionID;
  filterID;
  sourceTimeFunction;
  methodID;
  category;
  inversionType;
  constructor(derivedOrigin) {
    super();
    this.derivedOrigin = derivedOrigin;
  }
  /**
   * Parses a QuakeML momentTensor xml element into a MomentTensor object.
   *
   * @param momentTensorQML the momentTensor xml Element
   * @param allOrigins origins already extracted from the xml for linking moment tensors with origins
   * @param allMagnitudes magnitudes already extracted from the xml for linking moment tensors with magnitudes
   * @returns MomentTensor instance
   */
  static createFromXml(momentTensorQML, allOrigins, allMagnitudes) {
    if (momentTensorQML.localName !== "momentTensor") {
      throw new Error(
        `Cannot extract, not a QuakeML momentTensor: ${momentTensorQML.localName}`
      );
    }
    const derivedOriginID = _grabFirstElText3(
      momentTensorQML,
      "derivedOriginID"
    );
    if (!isNonEmptyStringArg(derivedOriginID)) {
      throw new Error("momentTensor missing derivedOriginID");
    }
    const derivedOrigin = allOrigins.find(
      (o) => o.publicId === derivedOriginID
    );
    if (!isDef(derivedOrigin)) {
      throw new Error("No origin with ID " + derivedOriginID);
    }
    const out = new _MomentTensor(derivedOrigin);
    out.populate(momentTensorQML);
    const dataUsedEls = Array.from(
      momentTensorQML.getElementsByTagNameNS(BED_NS, "dataUsed")
    );
    out.dataUsedList = dataUsedEls.map(DataUsed.createFromXml.bind(DataUsed));
    const momentMagnitudeID = _grabFirstElText3(
      momentTensorQML,
      "momentMagnitudeID"
    );
    out.momentMagnitude = allMagnitudes.find(
      (o) => o.publicId === momentMagnitudeID
    );
    if (momentMagnitudeID && !out.momentMagnitude) {
      throw new Error("No magnitude with ID " + momentMagnitudeID);
    }
    out.scalarMoment = _grabFirstElRealQuantity(
      momentTensorQML,
      "scalarMoment"
    );
    out.tensor = _grabFirstElType(Tensor.createFromXml.bind(Tensor))(
      momentTensorQML,
      "tensor"
    );
    out.variance = _grabFirstElFloat3(momentTensorQML, "variance");
    out.varianceReduction = _grabFirstElFloat3(
      momentTensorQML,
      "varianceReduction"
    );
    out.doubleCouple = _grabFirstElFloat3(momentTensorQML, "doubleCouple");
    out.clvd = _grabFirstElFloat3(momentTensorQML, "clvd");
    out.iso = _grabFirstElFloat3(momentTensorQML, "iso");
    out.greensFunctionID = _grabFirstElText3(
      momentTensorQML,
      "greensFunctionID"
    );
    out.filterID = _grabFirstElText3(momentTensorQML, "filterID");
    out.sourceTimeFunction = _grabFirstElType(
      SourceTimeFunction.createFromXml.bind(SourceTimeFunction)
    )(momentTensorQML, "sourceTimeFunction");
    out.methodID = _grabFirstElText3(momentTensorQML, "methodID");
    out.category = _grabFirstElText3(momentTensorQML, "category");
    out.inversionType = _grabFirstElText3(momentTensorQML, "inversionType");
    return out;
  }
};
var Tensor = class _Tensor {
  Mrr;
  Mtt;
  Mpp;
  Mrt;
  Mrp;
  Mtp;
  constructor(Mrr, Mtt, Mpp, Mrt, Mrp, Mtp) {
    this.Mrr = Mrr;
    this.Mtt = Mtt;
    this.Mpp = Mpp;
    this.Mrt = Mrt;
    this.Mrp = Mrp;
    this.Mtp = Mtp;
  }
  /**
   * Parses a QuakeML tensor xml element into a Tensor object.
   *
   * @param tensorQML the tensor xml Element
   * @returns Tensor instance
   */
  static createFromXml(tensorQML) {
    if (tensorQML.localName !== "tensor") {
      throw new Error(
        `Cannot extract, not a QuakeML tensor: ${tensorQML.localName}`
      );
    }
    const Mrr = _grabFirstElRealQuantity(tensorQML, "Mrr");
    if (!isObject(Mrr)) {
      throw new Error("tensor missing Mrr");
    }
    const Mtt = _grabFirstElRealQuantity(tensorQML, "Mtt");
    if (!isObject(Mtt)) {
      throw new Error("tensor missing Mtt");
    }
    const Mpp = _grabFirstElRealQuantity(tensorQML, "Mpp");
    if (!isObject(Mpp)) {
      throw new Error("tensor missing Mpp");
    }
    const Mrt = _grabFirstElRealQuantity(tensorQML, "Mrt");
    if (!isObject(Mrt)) {
      throw new Error("tensor missing Mrt");
    }
    const Mrp = _grabFirstElRealQuantity(tensorQML, "Mrp");
    if (!isObject(Mrp)) {
      throw new Error("tensor missing Mrp");
    }
    const Mtp = _grabFirstElRealQuantity(tensorQML, "Mtp");
    if (!isObject(Mtp)) {
      throw new Error("tensor missing Mtp");
    }
    const out = new _Tensor(Mrr, Mtt, Mpp, Mrt, Mrp, Mtp);
    return out;
  }
};
var SourceTimeFunction = class _SourceTimeFunction {
  type;
  duration;
  riseTime;
  decayTime;
  constructor(type, duration) {
    this.type = type;
    this.duration = duration;
  }
  /**
   * Parses a QuakeML sourceTimeFunction xml element into a SourceTimeFunction object.
   *
   * @param sourceTimeFunctionQML the sourceTimeFunction xml Element
   * @returns SourceTimeFunction instance
   */
  static createFromXml(sourceTimeFunctionQML) {
    if (sourceTimeFunctionQML.localName !== "sourceTimeFunction") {
      throw new Error(
        `Cannot extract, not a QuakeML sourceTimeFunction: ${sourceTimeFunctionQML.localName}`
      );
    }
    const type = _grabFirstElText3(sourceTimeFunctionQML, "type");
    if (!isNonEmptyStringArg(type)) {
      throw new Error("sourceTimeFunction missing type");
    }
    const duration = _grabFirstElFloat3(sourceTimeFunctionQML, "duration");
    if (!isDef(duration)) {
      throw new Error("sourceTimeFunction missing duration");
    }
    const out = new _SourceTimeFunction(type, duration);
    out.riseTime = _grabFirstElFloat3(sourceTimeFunctionQML, "riseTime");
    out.decayTime = _grabFirstElFloat3(sourceTimeFunctionQML, "decayTime");
    return out;
  }
};
var DataUsed = class _DataUsed {
  waveType;
  stationCount;
  componentCount;
  shortestPeriod;
  longestPeriod;
  constructor(waveType) {
    this.waveType = waveType;
  }
  /**
   * Parses a QuakeML dataUsed xml element into a DataUsed object.
   *
   * @param dataUsedQML the dataUsed xml Element
   * @returns SourceTimeFunction instance
   */
  static createFromXml(dataUsedQML) {
    if (dataUsedQML.localName !== "dataUsed") {
      throw new Error(
        `Cannot extract, not a QuakeML dataUsed: ${dataUsedQML.localName}`
      );
    }
    const waveType = _grabFirstElText3(dataUsedQML, "waveType");
    if (!isNonEmptyStringArg(waveType)) {
      throw new Error("dataUsed missing waveType");
    }
    const out = new _DataUsed(waveType);
    out.stationCount = _grabFirstElInt3(dataUsedQML, "stationCount");
    out.componentCount = _grabFirstElInt3(dataUsedQML, "componentCount");
    out.shortestPeriod = _grabFirstElFloat3(dataUsedQML, "shortestPeriod");
    out.longestPeriod = _grabFirstElFloat3(dataUsedQML, "longestPeriod");
    return out;
  }
};
var WaveformID = class _WaveformID {
  networkCode;
  stationCode;
  channelCode;
  locationCode;
  constructor(networkCode, stationCode) {
    this.networkCode = networkCode;
    this.stationCode = stationCode;
  }
  /**
   * Parses a QuakeML waveform ID xml element into a WaveformID object.
   *
   * @param waveformQML the waveform ID xml Element
   * @returns WaveformID instance
   */
  static createFromXml(waveformQML) {
    if (waveformQML.localName !== "waveformID") {
      throw new Error(
        `Cannot extract, not a QuakeML waveform ID: ${waveformQML.localName}`
      );
    }
    const networkCode = _grabAttribute3(waveformQML, "networkCode");
    if (!isNonEmptyStringArg(networkCode)) {
      throw new Error("waveformID missing networkCode");
    }
    const stationCode = _grabAttribute3(waveformQML, "stationCode");
    if (!isNonEmptyStringArg(stationCode)) {
      throw new Error("waveformID missing stationCode");
    }
    const out = new _WaveformID(networkCode, stationCode);
    out.channelCode = _grabAttribute3(waveformQML, "channelCode");
    out.locationCode = _grabAttribute3(waveformQML, "locationCode");
    return out;
  }
  toString() {
    return `${this.networkCode}.${this.stationCode}.${this.locationCode || "--"}.${this.channelCode || "---"}`;
  }
};
var Quantity = class _Quantity {
  value;
  uncertainty;
  lowerUncertainty;
  upperUncertainty;
  confidenceLevel;
  constructor(value) {
    this.value = value;
  }
  /**
   * Parses a QuakeML quantity xml element into a Quantity object.
   *
   * @param quantityQML the quantity xml Element
   * @param grab a callback to obtain the value
   * @param grabUncertainty a callback to obtain the uncertainties
   * @returns Quantity instance
   */
  static _createFromXml(quantityQML, grab, grabUncertainty) {
    const value = grab(quantityQML, "value");
    if (value === void 0) {
      throw new Error("missing value");
    }
    const out = new _Quantity(value);
    out.uncertainty = grabUncertainty(quantityQML, "uncertainty");
    out.lowerUncertainty = grabUncertainty(quantityQML, "lowerUncertainty");
    out.upperUncertainty = grabUncertainty(quantityQML, "upperUncertainty");
    out.confidenceLevel = _grabFirstElFloat3(quantityQML, "confidenceLevel");
    return out;
  }
  /**
   * Parses a QuakeML real quantity xml element into a RealQuantity object.
   *
   * @param realQuantityQML the real quantity xml Element
   * @returns RealQuantity instance
   */
  static createRealQuantityFromXml(realQuantityQML) {
    return _Quantity._createFromXml(
      realQuantityQML,
      _grabFirstElFloat3,
      _grabFirstElFloat3
    );
  }
  /**
   * Parses a QuakeML integer quantity xml element into a RealQuantity object.
   *
   * @param integerQuantityQML the integer quantity xml Element
   * @returns IntegerQuantity instance
   */
  static createIntegerQuantityFromXml(integerQuantityQML) {
    return _Quantity._createFromXml(
      integerQuantityQML,
      _grabFirstElFloat3,
      _grabFirstElInt3
    );
  }
  /**
   * Parses a QuakeML time quantity xml element into a TimeQuantity object.
   *
   * @param timeQuantityQML the time quantity xml Element
   * @returns TimeQuantity instance
   */
  static createTimeQuantityFromXml(timeQuantityQML) {
    return _Quantity._createFromXml(
      timeQuantityQML,
      _grabFirstElDateTime,
      _grabFirstElFloat3
    );
  }
};
var Comment2 = class _Comment {
  text;
  creationInfo;
  constructor(text) {
    this.text = text;
  }
  /**
   * Parses a QuakeML comment xml element into a Comment object.
   *
   * @param commentQML the comment xml Element
   * @returns Comment instance
   */
  static createFromXml(commentQML) {
    const text = _grabFirstElText3(commentQML, "text");
    if (text === void 0) {
      throw new Error("missing value");
    }
    const out = new _Comment(text);
    out.creationInfo = _grabFirstElCreationInfo(commentQML, "creationInfo");
    return out;
  }
};
var CreationInfo = class _CreationInfo {
  agencyID;
  agencyURI;
  author;
  authorURI;
  creationTime;
  version;
  /**
   * Parses a QuakeML creation info xml element into a CreationInfo object.
   *
   * @param creationInfoQML the creation info xml Element
   * @returns CreationInfo instance
   */
  static createFromXml(creationInfoQML) {
    const out = new _CreationInfo();
    out.agencyID = _grabFirstElText3(creationInfoQML, "agencyID");
    out.agencyURI = _grabFirstElText3(creationInfoQML, "agencyURI");
    out.author = _grabFirstElText3(creationInfoQML, "author");
    out.authorURI = _grabFirstElText3(creationInfoQML, "authorURI");
    out.creationTime = _grabFirstElDateTime(creationInfoQML, "creationTime");
    out.version = _grabFirstElText3(creationInfoQML, "version");
    return out;
  }
};
function parseQuakeML(rawXml, host) {
  const top = rawXml.documentElement;
  if (!top) {
    throw new Error("Can't get documentElement");
  }
  const eventParametersArray = Array.from(
    top.getElementsByTagName("eventParameters")
  );
  if (eventParametersArray.length !== 1) {
    throw new Error(
      `Document has ${eventParametersArray.length} eventParameters elements`
    );
  }
  return EventParameters.createFromXml(eventParametersArray[0], host);
}
function createQuakeFromValues(publicId, time, latitude, longitude, depth_meter) {
  const origin = new Origin(
    new Quantity(time),
    new Quantity(latitude),
    new Quantity(longitude)
  );
  origin.depth = new Quantity(depth_meter);
  const quake = new Quake();
  quake.publicId = publicId;
  quake.originList.push(origin);
  quake.preferredOrigin = origin;
  return quake;
}
function fetchQuakeML(url, timeoutSec2 = 10, nodata = 204) {
  const fetchInit = defaultFetchInitObj(XML_MIME);
  const host = new URL(url).hostname;
  return doFetchWithTimeout(url, fetchInit, timeoutSec2 * 1e3).then((response) => {
    if (response.status === 200) {
      return response.text();
    } else if (response.status === 204 || isDef(nodata) && response.status === nodata) {
      return FAKE_EMPTY_XML2;
    } else {
      throw new Error(`Status not successful: ${response.status}`);
    }
  }).then(function(rawXmlText) {
    return new DOMParser().parseFromString(rawXmlText, XML_MIME);
  }).then((rawXml) => {
    return parseQuakeML(rawXml, host);
  });
}
var _grabAllElComment = function(xml, tagName) {
  const out = [];
  if (isObject(xml)) {
    const elList = Array.from(xml.children).filter(
      (e) => e.tagName === tagName
    );
    for (const el of elList) {
      if (isObject(el)) {
        out.push(Comment2.createFromXml(el));
      }
    }
  }
  return out;
};
var _grabFirstElNS = function(xml, namespace, tagName) {
  let out = void 0;
  if (isObject(xml)) {
    const elList = xml.getElementsByTagNameNS(namespace, tagName);
    if (isObject(elList) && elList.length > 0) {
      const e = elList.item(0);
      if (e) {
        out = e;
      }
    }
  }
  return out;
};
var _grabFirstEl2 = function(xml, tagName) {
  if (isObject(xml)) {
    const elList = Array.from(xml.children).filter(
      (e) => e.tagName === tagName
    );
    if (elList.length > 0) {
      const e = elList[0];
      if (e) {
        return e;
      }
    }
  }
  return void 0;
};
var _grabFirstElText3 = function(xml, tagName) {
  let out = void 0;
  const el = _grabFirstEl2(xml, tagName);
  if (isObject(el)) {
    out = el.textContent;
    if (out === null) {
      out = void 0;
    }
  }
  return out;
};
var _grabFirstElBool = function(xml, tagName) {
  const el = _grabFirstElText3(xml, tagName);
  if (!isStringArg(el)) {
    return void 0;
  }
  switch (el) {
    case "true":
    case "1":
      return true;
    case "false":
    case "0":
      return false;
  }
  throw new Error("Invalid boolean: " + el);
};
var _grabFirstElInt3 = function(xml, tagName) {
  let out = void 0;
  const el = _grabFirstElText3(xml, tagName);
  if (isStringArg(el)) {
    out = parseInt(el);
  }
  return out;
};
var _grabFirstElFloat3 = function(xml, tagName) {
  let out = void 0;
  const el = _grabFirstElText3(xml, tagName);
  if (isStringArg(el)) {
    out = parseFloat(el);
  }
  return out;
};
var _grabFirstElDateTime = function(xml, tagName) {
  let out = void 0;
  const el = _grabFirstElText3(xml, tagName);
  if (isStringArg(el)) {
    out = isoToDateTime(el);
  }
  return out;
};
var _grabFirstElType = function(createFromXml) {
  return function(xml, tagName) {
    let out = void 0;
    const el = _grabFirstEl2(xml, tagName);
    if (isObject(el)) {
      out = createFromXml(el);
    }
    return out;
  };
};
var _grabFirstElRealQuantity = _grabFirstElType(
  Quantity.createRealQuantityFromXml.bind(Quantity)
);
var _grabFirstElIntegerQuantity = _grabFirstElType(
  Quantity.createIntegerQuantityFromXml.bind(Quantity)
);
var _grabFirstElTimeQuantity = _grabFirstElType(
  Quantity.createTimeQuantityFromXml.bind(Quantity)
);
var _grabFirstElCreationInfo = _grabFirstElType(
  CreationInfo.createFromXml.bind(CreationInfo)
);
var _grabAttribute3 = function(xml, tagName) {
  let out = void 0;
  if (isObject(xml)) {
    const a = xml.getAttribute(tagName);
    if (isStringArg(a)) {
      out = a;
    }
  }
  return out;
};
var _requireAttribute3 = function _requireAttribute4(xml, tagName) {
  const out = _grabAttribute3(xml, tagName);
  if (typeof out !== "string") {
    throw new Error(`Attribute ${tagName} not found.`);
  }
  return out;
};
var _grabAttributeNS2 = function(xml, namespace, tagName) {
  let out = void 0;
  if (isObject(xml)) {
    const a = xml.getAttributeNS(namespace, tagName);
    if (isStringArg(a)) {
      out = a;
    }
  }
  return out;
};
var parseUtil2 = {
  _grabFirstEl: _grabFirstEl2,
  _grabFirstElNS,
  _grabFirstElText: _grabFirstElText3,
  _grabFirstElFloat: _grabFirstElFloat3,
  _grabFirstElInt: _grabFirstElInt3,
  _grabAttribute: _grabAttribute3,
  _requireAttribute: _requireAttribute3,
  _grabAttributeNS: _grabAttributeNS2
};

// src/mseed3eh.ts
function ehToQuake(exHead) {
  const bag = extractBagEH(exHead);
  const origin = bag?.ev?.or;
  let q = null;
  if (origin != null) {
    const time = isoToDateTime(origin.tm);
    q = createQuakeFromValues("extraheader", time, origin.la, origin.lo, origin.dp * 1e3);
    if (bag?.ev?.mag?.v != null) {
      const magtype = bag.ev.mag.t == null ? "" : bag.ev.mag.t;
      const mag = new Magnitude(bag.ev.mag.v, magtype);
      q.preferredMagnitude = mag;
    }
  }
  return q;
}
function markerTypeFromEH(mtype) {
  if (mtype === "pk" || mtype === "pick") {
    return "pick";
  }
  if (mtype === "md" || mtype === "predicted") {
    return "predicted";
  }
  return mtype;
}
function ehToMarkers(exHead) {
  const bag = extractBagEH(exHead);
  const markList = bag?.mark;
  if (markList != null) {
    return markList.map((m) => {
      return {
        time: isoToDateTime(m.tm),
        name: m.n,
        markertype: m.mtype == null ? "unknown" : markerTypeFromEH(m.mtype),
        description: m.desc == null ? "" : m.desc
      };
    });
  }
  return [];
}
function extractBagEH(jsonEH) {
  if (!jsonEH || typeof jsonEH !== "object") {
    return null;
  }
  const eh = jsonEH;
  if (typeof eh.bag != "object") {
    return null;
  }
  const object = eh.bag;
  if (isValidBagJsonEHType(object)) {
    return object;
  } else {
    throw new TypeError(`Oops, we did not get Bag extra header JSON!`);
  }
}
function isValidBagChannelJsonEHType(v) {
  if (!v || typeof v !== "object") {
    return false;
  }
  const object = v;
  const answer = typeof object.la === "number" && typeof object.lo === "number" && (typeof object.code === "undefined" || typeof object.code === "string") && (typeof object.el === "undefined" || typeof object.el === "number") && (typeof object.dp === "undefined" || typeof object.dp === "number");
  return answer;
}
function isValidBagEventJsonEHType(v) {
  if (!v || typeof v !== "object") {
    return false;
  }
  const object = v;
  return (typeof object.id === "undefined" || typeof object.id === "string") && (typeof object.or === "undefined" || isValidBagOriginJsonEHType(object.or)) && (typeof object.mag === "undefined" || isValidBagMagJsonEHType(object.mag)) && (typeof object.mt === "undefined" || typeof object.mt === "object");
}
function isValidBagOriginJsonEHType(v) {
  if (!v || typeof v !== "object") {
    return false;
  }
  const object = v;
  return typeof object.la === "number" && typeof object.lo === "number" && typeof object.dp === "number" && typeof object.tm === "string";
}
function isValidBagMagJsonEHType(v) {
  if (!v || typeof v !== "object") {
    return false;
  }
  const object = v;
  return (typeof object.v === "undefined" || typeof object.v === "number") && (typeof object.t === "undefined" || typeof object.t === "string");
}
function isValidBagPathJsonEHType(v) {
  if (!v || typeof v !== "object") {
    return false;
  }
  const object = v;
  return (typeof object.gcarc === "undefined" || typeof object.gcarc === "number") && (typeof object.az === "undefined" || typeof object.az === "number") && (typeof object.baz === "undefined" || typeof object.baz === "number");
}
function isValidBagMarkJsonEHType(v) {
  if (!v || typeof v !== "object") {
    return false;
  }
  const object = v;
  return typeof object.n === "string" && typeof object.tm === "string" && (typeof object.mtype === "undefined" || typeof object.mtype === "string") && (typeof object.desc === "undefined" || typeof object.desc === "string");
}
function isValidBagJsonEHType(v) {
  if (!v || typeof v !== "object") {
    return false;
  }
  const object = v;
  if (!((typeof object.st === "undefined" || isValidBagChannelJsonEHType(object.st)) && (typeof object.ev === "undefined" || isValidBagEventJsonEHType(object.ev)) && (typeof object.path === "undefined" || isValidBagPathJsonEHType(object.path)) && (typeof object.y === "undefined" || typeof object.y === "object") && (typeof object.mark === "undefined" || Array.isArray(object.mark)))) {
    return false;
  }
  const markerList = object.mark;
  if (!(typeof markerList === "undefined" || Array.isArray(markerList))) {
    return false;
  } else {
    if (markerList != null) {
      for (const m of markerList) {
        if (!isValidBagMarkJsonEHType(m)) {
          return false;
        }
      }
    }
  }
  return true;
}

// src/mseed3.ts
import { DateTime as DateTime7, Duration as Duration6 } from "luxon";
var MINISEED_THREE_MIME = "application/vnd.fdsn.mseed3";
var UNKNOWN_DATA_VERSION = 0;
var CRC_OFFSET = 28;
var FIXED_HEADER_SIZE = 40;
var FDSN_PREFIX2 = "FDSN";
var LITTLE_ENDIAN = true;
var BIG_ENDIAN = false;
function toMSeed3(seis, extraHeaders) {
  const out = new Array(0);
  if (!isDef(extraHeaders)) {
    extraHeaders = {};
  }
  for (const seg of seis.segments) {
    const header = new MSeed3Header();
    let rawData;
    let encoding = 0;
    if (seg.isEncoded()) {
      const encoded = seg.getEncoded();
      if (encoded.length === 1) {
        rawData = encoded[0].dataView;
        encoding = encoded[0].compressionType;
      } else {
        const encodeTypeSet = /* @__PURE__ */ new Set();
        encoded.forEach((cur) => {
          encodeTypeSet.add(cur.compressionType);
        });
        const encodeTypes = Array.from(encodeTypeSet.values());
        if (encodeTypes.length > 1) {
          throw new Error(
            `more than one encoding type in seis segment: ${encodeTypes.length}`
          );
        } else if (encodeTypes.length === 0) {
          throw new Error(`zero encoding type in seis segment`);
        } else if (!encodeTypes[0]) {
          throw new Error(`only encoding type is undef`);
        }
        encoding = encodeTypes[0];
        if (!encoding) {
          throw new Error(`encoding is undefined`);
        }
        if (INTEGER || FLOAT || DOUBLE) {
          const totSize = encoded.reduce(
            (acc, cur) => acc + cur.dataView.byteLength,
            0
          );
          const combined = new Uint8Array(totSize);
          encoded.reduce((offset, cur) => {
            combined.set(
              new Uint8Array(
                cur.dataView.buffer,
                cur.dataView.byteOffset,
                cur.dataView.byteLength
              ),
              offset
            );
            return offset + cur.dataView.byteLength;
          }, 0);
          rawData = new DataView(combined.buffer);
          if (encoding === STEIM1 || encoding === STEIM2) {
            rawData.setUint32(
              8,
              encoded[encoded.length - 1].dataView.getUint32(8)
            );
          }
        } else {
          throw new Error(
            `Encoding type not steim 1 or 2 or primitive in seis segment: ${encoding}`
          );
        }
      }
    } else {
      rawData = new DataView(seg.y.buffer);
      if (seg.y instanceof Float32Array) {
        encoding = FLOAT;
      } else if (seg.y instanceof Int32Array) {
        encoding = INTEGER;
      } else if (seg.y instanceof Float64Array) {
        encoding = DOUBLE;
      } else {
        throw new Error("unable to save data of encoding: ");
      }
    }
    header.setStart(seg.startTime);
    header.encoding = encoding;
    if (seg.sampleRate > 1e-3) {
      header.sampleRateOrPeriod = seg.sampleRate;
    } else {
      header.sampleRateOrPeriod = -1 * seg.samplePeriod;
    }
    header.numSamples = seg.numPoints;
    header.publicationVersion = UNKNOWN_DATA_VERSION;
    const sid = seg.sourceId ? seg.sourceId : FDSNSourceId.createUnknown(seg.sampleRate);
    header.identifier = sid.toString();
    header.identifierLength = header.identifier.length;
    header.extraHeaders = extraHeaders;
    header.dataLength = rawData.byteLength;
    const record = new MSeed3Record(header, extraHeaders, rawData);
    record.calcSize();
    out.push(record);
  }
  return out;
}
function parseMSeed3Records(arrayBuffer) {
  const dataRecords = [];
  let offset = 0;
  while (offset < arrayBuffer.byteLength) {
    if (offset > arrayBuffer.byteLength - FIXED_HEADER_SIZE) {
      throw new Error(
        `Not enough bytes left for header, ${arrayBuffer.byteLength - offset} at offset=${offset}`
      );
    }
    const dataView = new DataView(arrayBuffer, offset);
    if (!(dataView.getUint8(0) === 77 && dataView.getUint8(1) === 83)) {
      throw new Error(
        `First byte must be M=77 S=83 at offset=${offset}, but was ${dataView.getUint8(
          0
        )} ${dataView.getUint8(1)}`
      );
    }
    const dr = MSeed3Record.parseSingleDataRecord(dataView);
    dataRecords.push(dr);
    offset += dr.getSize();
  }
  return dataRecords;
}
var MSeed3Record = class _MSeed3Record {
  header;
  extraHeaders;
  rawData;
  constructor(header, extraHeaders, rawData) {
    this.header = header;
    this.rawData = rawData;
    this.extraHeaders = extraHeaders;
  }
  /**
   * Parses an miniseed3 data record from a DataView.
   *
   * @param   dataView bytes to parse
   * @returns parsed record
   */
  static parseSingleDataRecord(dataView) {
    const header = MSeed3Header.createFromDataView(dataView);
    const ehoffset = header.getSize();
    const dataoffset = header.getSize() + header.extraHeadersLength;
    const extraDataView = new DataView(
      dataView.buffer,
      dataView.byteOffset + ehoffset,
      header.extraHeadersLength
    );
    const extraHeaders = parseExtraHeaders(extraDataView);
    const sliceStart = dataView.byteOffset + dataoffset;
    const rawData = new DataView(
      dataView.buffer.slice(sliceStart, sliceStart + header.dataLength)
    );
    const xr = new _MSeed3Record(header, extraHeaders, rawData);
    return xr;
  }
  /**
   * Calculates the byte size of the miniseed3 record to hold this data.
   * This should be called if the size is needed after modification
   * of the extraHeaders.
   *
   * @returns size in bytes
   */
  calcSize() {
    const json = JSON.stringify(this.extraHeaders);
    if (json.length > 2) {
      this.header.extraHeadersLength = json.length;
    } else {
      this.header.extraHeadersLength = 0;
    }
    return this.getSize();
  }
  /**
   * Gets the byte size of the miniseed3 record to hold this data.
   * Note that unless calcSize() has been called, this may not
   * take into account modifications to the extra headers.
   *
   * @returns size in bytes
   */
  getSize() {
    return this.header.getSize() + this.header.extraHeadersLength + this.header.dataLength;
  }
  /**
   * Decompresses the data , if the compression
   *  type is known
   *
   * @returns decompressed data as a typed array, usually Int32Array or Float32Array
   */
  decompress() {
    return this.asEncodedDataSegment().decode();
  }
  /**
   * Wraps data in an EncodedDataSegment for future decompression.
   *
   * @returns waveform data
   */
  asEncodedDataSegment() {
    let swapBytes = LITTLE_ENDIAN;
    if (this.header.encoding === 10 || this.header.encoding === 11 || this.header.encoding === 19) {
      swapBytes = BIG_ENDIAN;
    }
    return new EncodedDataSegment(
      this.header.encoding,
      this.rawData,
      this.header.numSamples,
      swapBytes
    );
  }
  /**
   * Just the header.identifier, included as codes() for compatiblility
   * with parsed miniseed2 data records.
   *
   * @returns string identifier
   */
  codes() {
    return this.header.identifier;
  }
  /**
   * Saves miniseed3 record into a DataView, recalculating crc.
   *
   * @param   dataView DataView to save into, must be large enough to hold the record.
   * @returns the number of bytes written to the DataView, can be used as offset
   * for writting the next record.
   */
  save(dataView) {
    const json = JSON.stringify(this.extraHeaders);
    if (json.length > 2) {
      this.header.extraHeadersLength = json.length;
    } else {
      this.header.extraHeadersLength = 0;
    }
    let offset = this.header.save(dataView, 0, true);
    if (json.length > 2) {
      for (let i = 0; i < json.length; i++) {
        dataView.setInt8(offset, json.charCodeAt(i));
        offset++;
      }
    }
    if (this.rawData !== null) {
      for (let i = 0; i < this.rawData.byteLength; i++) {
        dataView.setUint8(offset + i, this.rawData.getUint8(i));
      }
      offset += this.rawData.byteLength;
    } else {
      throw new Error("rawData is null");
    }
    const dvcrc = dataView.getUint32(CRC_OFFSET, true);
    if (dvcrc !== 0) {
      throw new Error(`CRC is not zero before calculate! ${dvcrc}`);
    }
    const crc = calculateCRC32C(dataView.buffer);
    dataView.setUint32(CRC_OFFSET, crc, true);
    return offset;
  }
  /**
   * Calculates crc by saving to a DataView, which sets the crc header to zero
   * and then calculates it based on the rest of the record.
   *
   * @returns         crc pulled from saved miniseed3 record
   */
  calcCrc() {
    const size = this.calcSize();
    const buff = new ArrayBuffer(size);
    const dataView = new DataView(buff);
    const offset = this.save(dataView);
    if (offset !== size) {
      throw new Error(`expect to write ${size} bytes but only ${offset}`);
    }
    const crc = dataView.getUint32(CRC_OFFSET, true);
    return crc;
  }
  toString() {
    const ehLines = JSON.stringify(this.extraHeaders, null, 2).split("\n");
    const indentLines = ehLines.join("\n          ");
    return `${this.header.toString()}
          extra headers: ${indentLines}`;
  }
};
var MSeed3Header = class _MSeed3Header {
  recordIndicator;
  formatVersion;
  flags;
  nanosecond;
  year;
  dayOfYear;
  hour;
  minute;
  second;
  encoding;
  sampleRateOrPeriod;
  numSamples;
  crc;
  publicationVersion;
  identifierLength;
  extraHeadersLength;
  identifier;
  extraHeaders;
  dataLength;
  constructor() {
    this.recordIndicator = "MS";
    this.formatVersion = 3;
    this.flags = 0;
    this.nanosecond = 0;
    this.year = 1970;
    this.dayOfYear = 1;
    this.hour = 0;
    this.minute = 0;
    this.second = 0;
    this.encoding = 3;
    this.sampleRateOrPeriod = 1;
    this.numSamples = 0;
    this.crc = 0;
    this.publicationVersion = UNKNOWN_DATA_VERSION;
    this.identifierLength = 0;
    this.extraHeadersLength = 2;
    this.identifier = "";
    this.extraHeaders = {};
    this.dataLength = 0;
  }
  /**
   * Parses an miniseed3 fixed header from a DataView.
   *
   * @param   dataView bytes to parse
   * @returns parsed header object
   */
  static createFromDataView(dataView) {
    const header = new _MSeed3Header();
    header.recordIndicator = makeString2(dataView, 0, 2);
    if (header.recordIndicator !== "MS") {
      throw new Error(
        "First 2 bytes of record should be MS but found " + header.recordIndicator
      );
    }
    header.formatVersion = dataView.getUint8(2);
    if (header.formatVersion !== 3) {
      throw new Error("Format Version should be 3, " + header.formatVersion);
    }
    header.flags = dataView.getUint8(3);
    const headerLittleEndian = true;
    header.nanosecond = dataView.getInt32(4, headerLittleEndian);
    header.year = dataView.getInt16(8, headerLittleEndian);
    if (checkByteSwap2(header.year)) {
      throw new Error("Looks like wrong byte order, year=" + header.year);
    }
    header.dayOfYear = dataView.getInt16(10, headerLittleEndian);
    header.hour = dataView.getUint8(12);
    header.minute = dataView.getUint8(13);
    header.second = dataView.getUint8(14);
    header.encoding = dataView.getUint8(15);
    header.sampleRateOrPeriod = dataView.getFloat64(16, headerLittleEndian);
    header.numSamples = dataView.getUint32(24, headerLittleEndian);
    header.crc = dataView.getUint32(28, headerLittleEndian);
    header.publicationVersion = dataView.getUint8(32);
    header.identifierLength = dataView.getUint8(33);
    header.extraHeadersLength = dataView.getUint16(34, headerLittleEndian);
    header.dataLength = dataView.getUint32(36, headerLittleEndian);
    header.identifier = makeString2(dataView, 40, header.identifierLength);
    return header;
  }
  get start() {
    return this.startAsDateTime();
  }
  get end() {
    return this.timeOfSample(this.numSamples - 1);
  }
  get sampleRate() {
    if (this.sampleRateOrPeriod < 0) {
      return -1 / this.sampleRateOrPeriod;
    } else {
      return this.sampleRateOrPeriod;
    }
  }
  get samplePeriod() {
    if (this.sampleRateOrPeriod <= 0) {
      return -1 * this.sampleRateOrPeriod;
    } else {
      return 1 / this.sampleRateOrPeriod;
    }
  }
  /**
   * Calculates size of the fixed header including the variable
   * length identifier, but without the extra headers.
   *
   * @returns size in bytes of fixed header
   */
  getSize() {
    return FIXED_HEADER_SIZE + this.identifier.length;
  }
  encodingName() {
    let encode_name = "unknown";
    if (this.encoding === 0) {
      encode_name = "Text";
    } else if (this.encoding === 1) {
      encode_name = "16-bit integer";
    } else if (this.encoding === 3) {
      encode_name = "32-bit integer";
    } else if (this.encoding === 4) {
      encode_name = "32-bit float";
    } else if (this.encoding === 5) {
      encode_name = "64-bit float";
    } else if (this.encoding === 11) {
      encode_name = "STEIM-2 integer compression";
    } else if (this.encoding === 10) {
      encode_name = "STEIM-1 integer compression";
    } else if (this.encoding === 19) {
      encode_name = "STEIM-3 integer compression";
    } else if (this.encoding === 100) {
      encode_name = "Opaque data";
    }
    return encode_name;
  }
  /**
   * Text representation of the miniseed3 header. This is modeled after
   * the output of mseed3-text from the mseed3-utils package from IRIS.
   *
   * @returns textual repersentation
   */
  toString() {
    const encode_name = this.encodingName();
    let bitFlagStr = "";
    if (this.flags & 1) {
      bitFlagStr = `${bitFlagStr}
                         [Bit 0] Calibration signals present`;
    }
    if (this.flags & 2) {
      bitFlagStr = `${bitFlagStr}
                         [Bit 1] Time tag is questionable`;
    }
    if (this.flags & 4) {
      bitFlagStr = `${bitFlagStr}
                         [Bit 2] Clock locked`;
    }
    if (this.flags & 8) {
      bitFlagStr = `${bitFlagStr}
                         [Bit 3] Undefined bit set`;
    }
    if (this.flags & 16) {
      bitFlagStr = `${bitFlagStr}
                         [Bit 4] Undefined bit set`;
    }
    if (this.flags & 32) {
      bitFlagStr = `${bitFlagStr}
                         [Bit 5] Undefined bit set`;
    }
    if (this.flags & 64) {
      bitFlagStr = `${bitFlagStr}
                         [Bit 6] Undefined bit set`;
    }
    if (this.flags & 128) {
      bitFlagStr = `${bitFlagStr}
                         [Bit 7] Undefined bit set`;
    }
    return `${this.identifier}, version ${this.publicationVersion}, ${this.getSize() + this.dataLength + this.extraHeadersLength} bytes (format: ${this.formatVersion})
             start time: ${this.getStartFieldsAsISO()} (${padZeros(this.dayOfYear, 3)})
      number of samples: ${this.numSamples}
       sample rate (Hz): ${this.sampleRate}
                  flags: [${(this.flags >>> 0).toString(2).padStart(8, "0")}] 8 bits${bitFlagStr}
                    CRC: ${crcToHexString(this.crc)}
    extra header length: ${this.extraHeadersLength} bytes
    data payload length: ${this.dataLength} bytes
       payload encoding: ${encode_name} (val: ${this.encoding})`;
  }
  /**
   * Start time in the format output by mseed3-utils from IRIS. Format is
   * yyyy,ooo,HH:mm:ss.SSSSSS
   *
   * @returns start time
   */
  startFieldsInUtilFormat() {
    return `${this.year},${padZeros(this.dayOfYear, 3)},${padZeros(this.hour, 2)}:${padZeros(this.minute, 2)}:${padZeros(this.second, 2)}.${padZeros(Math.floor(this.nanosecond / 1e3), 6)}`;
  }
  /**
   * Converts start time header fields to ISO8601 time string. This will include
   * factional seconds to nanosecond precision.
   *
   * @param trimMicroNano trim to microsecond precision if nanos are 000
   * @returns iso start time
   */
  getStartFieldsAsISO(trimMicroNano = true) {
    const d = this.startAsDateTime().set({ millisecond: 0 }).toISO({ includeOffset: false, suppressMilliseconds: true });
    let fracSec = "";
    if (trimMicroNano && this.nanosecond % 1e3 === 0) {
      fracSec = padZeros(this.nanosecond / 1e3, 6);
    } else {
      fracSec = padZeros(this.nanosecond, 9);
    }
    return `${d}.${fracSec}Z`;
  }
  /**
   * sets start time headers.
   *
   * @param starttime start as DateTime
   */
  setStart(starttime) {
    this.nanosecond = starttime.millisecond * 1e3;
    this.year = starttime.year;
    this.dayOfYear = starttime.ordinal;
    this.hour = starttime.hour;
    this.minute = starttime.minute;
    this.second = starttime.second;
  }
  /**
   * Calculates time of the ith sample.
   *
   * @param   i sample number
   * @returns the time
   */
  timeOfSample(i) {
    return this.start.plus(Duration6.fromMillis(1e3 * i / this.sampleRate));
  }
  /**
   * Writes to the given dataview.
   *
   * @param   dataView write buffer
   * @param   offset   offset within the buffer
   * @param   zeroCrc  optionally zero out the crc field in order to recalculate
   * @returns          new offset after this record
   */
  save(dataView, offset = 0, zeroCrc = false) {
    dataView.setUint8(offset, this.recordIndicator.charCodeAt(0));
    offset++;
    dataView.setUint8(offset, this.recordIndicator.charCodeAt(1));
    offset++;
    dataView.setUint8(offset, this.formatVersion);
    offset++;
    dataView.setUint8(offset, this.flags);
    offset++;
    dataView.setUint32(offset, this.nanosecond, true);
    offset += 4;
    dataView.setUint16(offset, this.year, true);
    offset += 2;
    dataView.setUint16(offset, this.dayOfYear, true);
    offset += 2;
    dataView.setUint8(offset, this.hour);
    offset++;
    dataView.setUint8(offset, this.minute);
    offset++;
    dataView.setUint8(offset, this.second);
    offset++;
    dataView.setUint8(offset, this.encoding);
    offset++;
    dataView.setFloat64(offset, this.sampleRateOrPeriod, true);
    offset += 8;
    dataView.setUint32(offset, this.numSamples, true);
    offset += 4;
    if (zeroCrc) {
      dataView.setUint32(offset, 0, true);
    } else {
      dataView.setUint32(offset, this.crc, true);
    }
    offset += 4;
    dataView.setUint8(offset, this.publicationVersion);
    offset++;
    dataView.setUint8(offset, this.identifier.length);
    offset++;
    dataView.setUint16(offset, this.extraHeadersLength, true);
    offset += 2;
    dataView.setUint32(offset, this.dataLength, true);
    offset += 4;
    for (let i = 0; i < this.identifier.length; i++) {
      dataView.setUint8(offset, this.identifier.charCodeAt(i));
      offset++;
    }
    return offset;
  }
  /**
   * Converts header start time to DateTime
   *
   * @returns         start time as DateTime
   */
  startAsDateTime() {
    return DateTime7.fromObject(
      {
        year: this.year,
        ordinal: this.dayOfYear,
        hour: this.hour,
        minute: this.minute,
        second: this.second,
        millisecond: Math.round(this.nanosecond / 1e6)
      },
      UTC_OPTIONS
    );
  }
};
function parseExtraHeaders(dataView) {
  if (dataView.byteLength === 0) {
    return {};
  }
  const firstChar = dataView.getUint8(0);
  if (firstChar === 123) {
    const jsonStr = makeString2(dataView, 0, dataView.byteLength);
    const v = JSON.parse(jsonStr);
    if (typeof v === "object") {
      return v;
    } else {
      throw new Error(
        `extra headers does not look like JSON object: ${jsonStr}"`
      );
    }
  } else {
    throw new Error(
      "do not understand extras with first char val: " + firstChar + " " + (firstChar === 123)
    );
  }
}
function padZeros(val, len) {
  let out = "" + val;
  while (out.length < len) {
    out = "0" + out;
  }
  return out;
}
function makeString2(dataView, offset, length) {
  const utf8decoder = new TextDecoder("utf-8");
  const u8arr = new Uint8Array(
    dataView.buffer,
    dataView.byteOffset + offset,
    length
  );
  return utf8decoder.decode(u8arr).trim();
}
function checkByteSwap2(year) {
  return year < 1960 || year > 2055;
}
function areContiguous2(dr1, dr2, sampRatio = 1.5) {
  const h1 = dr1.header;
  const h2 = dr2.header;
  return h1.end < h2.start && h1.end.plus(Duration6.fromMillis(1e3 * sampRatio / h1.sampleRate)) >= h2.start;
}
function createSeismogramSegment2(contig) {
  const contigData = contig.map((dr) => dr.asEncodedDataSegment());
  const out = new SeismogramSegment(
    contigData,
    contig[0].header.sampleRate,
    contig[0].header.start,
    FDSNSourceId.parse(contig[0].header.identifier)
  );
  const bag = extractBagEH(contig[0].extraHeaders);
  if (bag?.y?.si) {
    out.yUnit = bag?.y?.si;
  } else {
    console.log(`no yunit in seis ${contig[0].header.identifier}`);
  }
  return out;
}
function merge2(drList) {
  return new Seismogram(mergeSegments2(drList));
}
function mergeSegments2(drList) {
  const out = [];
  let currDR;
  drList.sort(function(a, b) {
    return a.header.start.valueOf() - b.header.start.valueOf();
  });
  let contig = [];
  for (let i = 0; i < drList.length; i++) {
    currDR = drList[i];
    if (contig.length === 0) {
      contig.push(currDR);
    } else if (areContiguous2(contig[contig.length - 1], currDR)) {
      contig.push(currDR);
    } else {
      out.push(createSeismogramSegment2(contig));
      contig = [currDR];
    }
  }
  if (contig.length > 0) {
    out.push(createSeismogramSegment2(contig));
    contig = [];
  }
  return out;
}
function byChannel2(drList) {
  const out = /* @__PURE__ */ new Map();
  let key;
  for (let i = 0; i < drList.length; i++) {
    const currDR = drList[i];
    key = currDR.codes();
    let drArray = out.get(key);
    if (!drArray) {
      drArray = [currDR];
      out.set(key, drArray);
    } else {
      drArray.push(currDR);
    }
  }
  return out;
}
function seismogramSegmentPerChannel2(drList) {
  let out = new Array(0);
  const byChannelMap = byChannel2(drList);
  byChannelMap.forEach(
    (segments) => out = out.concat(mergeSegments2(segments))
  );
  return out;
}
function seismogramPerChannel2(drList) {
  const out = [];
  const byChannelMap = byChannel2(drList);
  byChannelMap.forEach((segments) => out.push(merge2(segments)));
  return out;
}
function sddPerChannel(drList) {
  const out = [];
  const byChannelMap = byChannel2(drList);
  byChannelMap.forEach((segments) => {
    const sdd = SeismogramDisplayData.fromSeismogram(merge2(segments));
    out.push(sdd);
    segments.forEach((seg) => {
      const q = ehToQuake(seg.extraHeaders);
      if (q != null) {
        sdd.addQuake(q);
      }
      const marks = ehToMarkers(seg.extraHeaders);
      marks.forEach((mark) => sdd.addMarker(mark));
    });
  });
  return out;
}
function convertMS2toMSeed3(mseed2) {
  const out = [];
  for (let i = 0; i < mseed2.length; i++) {
    out.push(convertMS2Record(mseed2[i]));
  }
  return out;
}
function convertMS2Record(ms2record) {
  const xHeader = new MSeed3Header();
  const xExtras = {};
  const ms2H = ms2record.header;
  xHeader.flags = (ms2H.activityFlags & 1) * 2 + (ms2H.ioClockFlags & 64) * 4 + (ms2H.dataQualityFlags & 16) * 8;
  xHeader.year = ms2H.startBTime.year;
  xHeader.dayOfYear = ms2H.startBTime.jday;
  xHeader.hour = ms2H.startBTime.hour;
  xHeader.minute = ms2H.startBTime.min;
  xHeader.second = ms2H.startBTime.sec;
  xHeader.nanosecond = ms2H.startBTime.tenthMilli * 1e5 + ms2H.startBTime.microsecond * 1e3;
  xHeader.sampleRateOrPeriod = ms2H.sampleRate >= 1 ? ms2H.sampleRate : -1 / ms2H.sampleRate;
  xHeader.encoding = ms2record.header.encoding;
  xHeader.publicationVersion = UNKNOWN_DATA_VERSION;
  xHeader.dataLength = ms2record.data.byteLength;
  xHeader.identifier = FDSN_PREFIX2 + ":" + ms2H.netCode + SEP2 + ms2H.staCode + SEP2 + (ms2H.locCode ? ms2H.locCode : "") + SEP2 + ms2H.chanCode;
  xHeader.identifierLength = xHeader.identifier.length;
  xHeader.numSamples = ms2H.numSamples;
  xHeader.crc = 0;
  if (ms2H.typeCode) {
    if (ms2H.typeCode === R_TYPECODE) {
      xHeader.publicationVersion = 1;
    } else if (ms2H.typeCode === D_TYPECODE) {
      xHeader.publicationVersion = 2;
    } else if (ms2H.typeCode === Q_TYPECODE) {
      xHeader.publicationVersion = 3;
    } else if (ms2H.typeCode === M_TYPECODE) {
      xHeader.publicationVersion = 4;
    }
    if (ms2H.typeCode !== D_TYPECODE) {
      xExtras.DataQuality = ms2H.typeCode;
    }
  }
  if (xHeader.nanosecond < 0) {
    xHeader.second -= 1;
    xHeader.nanosecond += 1e9;
    if (xHeader.second < 0) {
      xHeader.second += 60;
      xHeader.minute -= 1;
      if (xHeader.minute < 0) {
        xHeader.minute += 60;
        xHeader.hour -= 1;
        if (xHeader.hour < 0) {
          xHeader.hour += 24;
          xHeader.dayOfYear = -1;
          if (xHeader.dayOfYear < 0) {
            xHeader.dayOfYear += 365;
            xHeader.year -= 1;
          }
        }
      }
    }
  }
  xHeader.extraHeadersLength = JSON.stringify(xExtras).length;
  const out = new MSeed3Record(xHeader, xExtras, ms2record.data);
  return out;
}
var SEP2 = "_";
var kCRCTable = new Int32Array([
  0,
  4067132163,
  3778769143,
  324072436,
  3348797215,
  904991772,
  648144872,
  3570033899,
  2329499855,
  2024987596,
  1809983544,
  2575936315,
  1296289744,
  3207089363,
  2893594407,
  1578318884,
  274646895,
  3795141740,
  4049975192,
  51262619,
  3619967088,
  632279923,
  922689671,
  3298075524,
  2592579488,
  1760304291,
  2075979607,
  2312596564,
  1562183871,
  2943781820,
  3156637768,
  1313733451,
  549293790,
  3537243613,
  3246849577,
  871202090,
  3878099393,
  357341890,
  102525238,
  4101499445,
  2858735121,
  1477399826,
  1264559846,
  3107202533,
  1845379342,
  2677391885,
  2361733625,
  2125378298,
  820201905,
  3263744690,
  3520608582,
  598981189,
  4151959214,
  85089709,
  373468761,
  3827903834,
  3124367742,
  1213305469,
  1526817161,
  2842354314,
  2107672161,
  2412447074,
  2627466902,
  1861252501,
  1098587580,
  3004210879,
  2688576843,
  1378610760,
  2262928035,
  1955203488,
  1742404180,
  2511436119,
  3416409459,
  969524848,
  714683780,
  3639785095,
  205050476,
  4266873199,
  3976438427,
  526918040,
  1361435347,
  2739821008,
  2954799652,
  1114974503,
  2529119692,
  1691668175,
  2005155131,
  2247081528,
  3690758684,
  697762079,
  986182379,
  3366744552,
  476452099,
  3993867776,
  4250756596,
  255256311,
  1640403810,
  2477592673,
  2164122517,
  1922457750,
  2791048317,
  1412925310,
  1197962378,
  3037525897,
  3944729517,
  427051182,
  170179418,
  4165941337,
  746937522,
  3740196785,
  3451792453,
  1070968646,
  1905808397,
  2213795598,
  2426610938,
  1657317369,
  3053634322,
  1147748369,
  1463399397,
  2773627110,
  4215344322,
  153784257,
  444234805,
  3893493558,
  1021025245,
  3467647198,
  3722505002,
  797665321,
  2197175160,
  1889384571,
  1674398607,
  2443626636,
  1164749927,
  3070701412,
  2757221520,
  1446797203,
  137323447,
  4198817972,
  3910406976,
  461344835,
  3484808360,
  1037989803,
  781091935,
  3705997148,
  2460548119,
  1623424788,
  1939049696,
  2180517859,
  1429367560,
  2807687179,
  3020495871,
  1180866812,
  410100952,
  3927582683,
  4182430767,
  186734380,
  3756733383,
  763408580,
  1053836080,
  3434856499,
  2722870694,
  1344288421,
  1131464017,
  2971354706,
  1708204729,
  2545590714,
  2229949006,
  1988219213,
  680717673,
  3673779818,
  3383336350,
  1002577565,
  4010310262,
  493091189,
  238226049,
  4233660802,
  2987750089,
  1082061258,
  1395524158,
  2705686845,
  1972364758,
  2279892693,
  2494862625,
  1725896226,
  952904198,
  3399985413,
  3656866545,
  731699698,
  4283874585,
  222117402,
  510512622,
  3959836397,
  3280807620,
  837199303,
  582374963,
  3504198960,
  68661723,
  4135334616,
  3844915500,
  390545967,
  1230274059,
  3141532936,
  2825850620,
  1510247935,
  2395924756,
  2091215383,
  1878366691,
  2644384480,
  3553878443,
  565732008,
  854102364,
  3229815391,
  340358836,
  3861050807,
  4117890627,
  119113024,
  1493875044,
  2875275879,
  3090270611,
  1247431312,
  2660249211,
  1828433272,
  2141937292,
  2378227087,
  3811616794,
  291187481,
  34330861,
  4032846830,
  615137029,
  3603020806,
  3314634738,
  939183345,
  1776939221,
  2609017814,
  2295496738,
  2058945313,
  2926798794,
  1545135305,
  1330124605,
  3173225534,
  4084100981,
  17165430,
  307568514,
  3762199681,
  888469610,
  3332340585,
  3587147933,
  665062302,
  2042050490,
  2346497209,
  2559330125,
  1793573966,
  3190661285,
  1279665062,
  1595330642,
  2910671697
]);
function calculateCRC32C(buf, initial = 0) {
  let ubuf;
  if (buf instanceof ArrayBuffer) {
    ubuf = new Uint8Array(buf);
  } else if (buf instanceof Uint8Array) {
    ubuf = buf;
  } else {
    throw new Error("arg must be ArrayBuffer or Uint8Array");
  }
  let crc = (initial | 0) ^ -1;
  for (let i = 0; i < ubuf.length; i++) {
    crc = kCRCTable[(crc ^ ubuf[i]) & 255] ^ crc >>> 8;
    let tmp = crc;
    tmp = (tmp ^ -1) >>> 0;
    if (tmp < 0) {
      tmp = 4294967295 + tmp + 1;
    }
  }
  return (crc ^ -1) >>> 0;
}
function crcToHexString(crc) {
  if (crc < 0) {
    crc = 4294967295 + crc + 1;
  }
  const s = crc.toString(16).toUpperCase();
  return "0x" + s;
}

// src/datalink.ts
import { DateTime as DateTime8 } from "luxon";
var DATALINK_PROTOCOL = "DataLink1.0";
var MODE = /* @__PURE__ */ ((MODE2) => {
  MODE2["Query"] = "QUERY";
  MODE2["Stream"] = "STREAM";
  return MODE2;
})(MODE || {});
var QUERY_MODE = "QUERY" /* Query */;
var STREAM_MODE = "STREAM" /* Stream */;
var MAX_PROC_NUM = Math.pow(2, 16) - 2;
var USER_BROWSER = "browser";
var DEFAULT_PROGRAM = "seisplotjs";
var DEFAULT_ARCH = "javascript";
var ERROR = "ERROR";
var OK = "OK";
var INFO = "INFO";
var ID = "ID";
var PACKET = "PACKET";
var STREAM = "STREAM";
var ENDSTREAM = "ENDSTREAM";
var MSEED_TYPE = "/MSEED";
var MSEED3_TYPE = "/MSEED3";
var IRIS_RINGSERVER_URL = "ws://rtserve.iris.washington.edu/datalink";
var defaultHandleResponse = function(dlResponse) {
  log(`Unhandled datalink response: ${dlResponse.toString()}`);
};
var DataLinkConnection = class _DataLinkConnection {
  url;
  /** @private */
  _mode;
  packetHandler;
  errorHandler;
  closeHandler;
  serverId;
  clientIdNum;
  programname;
  username;
  architecture;
  /** @private */
  _responseResolve;
  /** @private */
  _responseReject;
  webSocket;
  constructor(url, packetHandler, errorHandler) {
    this.webSocket = null;
    this.url = url ? url : IRIS_RINGSERVER_URL;
    this._mode = "QUERY" /* Query */;
    this.packetHandler = packetHandler;
    this.errorHandler = errorHandler;
    this.closeHandler = null;
    this.serverId = null;
    this.clientIdNum = Math.floor(Math.random() * MAX_PROC_NUM) + 1;
    this.programname = DEFAULT_PROGRAM;
    this.username = USER_BROWSER;
    this.architecture = DEFAULT_ARCH;
    this._responseResolve = null;
    this._responseReject = null;
  }
  /**
   * Set a callback function called when the connection is closed.
   *
   * @param  closeHandler callback function
   */
  setOnClose(closeHandler) {
    this.closeHandler = closeHandler;
  }
  /**
   * creates the websocket connection and sends the client ID.
   *
   *  @returns a Promise that resolves to the server's ID.
   */
  connect() {
    if (this.webSocket) {
      this.webSocket.close();
      this.webSocket = null;
    }
    return new Promise((resolve, reject) => {
      if (this.webSocket) {
        this.webSocket.close();
      }
      const webSocket = new WebSocket(this.url, DATALINK_PROTOCOL);
      this.webSocket = webSocket;
      webSocket.binaryType = "arraybuffer";
      webSocket.onmessage = (event) => {
        this.handle(event);
      };
      webSocket.onerror = (event) => {
        this.handleError(new Error("" + stringify(event)));
        reject(event);
      };
      webSocket.onclose = (closeEvent) => {
        this.webSocket = null;
        this._mode = "QUERY" /* Query */;
        if (this.closeHandler) {
          this.closeHandler(closeEvent);
        }
      };
      webSocket.onopen = () => {
        resolve(this);
      };
    }).then((datalink) => {
      return datalink.sendId();
    }).then((idmsg) => {
      this.serverId = idmsg;
      return idmsg;
    });
  }
  /**
   * @returns true if the websocket is connected (non-null)
   */
  isConnected() {
    return this.webSocket !== null;
  }
  /**
   * @returns the current mode, QUERY_MODE or STREAM_MODE
   */
  get mode() {
    return this._mode;
  }
  /**
   * Switches to streaming mode to receive data packets from the ringserver.
   *
   * @returns promise to the response
   */
  stream() {
    this._mode = "STREAM" /* Stream */;
    return this.awaitDLCommand(STREAM, "").then(
      (dlResponse) => _DataLinkConnection.ensureDataLinkResponse(dlResponse)
    );
  }
  /**
   * Switches back to query mode to enable commands to be sent to the ringserver.
   */
  endStream() {
    if (this.webSocket === null || this._mode === null || this._mode === "QUERY" /* Query */) {
      return;
    }
    this._mode = "QUERY" /* Query */;
    this.sendDLCommand(ENDSTREAM, "");
  }
  /**
   * Closes the connection and the underlying websocket. No communication
   * is possible until connect() is called again.
   */
  close() {
    if (this.webSocket) {
      this.endStream();
      if (this.webSocket) {
        this.webSocket.close();
      }
      this.webSocket = null;
      this._mode = "QUERY" /* Query */;
    }
  }
  /**
   * Send a ID Command. Command is a string.
   *
   * @returns a Promise that resolves to the response from the ringserver.
   */
  sendId() {
    return this.id(
      this.programname,
      this.username,
      stringify(this.clientIdNum),
      this.architecture
    ).then(
      (dlResponse) => _DataLinkConnection.ensureDataLinkResponse(dlResponse)
    ).then((dlResponse) => {
      if (dlResponse.type === "ID") {
        this.serverId = "" + dlResponse.message;
        return this.serverId;
      } else {
        throw new Error("not ID response: " + stringify(dlResponse.type));
      }
    });
  }
  /**
   * encodes as a Datalink packet, header with optional data section as
   * binary Uint8Array. Size of the binary data is appended
   * to the header if present.
   *
   * @param header the command/header string
   * @param data optional data portion
   * @returns datalink packet as an ArrayBuffer
   */
  encodeDL(header, data) {
    let cmdLen = header.length;
    let len = 3 + header.length;
    let lenStr = "";
    if (data && data.length > 0) {
      lenStr = String(data.length);
      len += lenStr.length + 1;
      cmdLen += lenStr.length + 1;
      len += data.length;
    }
    const rawPacket = new ArrayBuffer(len);
    const binaryPacket = new Uint8Array(rawPacket);
    const packet = new DataView(rawPacket);
    packet.setUint8(0, 68);
    packet.setUint8(1, 76);
    packet.setUint8(2, cmdLen);
    let i = 3;
    for (const c of header) {
      packet.setUint8(i, c.charCodeAt(0));
      i++;
    }
    const SPACE = " ";
    if (data && data.length > 0) {
      packet.setUint8(i, SPACE.charCodeAt(0));
      i++;
      for (const c of lenStr) {
        packet.setUint8(i, c.charCodeAt(0));
        i++;
      }
      binaryPacket.set(data, i);
    }
    return rawPacket;
  }
  /**
   * sends the header with optional binary data
   * as the data section. Size of the data is appended
   * to the header before sending if present.
   *
   * @param header header to send
   * @param data optional data to send
   */
  sendDLBinary(header, data) {
    const rawPacket = this.encodeDL(header, data);
    if (this.webSocket) {
      this.webSocket.send(rawPacket);
    } else {
      throw new Error("WebSocket has been closed.");
    }
  }
  /**
   * sends the command as header with optional dataString
   * as the data section. Size of the dataString is appended
   * to the header before sending.
   *
   * @param command the command/header string
   * @param dataString optional data portion of packet
   */
  sendDLCommand(command, dataString) {
    this.sendDLBinary(command, stringToUint8Array(dataString));
  }
  /**
   * Send a DataLink Command and await the response. Command is a string.
   *
   * @param header packet header
   * @param data optional data portion of packet
   * @returns a Promise that resolves with the webSocket MessageEvent.
   */
  awaitDLBinary(header, data) {
    const promise = new Promise(
      (resolve, reject) => {
        this._responseResolve = resolve;
        this._responseReject = reject;
        this.sendDLBinary(header, data);
      }
    ).then((response) => {
      this._responseResolve = null;
      this._responseReject = null;
      return response;
    }).catch((error) => {
      this._responseResolve = null;
      this._responseReject = null;
      throw error;
    });
    return promise;
  }
  /**
   * Send a DataLink Command and await the response. Command is a string.
   * Returns a Promise that resolves with the webSocket MessageEvent.
   *
   * @param command the command/header string
   * @param dataString optional data portion of packet
   * @returns promise to server's response
   */
  awaitDLCommand(command, dataString) {
    return this.awaitDLBinary(command, stringToUint8Array(dataString));
  }
  /**
   * Writes data to the ringserver and awaits a acknowledgement.
   *
   * @param   streamid    stream id for packet header
   * @param   hpdatastart start of timewindow the packet covers
   * @param   hpdataend   end of timewindow the packet covers
   * @param   data        optional data to send
   * @returns             promise to server's response
   */
  writeAck(streamid, hpdatastart, hpdataend, data) {
    const header = `WRITE ${streamid} ${dateTimeToHPTime(
      hpdatastart
    )} ${dateTimeToHPTime(hpdataend)} A`;
    return this.awaitDLBinary(header, data);
  }
  /**
   * Makes sure a response actually is a DataLinkResponse
   *
   * @param   dl datalink packet/response
   * @returns DataLinkResponse after checking instanceof
   * @throws Error if not a DataLinkResponse
   */
  static ensureDataLinkResponse(dl) {
    if (dl instanceof DataLinkResponse) {
      return dl;
    }
    throw new Error(`Expected DataLinkResponse but got ${dl.header}`);
  }
  /**
   * Makes sure a response actually is a DataLinkPacket
   *
   * @param   dl datalink packet/response
   * @returns DataLinkPacket after checking instanceof
   * @throws Error if not a DataLinkPacket
   */
  static ensureDataLinkPacket(dl) {
    if (dl instanceof DataLinkPacket) {
      return dl;
    }
    throw new Error(`Expected DataLinkPacket but got ${dl.type}`);
  }
  /**
   * Send id and await server's response. All of these are can more or less
   * be filled with dummy values. They are mostly used for logging and debugging
   * on the server side.
   *
   * @param programname name of program, ex seisplotjs
   * @param username name of user, ex browser
   * @param processid process number, used to differentiate between multiple running instances
   * @param architecture cpu architecture, ex javascript
   * @returns promise to servers response
   */
  id(programname, username, processid, architecture) {
    const command = `ID ${programname}:${username}:${processid}:${architecture}`;
    return this.awaitDLCommand(command).then(
      (dlResponse) => _DataLinkConnection.ensureDataLinkResponse(dlResponse)
    );
  }
  /**
   * Send info command for infoType.
   *
   * @param infoType type to get info for
   * @returns promise to server's response
   */
  info(infoType) {
    const command = `INFO ${infoType}`;
    return this.awaitDLCommand(command).then(
      (dlResponse) => _DataLinkConnection.ensureDataLinkResponse(dlResponse)
    );
  }
  infoStatus() {
    return this.info("STATUS").then((daResp) => {
      return StatusResponse.fromDatalinkResponse(daResp);
    });
  }
  infoStreams() {
    return this.info("STREAMS").then((daResp) => {
      return StreamsResponse.fromDatalinkResponse(daResp);
    });
  }
  infoConnections() {
    return this.info("CONNECTIONS").then((daResp) => {
      return ConnectionsResponse.fromDatalinkResponse(daResp);
    });
  }
  /**
   * Send position after command.
   *
   * @param time time to position after
   * @returns promise to server's response
   */
  positionAfter(time) {
    return this.positionAfterHPTime(dateTimeToHPTime(time)).then(
      (dlResponse) => _DataLinkConnection.ensureDataLinkResponse(dlResponse)
    );
  }
  /**
   * Send position after command.
   *
   * @param hpTime time to position after
   * @returns promise to server's response
   */
  positionAfterHPTime(hpTime) {
    const command = `POSITION AFTER ${hpTime}`;
    return this.awaitDLCommand(command).then(
      (dlResponse) => _DataLinkConnection.ensureDataLinkResponse(dlResponse)
    );
  }
  /**
   * Send match command.
   *
   * @param pattern regular expression to match streams
   * @returns promise to server's response
   */
  match(pattern) {
    const command = `MATCH`;
    return this.awaitDLCommand(command, pattern).then(
      (dlResponse) => _DataLinkConnection.ensureDataLinkResponse(dlResponse)
    );
  }
  /**
   * Send reject command.
   *
   * @param pattern regular expression to reject streams
   * @returns promise to server's response
   */
  reject(pattern) {
    const command = `REJECT ${pattern}`;
    return this.awaitDLCommand(command).then(
      (dlResponse) => _DataLinkConnection.ensureDataLinkResponse(dlResponse)
    );
  }
  /**
   * Read a single packet for the given id.
   *
   * @param packetId id of the packet of interest
   * @returns promise to server's response
   */
  read(packetId) {
    const command = `READ ${packetId}`;
    return this.awaitDLBinary(command).then(
      (dlResponse) => _DataLinkConnection.ensureDataLinkPacket(dlResponse)
    );
  }
  /**
   * Handles a web socket message from the data link connection.
   *
   * @private
   * @param wsEvent web socket event to handle
   */
  handle(wsEvent) {
    const rawData = wsEvent.data;
    if (rawData instanceof ArrayBuffer) {
      this.handleArrayBuffer(rawData);
    }
  }
  handleArrayBuffer(rawData) {
    const dlPreHeader = new DataView(rawData, 0, 3);
    if ("D" === String.fromCharCode(dlPreHeader.getUint8(0)) && "L" === String.fromCharCode(dlPreHeader.getUint8(1))) {
      const headerLen = dlPreHeader.getUint8(2);
      const header = dataViewToString(new DataView(rawData, 3, headerLen));
      if (header.startsWith(PACKET)) {
        const packet = new DataLinkPacket(
          header,
          new DataView(rawData, 3 + headerLen)
        );
        if (this.packetHandler) {
          try {
            this.packetHandler(packet);
          } catch (e) {
            this.handleError(toError(e));
          }
        } else {
          this.handleError(new Error("packetHandler not defined"));
        }
      } else {
        let dv;
        if (rawData.byteLength > 3 + headerLen) {
          dv = new DataView(rawData, 3 + headerLen);
        }
        const dlResponse = DataLinkResponse.parse(header, dv);
        if (dlResponse.type === "ENDSTREAM") {
          this._mode = "QUERY" /* Query */;
        } else {
          if (this._responseResolve) {
            this._responseResolve(dlResponse);
          } else {
            defaultHandleResponse(dlResponse);
          }
        }
      }
    } else {
      throw new Error("DataLink Packet did not start with DL");
    }
  }
  /**
   * handle errors that arise
   *
   * @private
   * @param   error the error
   */
  handleError(error) {
    if (this._responseReject) {
      this._responseReject(error);
    }
    if (this.errorHandler) {
      this.errorHandler(error);
    } else {
      log("datalink handleError: " + error.message);
    }
  }
};
var DataLinkResponse = class _DataLinkResponse {
  type;
  value;
  message;
  constructor(type, value, message) {
    this.type = type;
    this.value = value;
    this.message = message;
  }
  isError() {
    return this.type === ERROR;
  }
  toString() {
    return `${this.type} ${this.value} | ${this.message}`;
  }
  static parse(header, data) {
    let value = "";
    const s = header.split(" ");
    const type = s[0];
    let message = "";
    if (type === ID) {
      message = "" + header.substring(3);
    } else if (type === ENDSTREAM || type === INFO || type === OK || type === ERROR) {
      value = s[1];
      if (data) {
        message = dataViewToString(
          new DataView(data.buffer, 3 + header.length)
        );
      }
    } else {
      log(`unknown DataLink response type: ${type}  ${header}`);
      message = header.substring(type.length + 1);
    }
    return new _DataLinkResponse(type, value, message);
  }
};
var DataLinkPacket = class {
  header;
  data;
  streamId;
  pktid;
  hppackettime;
  hppacketstart;
  hppacketend;
  dataSize;
  _miniseed;
  _mseed3;
  constructor(header, dataview) {
    this._miniseed = null;
    this._mseed3 = null;
    this.header = header;
    this.data = dataview;
    const split = this.header.split(" ");
    this.streamId = split[1];
    this.pktid = split[2];
    this.hppackettime = split[3];
    this.hppacketstart = split[4];
    this.hppacketend = split[5];
    this.dataSize = Number.parseInt(split[6]);
    if (dataview.byteLength < this.dataSize) {
      throw new Error(
        `not enough bytes in dataview for packet:  ${this.dataSize}`
      );
    }
  }
  /**
   * Packet start time as a DateTime.
   *
   * @returns start time
   */
  get packetStart() {
    return hpTimeToDateTime(parseInt(this.hppacketstart));
  }
  /**
   * Packet end time as a DateTime.
   *
   * @returns end time
   */
  get packetEnd() {
    return hpTimeToDateTime(parseInt(this.hppacketend));
  }
  /**
   * Packet time as a DateTime.
   *
   * @returns packet time
   */
  get packetTime() {
    return hpTimeToDateTime(parseInt(this.hppackettime));
  }
  /**
   * is this packet a miniseed packet
   *
   * @returns          true if it is miniseed
   */
  isMiniseed() {
    return isDef(this._miniseed) || this.streamId.endsWith(MSEED_TYPE);
  }
  /**
   * Parsed payload as a miniseed data record, if the streamid
   * ends with '/MSEED', null otherwise.
   *
   * @returns miniseed DataRecord or null
   */
  asMiniseed() {
    if (!isDef(this._miniseed)) {
      if (this.streamId.endsWith(MSEED_TYPE)) {
        this._miniseed = parseSingleDataRecord(this.data);
      } else {
        this._miniseed = null;
      }
    }
    return this._miniseed;
  }
  /**
   * is this packet a miniseed3 packet
   *
   * @returns          true if it is miniseed3
   */
  isMiniseed3() {
    return isDef(this._mseed3) || this.streamId.endsWith(MSEED3_TYPE);
  }
  /**
   * Parsed payload as a miniseed3 data record, if the data format is 3, null otherwise.
   *
   * @returns miniseed3 DataRecord or null
   */
  asMiniseed3() {
    if (!isDef(this._mseed3)) {
      if (this.streamId.endsWith(MSEED3_TYPE)) {
        this._mseed3 = MSeed3Record.parseSingleDataRecord(this.data);
      } else if (this.streamId.endsWith(MSEED_TYPE)) {
        const ms2 = this.asMiniseed();
        if (ms2) {
          this._mseed3 = convertMS2Record(ms2);
        }
      } else {
        this._mseed3 = null;
      }
    }
    return this._mseed3;
  }
};
var DataLinkIdStats = class _DataLinkIdStats {
  version;
  serverId;
  capabilities;
  constructor(version2, serverId, capabilities) {
    this.version = version2;
    this.serverId = serverId;
    this.capabilities = capabilities;
  }
  /**
   * Parses the attributes of a <DataLink> xml element.
   *
   * @param  statusEl               DataLink XML element
   * @returns  the id stats
   */
  static parseXMLAttributes(statusEl) {
    const dlIdStats = new _DataLinkIdStats(
      parseUtil._requireAttribute(statusEl, "Version"),
      parseUtil._requireAttribute(statusEl, "ServerID"),
      parseUtil._requireAttribute(statusEl, "Capabilities").split(" ")
    );
    return dlIdStats;
  }
  toString() {
    return `
DataLink:
Version="${this.version}"
Id="${this.serverId}"
Capabilities="${this.capabilities.join(" ")}"`;
  }
};
var DataLinkStats = class _DataLinkStats {
  startTime;
  ringVersion;
  ringSize;
  packetSize;
  maximumPacketID;
  maximumPackets;
  memoryMappedRing;
  volatileRing;
  totalConnections;
  totalStreams;
  txPacketRate;
  txByteRate;
  rxPacketRate;
  rxByteRate;
  earliestPacketID;
  earliestPacketCreationTime;
  earliestPacketDataStartTime;
  earliestPacketDataEndTime;
  latestPacketID;
  latestPacketCreationTime;
  latestPacketDataStartTime;
  latestPacketDataEndTime;
  constructor(startTime, ringVersion, ringSize, packetSize, maximumPacketID, maximumPackets, memoryMappedRing, volatileRing, totalConnections, totalStreams, txPacketRate, txByteRate, rxPacketRate, rxByteRate, earliestPacketID, earliestPacketCreationTime, earliestPacketDataStartTime, earliestPacketDataEndTime, latestPacketID, latestPacketCreationTime, latestPacketDataStartTime, latestPacketDataEndTime) {
    this.startTime = startTime;
    this.ringVersion = ringVersion;
    this.ringSize = ringSize;
    this.packetSize = packetSize;
    this.maximumPacketID = maximumPacketID;
    this.maximumPackets = maximumPackets;
    this.memoryMappedRing = memoryMappedRing;
    this.volatileRing = volatileRing;
    this.totalConnections = totalConnections;
    this.totalStreams = totalStreams;
    this.txPacketRate = txPacketRate;
    this.txByteRate = txByteRate;
    this.rxPacketRate = rxPacketRate;
    this.rxByteRate = rxByteRate;
    this.earliestPacketID = earliestPacketID;
    this.earliestPacketCreationTime = earliestPacketCreationTime;
    this.earliestPacketDataStartTime = earliestPacketDataStartTime;
    this.earliestPacketDataEndTime = earliestPacketDataEndTime;
    this.latestPacketID = latestPacketID;
    this.latestPacketCreationTime = latestPacketCreationTime;
    this.latestPacketDataStartTime = latestPacketDataStartTime;
    this.latestPacketDataEndTime = latestPacketDataEndTime;
  }
  /**
   * Parses the attributes of a <Status> xml element.
   *
   * @param  statusEl   DataLink <Status> XML element
   * @returns  the stats
   */
  static parseXMLAttributes(statusEl) {
    const dlStats = new _DataLinkStats(
      daliDateTime(parseUtil._requireAttribute(statusEl, "StartTime")),
      parseUtil._requireAttribute(statusEl, "RingVersion"),
      parseInt(parseUtil._requireAttribute(statusEl, "RingSize")),
      parseInt(parseUtil._requireAttribute(statusEl, "PacketSize")),
      parseInt(parseUtil._requireAttribute(statusEl, "MaximumPacketID")),
      parseInt(parseUtil._requireAttribute(statusEl, "MaximumPackets")),
      parseUtil._requireAttribute(statusEl, "MemoryMappedRing") === "TRUE",
      parseUtil._requireAttribute(statusEl, "VolatileRing") === "TRUE",
      parseInt(parseUtil._requireAttribute(statusEl, "TotalConnections")),
      parseInt(parseUtil._requireAttribute(statusEl, "TotalStreams")),
      parseFloat(parseUtil._requireAttribute(statusEl, "TXPacketRate")),
      parseFloat(parseUtil._requireAttribute(statusEl, "TXByteRate")),
      parseFloat(parseUtil._requireAttribute(statusEl, "RXPacketRate")),
      parseFloat(parseUtil._requireAttribute(statusEl, "RXByteRate")),
      parseInt(parseUtil._requireAttribute(statusEl, "EarliestPacketID")),
      daliDateTime(
        parseUtil._requireAttribute(statusEl, "EarliestPacketCreationTime")
      ),
      daliDateTime(
        parseUtil._requireAttribute(statusEl, "EarliestPacketDataStartTime")
      ),
      daliDateTime(
        parseUtil._requireAttribute(statusEl, "EarliestPacketDataEndTime")
      ),
      parseInt(parseUtil._requireAttribute(statusEl, "LatestPacketID")),
      daliDateTime(
        parseUtil._requireAttribute(statusEl, "LatestPacketCreationTime")
      ),
      daliDateTime(
        parseUtil._requireAttribute(statusEl, "LatestPacketDataStartTime")
      ),
      daliDateTime(
        parseUtil._requireAttribute(statusEl, "LatestPacketDataEndTime")
      )
    );
    return dlStats;
  }
  toString() {
    return `
Status:
StartTime="${this.startTime.toISO()}"
RingVersion="${this.ringVersion}"
RingSize="${this.ringSize}"
PacketSize="${this.packetSize}"
MaximumPacketID="${this.maximumPacketID}"
MaximumPackets="${this.maximumPackets}"
MemoryMappedRing="${this.memoryMappedRing}"
VolatileRing="${this.volatileRing}"
TotalConnections="${this.totalConnections}"
TotalStreams="${this.totalStreams}"
TXPacketRate="${this.txPacketRate}"
TXByteRate="${this.txByteRate}"
RXPacketRate="${this.rxPacketRate}"
RXByteRate="${this.rxByteRate}"
EarliestPacketID="${this.earliestPacketID}"
EarliestPacketCreationTime="${this.earliestPacketCreationTime.toISO()}"
EarliestPacketDataStartTime="${this.earliestPacketDataStartTime.toISO()}"
EarliestPacketDataEndTime="${this.earliestPacketDataEndTime.toISO()}"
LatestPacketID="${this.latestPacketID}"
LatestPacketCreationTime="${this.latestPacketCreationTime.toISO()}"
LatestPacketDataStartTime="${this.latestPacketDataStartTime.toISO()}"
LatestPacketDataEndTime="${this.latestPacketDataEndTime.toISO()}"
    `;
  }
};
var ThreadStat = class _ThreadStat {
  flags;
  type;
  port;
  constructor(flags, type, port) {
    this.flags = flags;
    this.type = type;
    this.port = port;
  }
  /**
   * Parses the attributes of a <Status> xml element.
   *
   * @param  statusEl   DataLink <Status> XML element
   * @returns  the stats
   */
  static parseXMLAttributes(statusEl) {
    const threadStats = new _ThreadStat(
      parseUtil._requireAttribute(statusEl, "Flags").split(" "),
      parseUtil._requireAttribute(statusEl, "Type").split(" "),
      parseInt(parseUtil._requireAttribute(statusEl, "Port"))
    );
    return threadStats;
  }
  toString() {
    return `Thread  Port: ${this.port} Flags: ${this.flags.join(" ")} Type: ${this.type.join(" ")}`;
  }
};
var StatusResponse = class _StatusResponse {
  idStats;
  datalinkStats;
  threadStats;
  rawXml = "";
  constructor(idStats, datalinkStats, threadStats) {
    this.idStats = idStats;
    this.datalinkStats = datalinkStats;
    this.threadStats = threadStats;
  }
  static fromDatalinkResponse(daliResp) {
    if (daliResp.type === INFO) {
      const daliXml = new DOMParser().parseFromString(
        daliResp.message,
        "text/xml"
      );
      const sResp = _StatusResponse.fromXML(daliXml.documentElement);
      sResp.rawXml = daliResp.message;
      return sResp;
    } else {
      throw new Error("Datalink Response not OK", { cause: daliResp });
    }
  }
  static fromXML(daliXML) {
    const idStats = DataLinkIdStats.parseXMLAttributes(daliXML);
    const dlStats = DataLinkStats.parseXMLAttributes(
      daliXML.getElementsByTagName("Status")[0]
    );
    const threadListEl = daliXML.getElementsByTagName("ServerThreads")[0];
    let threads = [];
    if (threadListEl) {
      threads = Array.from(threadListEl.getElementsByTagName("Thread")).map(
        (threadEl) => ThreadStat.parseXMLAttributes(threadEl)
      );
    }
    return new _StatusResponse(idStats, dlStats, threads);
  }
  toString() {
    return `
${this.idStats.toString()}
${this.datalinkStats.toString()}
${this.threadStats.join("\n")}`;
  }
};
var StreamStat = class _StreamStat {
  name;
  earliestPacketID;
  earliestPacketDataStartTime;
  earliestPacketDataEndTime;
  latestPacketID;
  latestPacketDataStartTime;
  latestPacketDataEndTime;
  dataLatency;
  constructor(name, earliestPacketID, earliestPacketDataStartTime, earliestPacketDataEndTime, latestPacketID, latestPacketDataStartTime, latestPacketDataEndTime, dataLatency) {
    this.name = name;
    this.earliestPacketID = earliestPacketID;
    this.earliestPacketDataStartTime = earliestPacketDataStartTime;
    this.earliestPacketDataEndTime = earliestPacketDataEndTime;
    this.latestPacketID = latestPacketID;
    this.latestPacketDataStartTime = latestPacketDataStartTime;
    this.latestPacketDataEndTime = latestPacketDataEndTime;
    this.dataLatency = dataLatency;
  }
  static parseXMLAttributes(statusEl) {
    const sStat = new _StreamStat(
      parseUtil._requireAttribute(statusEl, "Name"),
      parseInt(parseUtil._requireAttribute(statusEl, "EarliestPacketID")),
      daliDateTime(
        parseUtil._requireAttribute(statusEl, "EarliestPacketDataStartTime")
      ),
      daliDateTime(
        parseUtil._requireAttribute(statusEl, "EarliestPacketDataEndTime")
      ),
      parseInt(parseUtil._requireAttribute(statusEl, "LatestPacketID")),
      daliDateTime(
        parseUtil._requireAttribute(statusEl, "LatestPacketDataStartTime")
      ),
      daliDateTime(
        parseUtil._requireAttribute(statusEl, "LatestPacketDataEndTime")
      ),
      parseFloat(parseUtil._requireAttribute(statusEl, "DataLatency"))
    );
    return sStat;
  }
  toString() {
    return `
    Name: ${this.name}
    EarliestPacketID="${this.earliestPacketID}"
    EarliestPacketDataStartTime="${this.earliestPacketDataStartTime.toISO()}"
    EarliestPacketDataEndTime="${this.earliestPacketDataEndTime.toISO()}"
    LatestPacketID="${this.latestPacketID}"
    LatestPacketDataStartTime="${this.latestPacketDataStartTime.toISO()}"
    LatestPacketDataEndTime="${this.latestPacketDataEndTime.toISO()}"
    DataLatency=${this.dataLatency}
    `;
  }
};
var StreamsResponse = class _StreamsResponse {
  datalinkStats;
  streams;
  constructor(datalinkStats, streams) {
    this.datalinkStats = datalinkStats;
    this.streams = streams;
  }
  static fromDatalinkResponse(daliResp) {
    if (daliResp.type === INFO) {
      const daliXml = new DOMParser().parseFromString(
        daliResp.message,
        "text/xml"
      );
      return _StreamsResponse.fromXML(daliXml.documentElement);
    } else {
      throw new Error("Datalink Response not OK", { cause: daliResp });
    }
  }
  /*
  <DataLink Version="2018.078"
  ServerID="South Carolina Seismic Network"
  Capabilities="DLPROTO:1.0 PACKETSIZE:512 WRITE">
  <Status
    StartTime="2022-09-21 12:13:29"
    RingVersion="1"
    RingSize="1073741824"
    PacketSize="512"
    MaximumPacketID="16777215"
    MaximumPackets="1698952"
    MemoryMappedRing="TRUE"
    VolatileRing="FALSE"
    TotalConnections="9"
    TotalStreams="90"
    TXPacketRate="57.0"
    TXByteRate="29167.9"
    RXPacketRate="18.0"
    RXByteRate="9210.9"
    EarliestPacketID="11763348"
    EarliestPacketCreationTime="2022-10-03 12:56:43.520738"
    EarliestPacketDataStartTime="2022-10-03 12:56:40.860000"
    EarliestPacketDataEndTime="2022-10-03 12:56:42.875000"
    LatestPacketID="13462299"
    LatestPacketCreationTime="2022-10-04 15:11:24.786990"
    LatestPacketDataStartTime="2022-10-04 15:11:19.580000"
    LatestPacketDataEndTime="2022-10-04 15:11:22.785000" />
    <StreamList TotalStreams="90" SelectedStreams="3">
      <Stream Name="CO_JSC_00_HHE/MSEED" EarliestPacketID="11763363" EarliestPacketDataStartTime="2022-10-03 12:56:37.178392" EarliestPacketDataEndTime="2022-10-03 12:56:40.808392" LatestPacketID="13462285" LatestPacketDataStartTime="2022-10-04 15:11:15.808392" LatestPacketDataEndTime="2022-10-04 15:11:18.788392" DataLatency="6.0" />
      <Stream Name="CO_JSC_00_HHN/MSEED" EarliestPacketID="11763389" EarliestPacketDataStartTime="2022-10-03 12:56:38.108392" EarliestPacketDataEndTime="2022-10-03 12:56:41.398392" LatestPacketID="13462284" LatestPacketDataStartTime="2022-10-04 15:11:15.668392" LatestPacketDataEndTime="2022-10-04 15:11:18.408392" DataLatency="6.4" />
      <Stream Name="CO_JSC_00_HHZ/MSEED" EarliestPacketID="11763402" EarliestPacketDataStartTime="2022-10-03 12:56:38.908392" EarliestPacketDataEndTime="2022-10-03 12:56:42.688392" LatestPacketID="13462252" LatestPacketDataStartTime="2022-10-04 15:11:14.428392" LatestPacketDataEndTime="2022-10-04 15:11:17.858392" DataLatency="6.9" />
    </StreamList>
  </DataLink>
     */
  static fromXML(daliXML) {
    const statusEl = daliXML.getElementsByTagName("Status")[0];
    const dlStats = DataLinkStats.parseXMLAttributes(statusEl);
    const streamListEl = daliXML.getElementsByTagName("StreamList")[0];
    const streamElList = streamListEl.getElementsByTagName("Stream");
    const streams = Array.from(streamElList).map(
      (streamEl) => StreamStat.parseXMLAttributes(streamEl)
    );
    const streamResp = new _StreamsResponse(dlStats, streams);
    return streamResp;
  }
  toString() {
    return `${this.datalinkStats.toString()}
    ${this.streams.map((s) => s.toString()).join("\n")}
    `;
  }
};
var ConnectionsResponse = class _ConnectionsResponse {
  daliXML;
  constructor(daliXML) {
    this.daliXML = daliXML;
  }
  static fromDatalinkResponse(daliResp) {
    if (daliResp.type === INFO) {
      return new _ConnectionsResponse(daliResp.message);
    } else {
      throw new Error("Datalink Response not OK", { cause: daliResp });
    }
  }
  static fromXML(daliXML) {
    const xmlString = new XMLSerializer().serializeToString(daliXML);
    return new _ConnectionsResponse(xmlString);
  }
  toString() {
    return `${this.daliXML.toString()}`;
  }
};
function daliDateTime(dalitime) {
  const iso = dalitime.replace(" ", "T");
  return isoToDateTime(iso);
}
function dateTimeToHPTime(m) {
  return m.valueOf() * 1e3;
}
function hpTimeToDateTime(hptime) {
  return DateTime8.fromMillis(hptime / 1e3, UTC_OPTIONS);
}
function stringToUint8Array(dataString) {
  let binaryData;
  if (isNonEmptyStringArg(dataString)) {
    binaryData = new Uint8Array(dataString.length);
    for (let i = 0; i < dataString.length; i++) {
      binaryData[i] = dataString.charCodeAt(i);
    }
  } else {
    binaryData = new Uint8Array(0);
  }
  return binaryData;
}

// src/dataset.ts
var dataset_exports = {};
__export(dataset_exports, {
  CATALOG_FILE: () => CATALOG_FILE,
  DATASET_DIR: () => DATASET_DIR,
  DOT_ZIP_EXT: () => DOT_ZIP_EXT,
  Dataset: () => Dataset,
  INVENTORY_FILE: () => INVENTORY_FILE,
  SEISMOGRAM_DIR: () => SEISMOGRAM_DIR,
  ZIP_FILENAME: () => ZIP_FILENAME,
  createExtraHeaders: () => createExtraHeaders,
  insertExtraHeaders: () => insertExtraHeaders,
  load: () => load,
  loadFromFile: () => loadFromFile,
  loadFromZip: () => loadFromZip,
  sddFromMSeed3: () => sddFromMSeed3
});
import { Duration as Duration8 } from "luxon";

// src/seismographmarker.ts
import { Duration as Duration7 } from "luxon";
function isValidMarker(v) {
  if (!v || typeof v !== "object") {
    return false;
  }
  const m = v;
  return typeof m.time === "string" && typeof m.name === "string" && typeof m.markertype === "string" && typeof m.description === "string" && (!("link" in m) || typeof m.link === "string");
}
function createMarkersForTravelTimes(quake, ttime) {
  return ttime.arrivals.map((a) => {
    return {
      markertype: "predicted",
      name: a.phase,
      time: quake.time.plus(Duration7.fromMillis(1e3 * a.time)),
      description: ""
    };
  });
}
function createMarkerForOriginTime(quake) {
  return {
    markertype: "predicted",
    name: "origin",
    time: quake.time,
    description: ""
  };
}

// src/traveltime.ts
var traveltime_exports = {};
__export(traveltime_exports, {
  FAKE_EMPTY_SVG: () => FAKE_EMPTY_SVG,
  FAKE_EMPTY_TEXT_HEADERS: () => FAKE_EMPTY_TEXT_HEADERS,
  FAKE_EMPTY_TEXT_MODEL: () => FAKE_EMPTY_TEXT_MODEL,
  IRIS_HOST: () => IRIS_HOST2,
  JSON_FORMAT: () => JSON_FORMAT,
  SVG_FORMAT: () => SVG_FORMAT,
  TEXT_FORMAT: () => TEXT_FORMAT,
  TraveltimeQuery: () => TraveltimeQuery,
  convertTravelTimeLineToObject: () => convertTravelTimeLineToObject,
  createEmptyTraveltimeJson: () => createEmptyTraveltimeJson,
  createOriginArrival: () => createOriginArrival,
  isValidTraveltimeArrivalType: () => isValidTraveltimeArrivalType,
  isValidTraveltimeJsonType: () => isValidTraveltimeJsonType
});

// src/fdsncommon.ts
var fdsncommon_exports = {};
__export(fdsncommon_exports, {
  FDSNCommon: () => FDSNCommon,
  IRIS_HOST: () => IRIS_HOST,
  LatLonBox: () => LatLonBox,
  LatLonRadius: () => LatLonRadius,
  LatLonRegion: () => LatLonRegion
});
var IRIS_HOST = "service.iris.edu";
var FDSNCommon = class {
  /** @private */
  _specVersion;
  /** @private */
  _protocol;
  /** @private */
  _host;
  /** @private */
  _port;
  /** @private */
  _nodata;
  /** @private */
  _timeoutSec;
  constructor(host) {
    this._specVersion = "1";
    this._host = IRIS_HOST;
    this._protocol = checkProtocol();
    if (isNonEmptyStringArg(host)) {
      this._host = host;
    }
    this._port = 80;
    this._timeoutSec = 30;
  }
};
var LatLonRegion = class {
};
var LatLonBox = class extends LatLonRegion {
  west;
  east;
  south;
  north;
  constructor(west, east, south, north) {
    super();
    this.west = west;
    this.east = east;
    this.south = south;
    this.north = north;
  }
  asLeafletBounds() {
    return [
      [this.south, this.west],
      [this.north, this.east]
    ];
  }
};
var LatLonRadius = class extends LatLonRegion {
  latitude;
  longitude;
  minRadius;
  maxRadius;
  constructor(latitude, longitude, minRadius, maxRadius) {
    super();
    this.latitude = latitude;
    this.longitude = longitude;
    this.minRadius = minRadius;
    this.maxRadius = maxRadius;
  }
};

// src/traveltime.ts
var IRIS_HOST2 = "service.iris.edu";
var TEXT_FORMAT = "text";
var JSON_FORMAT = "json";
var SVG_FORMAT = "svg";
function isValidTraveltimeJsonType(v) {
  if (!v || typeof v !== "object") {
    return false;
  }
  const object = v;
  if (!(typeof object.model === "string" && (typeof object.sourcedepth === "number" || typeof object.sourceDepth === "number") && (typeof object.receiverdepth === "number" || typeof object.receiverDepth === "number"))) {
    return false;
  }
  if (typeof object.sourceDepth === "number") {
    object.sourcedepth = object.sourceDepth;
    object.sourceDepth = void 0;
  }
  if (typeof object.receiverDepth === "number") {
    object.receiverdepth = object.receiverDepth;
    object.receiverDepth = void 0;
  }
  if (!Array.isArray(object.phases)) {
    return false;
  }
  if (!Array.isArray(object.arrivals)) {
    return false;
  }
  return true;
}
function isValidTraveltimeArrivalType(v) {
  if (!v || typeof v !== "object") {
    return false;
  }
  const object = v;
  return typeof object.distdeg === "number" && typeof object.name === "string" && typeof object.time === "number" && typeof object.rayparam === "number" && typeof object.takeoff === "number" && typeof object.incident === "number" && typeof object.puristdist === "number" && typeof object.puristname === "string";
}
function convertTravelTimeLineToObject(ttimeline) {
  const items = ttimeline.trim().split(/\s+/);
  return {
    distdeg: parseFloat(items[0]),
    phase: items[2],
    time: parseFloat(items[3]),
    rayparam: parseFloat(items[4]),
    takeoff: parseFloat(items[5]),
    incident: parseFloat(items[6]),
    puristdist: parseFloat(items[7]),
    puristname: items[9]
  };
}
function createOriginArrival(distdeg) {
  return {
    distdeg,
    phase: "origin",
    time: 0,
    rayparam: 0,
    takeoff: 0,
    incident: 0,
    puristdist: distdeg,
    puristname: "origin"
  };
}
var TraveltimeQuery = class extends FDSNCommon {
  /** @private */
  _evdepth;
  /** @private */
  _distdeg;
  /** @private */
  _model;
  /** @private */
  _phases;
  /** @private */
  _stalat;
  /** @private */
  _stalon;
  /** @private */
  _receiverdepth;
  /** @private */
  _evlat;
  /** @private */
  _evlon;
  /** @private */
  _format;
  /** @private */
  _noheader;
  constructor(host) {
    if (!isNonEmptyStringArg(host)) {
      host = IRIS_HOST2;
    }
    super(host);
    this._evdepth = 0;
    this._format = JSON_FORMAT;
    this._noheader = false;
  }
  protocol(value) {
    doStringGetterSetter(this, "protocol", value);
    return this;
  }
  getProtocol() {
    return this._protocol;
  }
  host(value) {
    doStringGetterSetter(this, "host", value);
    return this;
  }
  getHost() {
    return this._host;
  }
  /**
   * Gets/Sets the remote port to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  port(value) {
    doIntGetterSetter(this, "port", value);
    return this;
  }
  getPort() {
    return this._port;
  }
  /**
   * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
   * the status code when no matching data is found by the service.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  nodata(value) {
    doIntGetterSetter(this, "nodata", value);
    return this;
  }
  getNodata() {
    return this._nodata;
  }
  specVersion(value) {
    doStringGetterSetter(this, "specVersion", value);
    return this;
  }
  getSpecVersion() {
    return this._specVersion;
  }
  evdepth(value) {
    doFloatGetterSetter(this, "evdepth", value);
    return this;
  }
  evdepthInMeter(value) {
    doFloatGetterSetter(this, "evdepth", isDef(value) ? value / 1e3 : value);
    return this;
  }
  getEvdepth() {
    return this._evdepth;
  }
  distdeg(value) {
    if (typeof value === "number") {
      this._distdeg = [value];
    } else {
      this._distdeg = value;
    }
    return this;
  }
  getDistdeg() {
    return this._distdeg;
  }
  model(value) {
    doStringGetterSetter(this, "model", value);
    return this;
  }
  getModel() {
    return this._model;
  }
  phases(value) {
    doStringGetterSetter(this, "phases", value);
    return this;
  }
  getPhases() {
    return this._phases;
  }
  stalat(value) {
    doFloatGetterSetter(this, "stalat", value);
    return this;
  }
  getStalat() {
    return this._stalat;
  }
  stalon(value) {
    doFloatGetterSetter(this, "stalon", value);
    return this;
  }
  getStalon() {
    return this._stalon;
  }
  latLonFromStation(station) {
    this.stalat(station.latitude);
    this.stalon(station.longitude);
    return this;
  }
  receiverdepth(value) {
    doFloatGetterSetter(this, "receiverdepth", value);
    return this;
  }
  receiverdepthInMeter(value) {
    doFloatGetterSetter(
      this,
      "receiverdepth",
      isDef(value) ? value / 1e3 : value
    );
    return this;
  }
  receiverdepthFromChannel(channel) {
    return this.receiverdepth(channel.depth / 1e3);
  }
  getReceiverdepth() {
    return this._receiverdepth;
  }
  evlat(value) {
    doFloatGetterSetter(this, "evlat", value);
    return this;
  }
  getEvlat() {
    return this._evlat;
  }
  evlon(value) {
    doFloatGetterSetter(this, "evlon", value);
    return this;
  }
  getEvlon() {
    return this._evlon;
  }
  latLonFromQuake(quake) {
    this.evlat(quake.latitude);
    this.evlon(quake.longitude);
    this.evdepthInMeter(quake.depth);
    return this;
  }
  format(value) {
    doStringGetterSetter(this, "format", value);
    return this;
  }
  getFormat() {
    return this._format;
  }
  noheader(value) {
    doBoolGetterSetter(this, "noheader", value);
    return this;
  }
  getNoheader() {
    return this._noheader;
  }
  /**
   * Get/Set the timeout in seconds for the request. Default is 30.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  timeout(value) {
    doFloatGetterSetter(this, "timeoutSec", value);
    return this;
  }
  getTimeout() {
    return this._timeoutSec;
  }
  queryText() {
    this.format(TEXT_FORMAT);
    const url = this.formURL();
    const fetchInit = defaultFetchInitObj(TEXT_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then(
      (response) => {
        if (response.status === 204 || isDef(this._nodata) && response.status === this._nodata) {
          return FAKE_EMPTY_TEXT_MODEL + (isDef(this._model) ? this.getModel() : "") + FAKE_EMPTY_TEXT_HEADERS;
        } else {
          return response.text();
        }
      }
    );
  }
  queryJson() {
    this.format(JSON_FORMAT);
    const url = this.formURL();
    const fetchInit = defaultFetchInitObj(JSON_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then((response) => {
      if (response.status === 204 || isDef(this._nodata) && response.status === this._nodata) {
        return createEmptyTraveltimeJson(this);
      } else {
        return response.json();
      }
    }).then((jsonValue) => {
      if (isValidTraveltimeJsonType(jsonValue)) {
        return jsonValue;
      } else {
        throw new TypeError(`Oops, we did not get root traveltime JSON!`);
      }
    });
  }
  querySvg() {
    this.format(SVG_FORMAT);
    const url = this.formURL();
    const fetchInit = defaultFetchInitObj(SVG_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then((response) => {
      if (response.status === 200) {
        return response.text();
      } else if (response.status === 204 || isDef(this._nodata) && response.status === this._nodata) {
        return FAKE_EMPTY_SVG;
      } else {
        throw new Error(`Status not successful: ${response.status}`);
      }
    }).then(function(rawXmlText) {
      return new DOMParser().parseFromString(rawXmlText, SVG_MIME);
    }).then((xml) => {
      const elArray = xml.getElementsByTagName("svg");
      if (elArray.length > 0) {
        return elArray[0];
      } else {
        throw new Error("Can't find svg element in response");
      }
    });
  }
  queryWadl() {
    return fetch(this.formWadlURL()).then((response) => {
      if (response.ok) {
        return response.text().then(
          (textResponse) => new window.DOMParser().parseFromString(textResponse, "text/xml")
        );
      } else {
        throw new Error(
          `Fetching over network was not ok: ${response.status} ${response.statusText}`
        );
      }
    });
  }
  query() {
    if (this._format === JSON_FORMAT) {
      return this.queryJson();
    } else if (this._format === SVG_FORMAT) {
      return this.querySvg();
    } else if (this._format === TEXT_FORMAT) {
      return this.queryText();
    } else {
      throw new Error("Unknown format: " + this._format);
    }
  }
  formBaseURL() {
    let colon = ":";
    if (this._protocol.endsWith(colon)) {
      colon = "";
    }
    const url = this._protocol + colon + "//" + this._host + (this._port === 80 ? "" : ":" + this._port) + "/irisws/traveltime/" + this._specVersion + "/";
    return url;
  }
  formURL() {
    let url = this.formBaseURL() + "query?";
    if (isDef(this._noheader) && this._noheader) {
      url = url + "noheader=true&";
    }
    if (isDef(this._evdepth) && this._evdepth !== 0) {
      url = url + makeParam("evdepth", this._evdepth);
    }
    if (isDef(this._receiverdepth) && this._receiverdepth !== 0) {
      url = url + makeParam("receiverdepth", this._receiverdepth);
    }
    if (isDef(this._stalat) && isDef(this._stalon)) {
      url = url + makeParam(
        "staloc",
        "[" + stringify(this._stalat) + "," + stringify(this._stalon) + "]"
      );
    }
    if (isDef(this._evlat) && isDef(this._evlon)) {
      url = url + makeParam(
        "evloc",
        "[" + stringify(this._evlat) + "," + stringify(this._evlon) + "]"
      );
    }
    if (isDef(this._distdeg)) {
      url = url + makeParam("distdeg", this._distdeg.join(","));
    }
    if (isDef(this._model)) {
      url = url + makeParam("model", this._model);
    }
    if (isDef(this._phases)) {
      url = url + makeParam("phases", this._phases);
    }
    if (isDef(this._format)) {
      url = url + makeParam("format", this._format);
    }
    if (isDef(this._nodata)) {
      url = url + makeParam("nodata", this._nodata);
    }
    if (url.endsWith("&") || url.endsWith("?")) {
      url = url.substr(0, url.length - 1);
    }
    return url;
  }
  queryTauPVersion() {
    return fetch(this.formTauPVersionURL()).then((response) => {
      if (response.ok) {
        return response.text();
      } else {
        throw new Error(
          "Fetching over network was not ok: " + response.status + " " + response.statusText
        );
      }
    });
  }
  formTauPVersionURL() {
    return this.formBaseURL() + "taupversion";
  }
  formWadlURL() {
    return this.formBaseURL() + "application.wadl";
  }
};
var FAKE_EMPTY_TEXT_MODEL = `Model: `;
var FAKE_EMPTY_TEXT_HEADERS = `
Distance   Depth   Phase   Travel    Ray Param  Takeoff  Incident  Purist    Purist
  (deg)     (km)   Name    Time (s)  p (s/deg)   (deg)    (deg)   Distance   Name
-----------------------------------------------------------------------------------
`;
function createEmptyTraveltimeJson(ttquery) {
  const out = {
    model: isDef(ttquery._model) ? ttquery._model : "",
    sourcedepth: isDef(ttquery._evdepth) ? ttquery._evdepth : 0,
    receiverdepth: isDef(ttquery._receiverdepth) ? ttquery._receiverdepth : 0,
    phases: isDef(ttquery._phases) ? ttquery._phases.split(",") : [],
    arrivals: []
  };
  return out;
}
var FAKE_EMPTY_SVG = `
<svg version="1.1" baseProfile="full" xmlns="http://www.w3.org/2000/svg" width="500" height="500" viewBox="0 0 14016.2 14016.2">
<!--
 This script will plot ray paths generated by TauP using SVG. -->
<defs>
    <style type="text/css"><![CDATA[
        circle {
            vector-effect: non-scaling-stroke;
            stroke: grey;
            fill: none;
        }
        polyline {
            vector-effect: non-scaling-stroke;
            stroke: black;
            fill: none;
        }
    ]]></style>
</defs>
<g transform="translate(7008.1,7008.1)" >
<!-- draw surface and label distances.-->
<!-- tick marks every 30 degrees.-->
  <polyline points="    0.00  -6371.00,     0.00  -6689.55" />
  <polyline points=" 3185.50  -5517.45,  3344.78  -5793.32" />
  <polyline points=" 5517.45  -3185.50,  5793.32  -3344.77" />
  <polyline points=" 6371.00      0.00,  6689.55      0.00" />
  <polyline points=" 5517.45   3185.50,  5793.32   3344.77" />
  <polyline points=" 3185.50   5517.45,  3344.78   5793.32" />
  <polyline points="    0.00   6371.00,     0.00   6689.55" />
  <polyline points="-3185.50   5517.45, -3344.77   5793.32" />
  <polyline points="-5517.45   3185.50, -5793.32   3344.77" />
  <polyline points="-6371.00      0.00, -6689.55      0.00" />
  <polyline points="-5517.45  -3185.50, -5793.32  -3344.78" />
  <polyline points="-3185.50  -5517.45, -3344.78  -5793.32" />
  <circle cx="0.0" cy="0.0" r="6371.0" />
  </g>
</svg>
`;

// src/dataset.ts
import JSZip from "jszip";
var DATASET_DIR = "dataset";
var DOT_ZIP_EXT = ".zip";
var ZIP_FILENAME = DATASET_DIR + DOT_ZIP_EXT;
var SEISMOGRAM_DIR = "seismograms";
var CATALOG_FILE = "catalog.quakeml";
var INVENTORY_FILE = "inventory.staxml";
var Dataset = class _Dataset {
  name = "dataset";
  catalog;
  inventory;
  waveforms;
  processedWaveforms;
  extra;
  constructor() {
    this.catalog = new Array(0);
    this.inventory = new Array(0);
    this.waveforms = new Array(0);
    this.processedWaveforms = new Array(0);
    this.extra = /* @__PURE__ */ new Map();
  }
  async saveToZipFile(filename = ZIP_FILENAME) {
    let dirname = DATASET_DIR;
    if (filename.endsWith(DOT_ZIP_EXT)) {
      dirname = filename.slice(0, -4);
    }
    const zipfile = new JSZip();
    const zip = zipfile.folder(dirname);
    if (!zip) {
      throw new Error("unable to create subfolder in zip file: " + dirname);
    }
    zip.file("Hello.txt", "Hello World\n");
    const seisFolder = zip.folder(SEISMOGRAM_DIR);
    if (seisFolder === null) {
      throw new Error("can't make folder");
    }
    for (const [key, val] of this.waveformsToMSeed3()) {
      seisFolder.file(key, val);
    }
    const content = await zipfile.generateAsync({
      type: "uint8array",
      compression: "DEFLATE"
    });
    downloadBlobAsFile(content, filename);
  }
  waveformsToMSeed3() {
    const out = /* @__PURE__ */ new Map();
    const ext = "ms3";
    this.waveforms.forEach((sdd) => {
      if (sdd.seismogram) {
        const mseed3Records = toMSeed3(
          sdd.seismogram,
          createExtraHeaders("spjs", sdd)
        );
        const byteSize = mseed3Records.reduce(
          (acc, cur) => acc + cur.calcSize(),
          0
        );
        const outBuf = new ArrayBuffer(byteSize);
        let offset = 0;
        mseed3Records.forEach((ms3Rec) => {
          const recSize = ms3Rec.calcSize();
          const dv = new DataView(outBuf, offset, recSize);
          ms3Rec.save(dv);
          offset += recSize;
        });
        let i = 1;
        let seisId;
        if (!!sdd.id && sdd.id.length > 0) {
          seisId = sdd.id;
        } else {
          seisId = sdd.codes();
        }
        let filename = `${seisId}.${ext}`;
        if (out.has(filename)) {
          seisId = `${seisId}_${sdd.startTime.year}-${sdd.startTime.month}-${sdd.startTime.day}`;
        }
        while (out.has(filename)) {
          i += 1;
          filename = `${seisId}_${i}.${ext}`;
        }
        out.set(filename, outBuf);
      }
    });
    return out;
  }
  merge(other) {
    const out = new _Dataset();
    out.waveforms = this.waveforms.concat(other.waveforms);
    out.inventory = this.inventory.concat(other.inventory);
    out.catalog = this.catalog.concat(other.catalog);
    return out;
  }
  associateQuakes(timeOverlapSecs = 1800) {
    this.waveforms.forEach((w) => {
      if (!w.hasQuake()) {
        this.catalog.forEach((q) => {
          if (q.hasPreferredOrigin()) {
            if (q.preferredOrigin?.time) {
              const dur = Duration8.fromMillis(1e3 * timeOverlapSecs);
              const twindow = startDuration(q.preferredOrigin?.time, dur);
              if (twindow.overlaps(w.timeRange)) {
                w.addQuake(q);
              }
            }
          }
        });
      }
    });
  }
  associateChannels() {
    this.waveforms.forEach((sdd) => {
      if (!sdd.hasChannel()) {
        for (const c of allChannels(this.inventory)) {
          if (c.sourceId.equals(sdd.sourceId) && sdd.timeRange.overlaps(c.timeRange)) {
            sdd.channel = c;
            break;
          }
        }
      }
    });
  }
};
function load(url) {
  const fetchInitOptions = defaultFetchInitObj(BINARY_MIME);
  return doFetchWithTimeout(url, fetchInitOptions).then(function(response) {
    if (response.status === 200 || response.status === 0) {
      return response.blob();
    } else {
      throw new Error("No data");
    }
  }).then((data) => JSZip.loadAsync(data)).then((zip) => loadFromZip(zip));
}
async function loadFromFile(file) {
  const zip = await new JSZip().loadAsync(file);
  return loadFromZip(zip);
}
async function loadFromZip(zip) {
  const promiseArray = new Array(0);
  let datasetDir;
  const possibleDirs = zip.folder(new RegExp("/" + SEISMOGRAM_DIR));
  if (possibleDirs.length === 0) {
    throw new Error("Unable to find dataset directory in zip file");
  } else {
    const tmpdatasetDir = zip.folder(
      possibleDirs[0].name.slice(0, -1 * (SEISMOGRAM_DIR.length + 1))
    );
    if (tmpdatasetDir === null) {
      throw new Error("Unable to find dataset directory in zip file");
    } else {
      datasetDir = tmpdatasetDir;
    }
    const seisDir = datasetDir.folder(SEISMOGRAM_DIR);
    if (isDef(seisDir)) {
      seisDir.forEach(function(relativePath, file) {
        if (file.name.endsWith(".ms3")) {
          const seisPromise = file.async("arraybuffer").then(function(buffer) {
            const ms3records = parseMSeed3Records(buffer);
            return sddFromMSeed3(ms3records);
          });
          promiseArray.push(seisPromise);
        }
      });
    }
  }
  const sddListList = await Promise.all(promiseArray);
  const sddList_1 = sddListList.reduce(
    (acc, sddList) => acc.concat(sddList),
    new Array(0)
  );
  const catalogFile = datasetDir.file(CATALOG_FILE);
  const qml = catalogFile ? catalogFile.async("string").then(function(rawXmlText) {
    if (rawXmlText.length === 0) {
      return [];
    } else if (rawXmlText.length < 10) {
      throw new Error(`qml text is really short: ${rawXmlText}`);
    } else {
      const rawXml = new DOMParser().parseFromString(rawXmlText, XML_MIME);
      return parseQuakeML(rawXml).eventList;
    }
  }) : [];
  const inventoryFile = datasetDir.file(INVENTORY_FILE);
  const staml = inventoryFile ? inventoryFile.async("string").then(function(rawXmlText_1) {
    if (rawXmlText_1.length === 0) {
      return [];
    } else if (rawXmlText_1.length < 10) {
      throw new Error(`staxml text is really short: ${rawXmlText_1}`);
    } else {
      const rawXml_2 = new DOMParser().parseFromString(
        rawXmlText_1,
        XML_MIME
      );
      return parseStationXml(rawXml_2);
    }
  }) : [];
  const promises = await Promise.all([sddList_1, qml, staml]);
  const dataset = new Dataset();
  dataset.waveforms = promises[0];
  dataset.catalog = promises[1];
  dataset.inventory = promises[2];
  dataset.associateChannels();
  dataset.associateQuakes();
  return dataset;
}
function sddFromMSeed3(ms3records, ds) {
  const out = [];
  const byChannelMap = byChannel2(ms3records);
  byChannelMap.forEach((ms3segments) => {
    const seis = merge2(ms3segments);
    const sdd = SeismogramDisplayData.fromSeismogram(seis);
    ms3segments.forEach((msr) => {
      insertExtraHeaders(msr.extraHeaders, sdd, "spjs", ds);
    });
    out.push(sdd);
  });
  return out;
}
function insertExtraHeaders(eh, sdd, key, ds) {
  const myEH = eh[key];
  if (!myEH) {
    return;
  }
  if (typeof myEH === "object") {
    if ("quake" in myEH) {
      const qList = myEH["quake"];
      if (qList && Array.isArray(qList)) {
        for (const pid of qList) {
          if (ds) {
            for (const q of ds.catalog) {
              if (q.publicId === pid) {
                sdd.addQuake(q);
              }
            }
          } else {
            qList.forEach((q) => sdd.addQuakeId(q));
          }
        }
      }
    }
    if ("traveltimes" in myEH && Array.isArray(myEH["traveltimes"])) {
      for (const tt of myEH["traveltimes"]) {
        if (isValidTraveltimeArrivalType(tt)) {
          sdd.traveltimeList.push(tt);
        }
      }
    }
    if ("markers" in myEH && Array.isArray(myEH["markers"])) {
      const markers = myEH["markers"];
      markers.forEach((m) => {
        if (m && typeof m === "object") {
          if ("time" in m && typeof m.time === "string") {
            m.time = isoToDateTime(m.time);
          }
          if (isValidMarker(m)) {
            sdd.markerList.push(m);
          }
        }
      });
    }
  }
}
function createExtraHeaders(key, sdd) {
  const h = {};
  const out = {};
  out[key] = h;
  if (sdd.quakeList && sdd.quakeList.length > 0) {
    h["quake"] = sdd.quakeList.map((q) => q.publicId);
  }
  if (sdd.traveltimeList && sdd.traveltimeList.length > 0) {
    h["traveltimes"] = sdd.traveltimeList;
  }
  if (sdd.markerList && sdd.markerList.length > 0) {
    h["markers"] = sdd.markerList;
  }
  return out;
}

// src/fdsnavailability.ts
var fdsnavailability_exports = {};
__export(fdsnavailability_exports, {
  AvailabilityQuery: () => AvailabilityQuery,
  EMPTY_JSON: () => EMPTY_JSON,
  FORMAT_GEOCSV: () => FORMAT_GEOCSV,
  FORMAT_JSON: () => FORMAT_JSON,
  FORMAT_REQUEST: () => FORMAT_REQUEST,
  FORMAT_TEXT: () => FORMAT_TEXT,
  IRIS_HOST: () => IRIS_HOST,
  SERVICE_NAME: () => SERVICE_NAME,
  SERVICE_VERSION: () => SERVICE_VERSION,
  isValidDatasource: () => isValidDatasource,
  isValidRootType: () => isValidRootType
});
var FORMAT_JSON = "json";
var FORMAT_TEXT = "text";
var FORMAT_GEOCSV = "geocsv";
var FORMAT_REQUEST = "request";
var EMPTY_JSON = {
  version: {},
  datasources: []
};
var SERVICE_VERSION = 1;
var SERVICE_NAME = `fdsnws-availability-${SERVICE_VERSION}`;
var AvailabilityQuery = class extends FDSNCommon {
  /** @private */
  _networkCode;
  /** @private */
  _stationCode;
  /** @private */
  _locationCode;
  /** @private */
  _channelCode;
  /** @private */
  _startTime;
  /** @private */
  _endTime;
  /** @private */
  _quality;
  /** @private */
  _merge;
  /** @private */
  _show;
  /** @private */
  _mergeGaps;
  /** @private */
  _limit;
  /** @private */
  _orderby;
  /** @private */
  _includerestricted;
  /** @private */
  _format;
  constructor(host) {
    if (!isNonEmptyStringArg(host)) {
      host = IRIS_HOST;
    }
    super(host);
  }
  /**
   * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
   *  Setting this is probably a bad idea as the code may not be compatible with
   *  the web service.
   *
   * @param value spec version, usually 1
   * @returns the query when setting, the current value when no argument
   */
  specVersion(value) {
    doStringGetterSetter(this, "specVersion", value);
    return this;
  }
  getSpecVersion() {
    return this._specVersion;
  }
  /**
   * Gets/Sets the protocol, http or https. This should match the protocol
   *  of the page loaded, but is autocalculated and generally need not be set.
   *
   * @param value protocol, usually http or https
   * @returns the query when setting, the current value when no argument
   */
  protocol(value) {
    doStringGetterSetter(this, "protocol", value);
    return this;
  }
  getProtocol() {
    return this._protocol;
  }
  /**
   * Gets/Sets the remote host to connect to.
   *
   * @param value host
   * @returns the query when setting, the current value when no argument
   */
  host(value) {
    doStringGetterSetter(this, "host", value);
    return this;
  }
  getHost() {
    return this._host;
  }
  /**
   * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
   * the status code when no matching data is found by the service.
   *
   * @param value number for nodata, usually 404 or 204
   * @returns the query when setting, the current value when no argument
   */
  nodata(value) {
    doIntGetterSetter(this, "nodata", value);
    return this;
  }
  getNodata() {
    return this._nodata;
  }
  /**
   * Gets/Sets the port, not usually set unless not on standard http or https ports
   *
   * @param value port
   * @returns the query when setting, the current value when no argument
   */
  port(value) {
    doIntGetterSetter(this, "port", value);
    return this;
  }
  getPort() {
    return this._port;
  }
  /**
   * Gets/Sets the network code to check.
   *
   * @param value network code like IU
   * @returns the query when setting, the current value when no argument
   */
  networkCode(value) {
    doStringGetterSetter(this, "networkCode", value);
    return this;
  }
  getNetworkCode() {
    return this._networkCode;
  }
  /**
   * Gets/Sets the station code to check.
   *
   * @param value station code like ANMO
   * @returns the query when setting, the current value when no argument
   */
  stationCode(value) {
    doStringGetterSetter(this, "stationCode", value);
    return this;
  }
  getStationCode() {
    return this._stationCode;
  }
  /**
   * Gets/Sets the location code to check.
   *
   * @param value location code like 00
   * @returns the query when setting, the current value when no argument
   */
  locationCode(value) {
    doStringGetterSetter(this, "locationCode", value);
    return this;
  }
  getLocationCode() {
    return this._locationCode;
  }
  /**
   * Gets/Sets the channel code to check.
   *
   * @param value channel code like BHZ
   * @returns the query when setting, the current value when no argument
   */
  channelCode(value) {
    doStringGetterSetter(this, "channelCode", value);
    return this;
  }
  getChannelCode() {
    return this._channelCode;
  }
  /**
   * Gets/Sets the start time parameter for the query.
   *
   * @param value start time
   * @returns the query when setting, the current value when no argument
   */
  startTime(value) {
    doMomentGetterSetter(this, "startTime", value);
    return this;
  }
  getStartTime() {
    return this._startTime;
  }
  /**
   * Gets/Sets the end time parameter for the query.
   *
   * @param value end time
   * @returns the query when setting, the current value when no argument
   */
  endTime(value) {
    doMomentGetterSetter(this, "endTime", value);
    return this;
  }
  getEndTime() {
    return this._endTime;
  }
  /**
   * Sets startTime and endTime using the given time window
   *
   * @param   se time window
   * @returns    the query
   */
  timeRange(se) {
    this.startTime(validStartTime(se));
    this.endTime(validEndTime(se));
    return this;
  }
  /**
   * Gets/Sets the quality parameter for the query.
   *
   * @param value quality
   * @returns the query when setting, the current value when no argument
   */
  quality(value) {
    doStringGetterSetter(this, "quality", value);
    return this;
  }
  getQuality() {
    return this._quality;
  }
  /**
   * Gets/Sets the merge parameter for the query.
   *
   * @param value merge
   * @returns the query when setting, the current value when no argument
   */
  merge(value) {
    doStringGetterSetter(this, "merge", value);
    return this;
  }
  getMerge() {
    return this._merge;
  }
  /**
   * Gets/Sets the mergegaps parameter for the query.
   *
   * @param value merge gaps
   * @returns the query when setting, the current value when no argument
   */
  mergeGaps(value) {
    doFloatGetterSetter(this, "mergeGaps", value);
    return this;
  }
  getMergeGaps() {
    return this._mergeGaps;
  }
  /**
   * Gets/Sets the show parameter for the query.
   *
   * @param value show
   * @returns the query when setting, the current value when no argument
   */
  show(value) {
    doStringGetterSetter(this, "show", value);
    return this;
  }
  getShow() {
    return this._show;
  }
  /**
   * Gets/Sets the limit parameter for the query.
   *
   * @param value limit
   * @returns the query when setting, the current value when no argument
   */
  limit(value) {
    doIntGetterSetter(this, "limit", value);
    return this;
  }
  getLimit() {
    return this._limit;
  }
  /**
   * Gets/Sets the order by parameter for the query.
   *
   * @param value order by
   * @returns the query when setting, the current value when no argument
   */
  orderby(value) {
    doStringGetterSetter(this, "orderBy", value);
    return this;
  }
  getOrderBy() {
    return this._orderby;
  }
  /**
   * Gets/Sets the include restricted data parameter for the query.
   *
   * @param value true to include restricted data
   * @returns the query when setting, the current value when no argument
   */
  includeRestricted(value) {
    doBoolGetterSetter(this, "includerestricted", value);
    return this;
  }
  getIncludeRestricted() {
    return this._includerestricted;
  }
  /**
   * Gets/Sets the format parameter for the query. Usually not needed as is set
   * by the various query methods.
   *
   * @param value format
   * @returns the query when setting, the current value when no argument
   */
  format(value) {
    doStringGetterSetter(this, "format", value);
    return this;
  }
  getFormat() {
    return this._format;
  }
  /**
   * Get/Set the timeout in seconds for the request. Default is 30.
   *
   * @param value timeout in seconds
   * @returns the query when setting, the current value when no argument
   */
  timeout(value) {
    doFloatGetterSetter(this, "timeoutSec", value);
    return this;
  }
  getTimeout() {
    return this._timeoutSec;
  }
  /**
   * Calls query on the remote service, using configured parameters. Calls
   * queryJson internally, then unpacks the result into array of SeismogramDisplayData
   * objects.
   *
   * @returns          promise to array of SeismogramDisplayData, each representing
   * a channel-time window
   */
  query() {
    return this.queryJson().then((json) => {
      return this.extractFromJson(json);
    });
  }
  /**
   * Calls the query function the remote server and parses the returned data as json.
   *
   * @returns promise to the result as json
   */
  queryJson() {
    this.format(FORMAT_JSON);
    const url = this.formURL("query");
    const fetchInit = defaultFetchInitObj(JSON_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then((response) => {
      if (response.status === 204 || isDef(this._nodata) && response.status === this._nodata) {
        return EMPTY_JSON;
      }
      const contentType = response.headers.get("content-type");
      if (isNonEmptyStringArg(contentType) && contentType.includes(JSON_MIME)) {
        return response.json();
      }
      throw new TypeError(`Oops, we did not get JSON! ${contentType}`);
    }).then((jsonValue) => {
      if (isValidRootType(jsonValue)) {
        return jsonValue;
      }
      throw new TypeError(`Oops, we did not get valid root type json`);
    });
  }
  /**
   * Calls extent on the remote service, using configured parameters. Calls
   * extentJson internally, then unpacks the result into array of SeismogramDisplayData
   * objects.
   *
   * @returns          promise to array of SeismogramDisplayData, each representing
   * a channel-time window
   */
  extent() {
    return this.extentJson().then((json) => {
      return this.extractFromJson(json);
    });
  }
  /**
   * Call the extend function on the remote server and parses the returned data as json.
   *
   * @returns promise to the result as json
   */
  extentJson() {
    this.format(FORMAT_JSON);
    const url = this.formURL("extent");
    const fetchInit = defaultFetchInitObj(JSON_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then((response) => {
      if (response.status === 204 || isDef(this._nodata) && response.status === this._nodata) {
        return EMPTY_JSON;
      }
      const contentType = response.headers.get("content-type");
      if (isNonEmptyStringArg(contentType) && contentType.includes(JSON_MIME)) {
        return response.json();
      }
      throw new TypeError(`Oops, we did not get JSON! ${contentType}`);
    }).then((jsonValue) => {
      if (isValidRootType(jsonValue)) {
        return jsonValue;
      }
      throw new TypeError(`Oops, we did not get valid root type json`);
    });
  }
  /**
   * Calls query on the remote service using POST, using configured parameters
   * and forms the POST body using the channelTimeList. Calls
   * postQueryJson internally, then unpacks the result into array of SeismogramDisplayData
   * objects.
   *
   * @param channelTimeList array of channel-time windows for the request
   * @returns          promise to array of SeismogramDisplayData, each representing
   * a channel-time window
   */
  postQuery(channelTimeList) {
    return this.postQueryJson(channelTimeList).then((json) => {
      return this.extractFromJson(json);
    });
  }
  postExtent(channelTimeList) {
    return this.postExtentJson(channelTimeList).then((json) => {
      return this.extractFromJson(json);
    });
  }
  postExtentJson(channelTimeList) {
    return this.postJson(channelTimeList, "extent");
  }
  postQueryJson(channelTimeList) {
    return this.postJson(channelTimeList, "query");
  }
  postJson(channelTimeList, method) {
    this.format(FORMAT_JSON);
    return this.postRaw(channelTimeList, method).then((response) => {
      if (response.status === 204 || isDef(this._nodata) && response.status === this._nodata) {
        return EMPTY_JSON;
      }
      const contentType = response.headers.get("content-type");
      if (isNonEmptyStringArg(contentType) && contentType.includes(JSON_MIME)) {
        return response.json();
      }
      throw new TypeError(`Oops, we did not get JSON! ${contentType}`);
    }).then((jsonValue) => {
      if (isValidRootType(jsonValue)) {
        return jsonValue;
      }
      throw new TypeError(`Oops, we did not get valid root type json`);
    });
  }
  postRaw(channelTimeList, method) {
    if (channelTimeList.length === 0) {
      return Promise.resolve(
        new Response(null, {
          status: 204
        })
      );
    } else {
      const fetchInit = defaultFetchInitObj(JSON_MIME);
      fetchInit.method = "POST";
      fetchInit.body = this.createPostBody(channelTimeList);
      return fetch(this.formBaseURL() + `/${method}?`, fetchInit).then(
        function(response) {
          if (response.ok) {
            return response;
          }
          throw new Error("Fetch response was not ok.");
        }
      );
    }
  }
  extractFromJson(jsonChanTimes) {
    const out = [];
    const knownNets = /* @__PURE__ */ new Map();
    if (isDef(jsonChanTimes.datasources)) {
      for (const ds of jsonChanTimes.datasources) {
        if (isValidDatasource(ds)) {
          let n = knownNets.get(ds.network);
          if (!n) {
            n = new Network(ds.network);
            knownNets.set(ds.network, n);
          }
          let s = null;
          for (const ss of n.stations) {
            if (ss.stationCode === ds.station) {
              s = ss;
            }
          }
          if (!s) {
            s = new Station(n, ds.station);
            n.stations.push(s);
          }
          const c = new Channel(s, ds.channel, ds.location);
          if (isNonEmptyStringArg(ds.earliest) && isNonEmptyStringArg(ds.latest)) {
            out.push(
              SeismogramDisplayData.fromChannelAndTimes(
                c,
                isoToDateTime(ds.earliest),
                isoToDateTime(ds.latest)
              )
            );
          } else if (ds.timespans) {
            for (const ts of ds.timespans) {
              if (Array.isArray(ts) && ts.length === 2 && typeof ts[0] === "string" && typeof ts[1] === "string") {
                out.push(
                  SeismogramDisplayData.fromChannelAndTimes(
                    c,
                    isoToDateTime(ts[0]),
                    isoToDateTime(ts[1])
                  )
                );
              } else {
                throw new TypeError("invalid timespans: " + stringify(ts));
              }
            }
          }
        }
      }
    }
    return out;
  }
  createPostBody(channelTimeList) {
    let out = "";
    if (this._quality) {
      out += this.makePostParm("quality", this.quality());
    }
    if (this._merge) {
      out += this.makePostParm("merge", this.merge());
    }
    if (isNumArg(this._mergeGaps) && (this._format === "query" || this._format === "queryauth")) {
      out += this.makePostParm("mergegaps", this.mergeGaps());
    }
    if (this._show && (this._format === "query" || this._format === "queryauth")) {
      out += this.makePostParm("show", this.show());
    }
    if (isNumArg(this._limit) && this._limit > 0) {
      out += this.makePostParm("limit", this.limit());
    }
    if (this._orderby) {
      out += this.makePostParm("orderby", this.orderby());
    }
    if (this._includerestricted) {
      out += this.makePostParm("includerestricted", this.includeRestricted());
    }
    if (this._format) {
      out += this.makePostParm("format", this.format());
    }
    if (this._nodata) {
      out += this.makePostParm("nodata", this.nodata());
    }
    for (const ct of channelTimeList) {
      if (isDef(ct.channel)) {
        const sta = ct.channel.station;
        const net = sta.network;
        out += `${net.networkCode} ${sta.stationCode} ${ct.channel.locationCode} ${ct.channel.channelCode} ${toIsoWoZ(ct.startTime)} ${toIsoWoZ(ct.endTime)}`;
        out += "\n";
      } else {
        throw new Error("Channel in missing in createPostBody");
      }
    }
    return out;
  }
  formBaseURL() {
    let colon = ":";
    if (this._protocol.endsWith(colon)) {
      colon = "";
    }
    return this._protocol + colon + "//" + this._host + (this._port === 80 ? "" : ":" + stringify(this._port)) + "/fdsnws/availability/" + this._specVersion;
  }
  formVersionURL() {
    return this.formBaseURL() + "/version";
  }
  /**
   * Queries the remote web service to get its version
   *
   * @returns Promise to version string
   */
  queryVersion() {
    const url = this.formVersionURL();
    const fetchInit = defaultFetchInitObj(TEXT_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then(
      (response) => {
        if (response.status === 200) {
          return response.text();
        } else {
          throw new Error(`Status not 200: ${response.status}`);
        }
      }
    );
  }
  makePostParm(name, val) {
    return name + "=" + stringify(val) + "\n";
  }
  formURL(method) {
    if (hasNoArgs(method)) {
      method = "query";
    }
    let url = this.formBaseURL() + `/${method}?`;
    if (this._networkCode) {
      url = url + makeParam("net", this._networkCode);
    }
    if (this._stationCode) {
      url = url + makeParam("sta", this._stationCode);
    }
    if (this._locationCode) {
      url = url + makeParam("loc", this._locationCode);
    }
    if (this._channelCode) {
      url = url + makeParam("cha", this._channelCode);
    }
    if (this._startTime) {
      url = url + makeParam("starttime", toIsoWoZ(this._startTime));
    }
    if (this._endTime) {
      url = url + makeParam("endtime", toIsoWoZ(this._endTime));
    }
    if (this._quality) {
      url = url + makeParam("quality", this._quality);
    }
    if (this._merge) {
      url = url + makeParam("merge", this._merge);
    }
    if (this._mergeGaps) {
      url = url + makeParam("mergegaps", this._mergeGaps);
    }
    if (this._show) {
      url = url + makeParam("show", this._show);
    }
    if (isNumArg(this._limit) && this._limit > 0) {
      url = url + makeParam("limit", this._limit);
    }
    if (this._orderby) {
      url = url + makeParam("orderby", this._orderby);
    }
    if (this._includerestricted) {
      url = url + makeParam("includerestricted", this._includerestricted);
    }
    if (this._format) {
      url = url + makeParam("format", this._format);
    }
    if (this._nodata) {
      url = url + makeParam("nodata", this._nodata);
    }
    if (url.endsWith("&") || url.endsWith("?")) {
      url = url.substr(0, url.length - 1);
    }
    return url;
  }
};
function isValidRootType(jsonValue) {
  if (!jsonValue || typeof jsonValue !== "object") {
    throw new TypeError("json is not object");
  }
  const jsonObj = jsonValue;
  if (Array.isArray(jsonObj.datasources) && jsonObj.datasources.every(isValidDatasource) && typeof jsonObj.version === "number") {
    return true;
  } else {
    throw new TypeError("json is not valid for FDSN Availability");
  }
}
function isValidDatasource(jsonValue) {
  if (!jsonValue || typeof jsonValue !== "object") {
    throw new TypeError("json is not object");
  }
  const jsonObj = jsonValue;
  if (typeof jsonObj.network === "string" && typeof jsonObj.station === "string" && typeof jsonObj.location === "string" && typeof jsonObj.channel === "string") {
    return true;
  } else {
    throw new TypeError("json datasource is not valid for FDSN Availability");
  }
}

// src/fdsndatacenters.ts
var fdsndatacenters_exports = {};
__export(fdsndatacenters_exports, {
  DataCentersQuery: () => DataCentersQuery,
  FDSN_HOST: () => FDSN_HOST,
  isValidRootType: () => isValidRootType2
});

// src/fdsndataselect.ts
var fdsndataselect_exports = {};
__export(fdsndataselect_exports, {
  DataSelectQuery: () => DataSelectQuery,
  FORMAT_MINISEED: () => FORMAT_MINISEED,
  FORMAT_MINISEED_THREE: () => FORMAT_MINISEED_THREE,
  IRIS_HOST: () => IRIS_HOST,
  SERVICE_NAME: () => SERVICE_NAME2,
  SERVICE_VERSION: () => SERVICE_VERSION2,
  createDataSelectQuery: () => createDataSelectQuery
});
var FORMAT_MINISEED = "miniseed";
var FORMAT_MINISEED_THREE = "miniseed3";
var SERVICE_VERSION2 = 1;
var SERVICE_NAME2 = `fdsnws-dataselect-${SERVICE_VERSION2}`;
var DataSelectQuery = class _DataSelectQuery extends FDSNCommon {
  /** @private */
  _networkCode;
  /** @private */
  _stationCode;
  /** @private */
  _locationCode;
  /** @private */
  _channelCode;
  /** @private */
  _startTime;
  /** @private */
  _endTime;
  /** @private */
  _quality;
  /** @private */
  _minimumLength;
  /** @private */
  _longestOnly;
  /** @private */
  _repository;
  /** @private */
  _format;
  constructor(host) {
    if (!isNonEmptyStringArg(host)) {
      host = IRIS_HOST;
    }
    super(host);
  }
  /**
   * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
   *  Setting this is probably a bad idea as the code may not be compatible with
   *  the web service.
   *
   * @param value spec version, usually 1
   * @returns new value if getting, this if setting
   */
  specVersion(value) {
    doStringGetterSetter(this, "specVersion", value);
    return this;
  }
  getSpecVersion() {
    return this._specVersion;
  }
  /**
   * Gets/Sets the protocol, http or https. This should match the protocol
   *  of the page loaded, but is autocalculated and generally need not be set.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  protocol(value) {
    doStringGetterSetter(this, "protocol", value);
    return this;
  }
  getProtocol() {
    return this._protocol;
  }
  /**
   * Gets/Sets the remote host to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  host(value) {
    doStringGetterSetter(this, "host", value);
    return this;
  }
  getHost() {
    return this._host;
  }
  /**
   * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
   * the status code when no matching data is found by the service.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  nodata(value) {
    doIntGetterSetter(this, "nodata", value);
    return this;
  }
  getNodata() {
    return this._nodata;
  }
  /**
   * Gets/Sets the remote port to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  port(value) {
    doIntGetterSetter(this, "port", value);
    return this;
  }
  getPort() {
    return this._port;
  }
  /**
   * Get/Set the network query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  networkCode(value) {
    doStringGetterSetter(this, "networkCode", value);
    return this;
  }
  getNetworkCode() {
    return this._networkCode;
  }
  /**
   * Get/Set the station query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  stationCode(value) {
    doStringGetterSetter(this, "stationCode", value);
    return this;
  }
  getStationCode() {
    return this._stationCode;
  }
  /**
   * Get/Set the location code query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  locationCode(value) {
    doStringGetterSetter(this, "locationCode", value);
    return this;
  }
  getLocationCode() {
    return this._locationCode;
  }
  /**
   * Get/Set the channel query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  channelCode(value) {
    doStringGetterSetter(this, "channelCode", value);
    return this;
  }
  getChannelCode() {
    return this._channelCode;
  }
  nslcCodes(channelId) {
    this.networkCode(channelId.networkCode);
    this.stationCode(channelId.stationCode);
    this.locationCode(channelId.locationCode);
    this.channelCode(channelId.channelCode);
    return this;
  }
  /**
   * Get/Set the starttime query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  startTime(value) {
    doMomentGetterSetter(this, "startTime", value);
    return this;
  }
  getStartTime() {
    return this._startTime;
  }
  /**
   * Get/Set the endtime query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  endTime(value) {
    doMomentGetterSetter(this, "endTime", value);
    return this;
  }
  getEndTime() {
    return this._endTime;
  }
  /**
   * Sets startTime and endTime using the given time range
   *
   * @param   se time range
   * @returns     this
   */
  timeRange(se) {
    this.startTime(validStartTime(se));
    this.endTime(validEndTime(se));
    return this;
  }
  /**
   * Get/Set the quality query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  quality(value) {
    doStringGetterSetter(this, "quality", value);
    return this;
  }
  getQuality() {
    return this._quality;
  }
  /**
   * Get/Set the minimum length query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minimumLength(value) {
    doFloatGetterSetter(this, "minimumLength", value);
    return this;
  }
  getMinimumLength() {
    return this._minimumLength;
  }
  /**
   * Get/Set the longest only query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  longestOnly(value) {
    doBoolGetterSetter(this, "longestOnly", value);
    return this;
  }
  getLongestOnly() {
    return this._longestOnly;
  }
  /**
   * set or get the repository paramter. This is an IRIS-specific
   * parameter that will not work with other dataselect web services.
   *
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  repository(value) {
    doStringGetterSetter(this, "repository", value);
    return this;
  }
  getRepository() {
    return this._repository;
  }
  /**
   * Get/Set the format query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  format(value) {
    doStringGetterSetter(this, "format", value);
    return this;
  }
  getFormat() {
    return this._format;
  }
  /**
   * Get/Set the timeout in seconds for the request. Default is 30.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  timeout(value) {
    doFloatGetterSetter(this, "timeoutSec", value);
    return this;
  }
  getTimeout() {
    return this._timeoutSec;
  }
  /**
   * queries the web service using the configured parameters, parsing the response
   * into miniseed data records.
   *
   * @returns Promise to Array of miniseed.DataRecords
   */
  queryDataRecords() {
    this.format(FORMAT_MINISEED);
    const url = this.formURL();
    const fetchInit = defaultFetchInitObj(MINISEED_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then((response) => {
      if (response.status === 204 || isDef(this._nodata) && response.status === this.getNodata()) {
        return new ArrayBuffer(0);
      } else {
        return response.arrayBuffer();
      }
    }).then(function(rawBuffer) {
      const dataRecords = parseDataRecords(rawBuffer);
      return dataRecords;
    });
  }
  /**
   * queries the web service using the configured parameters, parsing the response
   * into miniseed data records.
   *
   * @returns Promise to Array of miniseed.DataRecords
   */
  queryMS3Records() {
    this.format(FORMAT_MINISEED_THREE);
    const url = this.formURL();
    const fetchInit = defaultFetchInitObj(MINISEED_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then((response) => {
      if (response.status === 204 || isDef(this._nodata) && response.status === this.getNodata()) {
        return new ArrayBuffer(0);
      } else {
        return response.arrayBuffer();
      }
    }).then(function(rawBuffer) {
      const dataRecords = parseMSeed3Records(rawBuffer);
      return dataRecords;
    });
  }
  /**
   * queries the web service using the configured parameters, parsing the response
   * into miniseed data records and then combining the data records into
   * Seismogram objects.
   *
   * @returns Promise to Array of Seismogram objects
   */
  querySeismograms() {
    if (this._format === FORMAT_MINISEED_THREE) {
      return this.queryMS3Records().then((dataRecords) => {
        return seismogramPerChannel2(dataRecords);
      });
    } else {
      return this.queryDataRecords().then((dataRecords) => {
        return seismogramPerChannel(dataRecords);
      });
    }
  }
  postQueryDataRecords(channelTimeList) {
    return this.postQueryRaw(channelTimeList).then((fetchResponse) => {
      if (fetchResponse.ok) {
        return fetchResponse.arrayBuffer().then((ab) => {
          return parseDataRecords(ab);
        });
      } else {
        log("fetchResponse not ok");
        return [];
      }
    });
  }
  postQueryMS3Records(channelTimeList) {
    return this.postQueryRaw(channelTimeList).then((fetchResponse) => {
      if (fetchResponse.ok) {
        return fetchResponse.arrayBuffer().then((ab) => {
          return parseMSeed3Records(ab);
        });
      } else {
        log("fetchResponse not ok");
        return [];
      }
    });
  }
  /**
   * query the dataselect server using post, which allows for multiple
   * channel-timeranges at once. This assumes that there are not multiple
   * time ranges for the same channel as the results, encapsulated as
   * SeismogramDisplayData objects, are returned one seismogram
   * per channel, which may contain gaps. The original channel and timerange are
   * also populated with each result.
   *
   * @param   sddList array of SeismogramDisplayData objects
   * that will be filled in with the resulting seismogram
   * @returns Promise to the input Array of SeismogramDisplayData objects, each with the
   * seismogram containing the data returned from the server
   */
  postQuerySeismograms(sddList) {
    let seismogramPromise;
    if (this._format === FORMAT_MINISEED_THREE) {
      seismogramPromise = this.postQueryMS3Records(sddList).then(
        (dataRecords) => {
          return seismogramSegmentPerChannel2(dataRecords);
        }
      );
    } else {
      seismogramPromise = this.postQueryDataRecords(sddList).then(
        (dataRecords) => {
          return seismogramSegmentPerChannel(dataRecords);
        }
      );
    }
    return seismogramPromise.then((seisArray) => {
      for (const sdd of sddList) {
        const sddNslc = sdd.nslcId;
        const segList = seisArray.filter(
          (s) => s.nslcId.equals(sddNslc) && s.timeRange.overlaps(sdd.timeRange)
        );
        if (segList.length > 0) {
          const seis = new Seismogram(segList).trim(sdd.timeRange);
          sdd.seismogram = seis;
        }
      }
      return sddList;
    });
  }
  postQueryRaw(sddList) {
    if (sddList.length === 0) {
      return Promise.resolve(
        new Response(null, {
          status: 204
        })
      );
    } else {
      return this.postQueryRawWithBody(_DataSelectQuery.createPostBody(sddList));
    }
  }
  postQueryRawWithBody(body) {
    const fetchInit = defaultFetchInitObj(MINISEED_MIME);
    fetchInit.method = "POST";
    fetchInit.body = body;
    return doFetchWithTimeout(
      this.formPostURL(),
      fetchInit,
      this._timeoutSec * 1e3
    );
  }
  static createPostBody(sddList) {
    let out = "";
    for (const sdd of sddList) {
      const locCode = sdd.locationCode.trim() === "" ? "--" : sdd.locationCode;
      out += `${sdd.networkCode} ${sdd.stationCode} ${locCode} ${sdd.channelCode} ${toIsoWoZ(sdd.startTime)} ${toIsoWoZ(sdd.endTime)}`;
      out += "\n";
    }
    return out;
  }
  formBaseURL() {
    let colon = ":";
    if (this._protocol.endsWith(colon)) {
      colon = "";
    }
    return this._protocol + colon + "//" + this._host + (this._port === 80 ? "" : ":" + String(this._port)) + "/fdsnws/dataselect/" + this._specVersion;
  }
  formVersionURL() {
    return this.formBaseURL() + "/version";
  }
  /**
   * Queries the remote web service to get its version
   *
   * @returns Promise to version string
   */
  queryVersion() {
    const url = this.formVersionURL();
    const fetchInit = defaultFetchInitObj(TEXT_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then(
      (response) => {
        if (response.status === 200) {
          return response.text();
        } else {
          throw new Error(`Status not 200: ${response.status}`);
        }
      }
    );
  }
  formPostURL() {
    return this.formBaseURL() + "/query";
  }
  formURL() {
    let url = this.formBaseURL() + "/query?";
    if (isStringArg(this._networkCode) && this._networkCode.length > 0 && this._networkCode !== "*") {
      url = url + makeParam("net", this._networkCode);
    }
    if (isStringArg(this._stationCode) && this._stationCode.length > 0 && this._stationCode !== "*") {
      url = url + makeParam("sta", this._stationCode);
    }
    if (isStringArg(this._locationCode) && this._locationCode.length > 0 && this._locationCode !== "*") {
      url = url + makeParam("loc", this._locationCode);
    }
    if (isStringArg(this._channelCode) && this._channelCode.length > 0 && this._channelCode !== "*") {
      url = url + makeParam("cha", this._channelCode);
    }
    if (this._startTime) {
      url = url + makeParam("starttime", toIsoWoZ(this._startTime));
    }
    if (this._endTime) {
      url = url + makeParam("endtime", toIsoWoZ(this._endTime));
    }
    if (this._quality) {
      url = url + makeParam("quality", this._quality);
    }
    if (this._minimumLength) {
      url = url + makeParam("minimumlength", this._minimumLength);
    }
    if (this._repository) {
      url = url + makeParam("repository", this._repository);
    }
    if (this._longestOnly) {
      url = url + makeParam("longestonly", this._longestOnly);
    }
    if (this._format) {
      url = url + makeParam("format", this._format);
    }
    if (this._nodata) {
      url = url + makeParam("nodata", this._nodata);
    }
    if (url.endsWith("&") || url.endsWith("?")) {
      url = url.substr(0, url.length - 1);
    }
    return url;
  }
};
function createDataSelectQuery(params) {
  if (!params || typeof params !== "object") {
    throw new Error("params null or not an object");
  }
  const out = new DataSelectQuery();
  if (params.net) {
    out.networkCode(params.net);
  }
  if (params.network) {
    out.networkCode(params.network);
  }
  if (params.networkCode) {
    out.networkCode(params.networkCode);
  }
  if (params.sta) {
    out.stationCode(params.sta);
  }
  if (params.station) {
    out.stationCode(params.station);
  }
  if (params.stationCode) {
    out.stationCode(params.stationCode);
  }
  if (params.loc) {
    out.locationCode(params.loc);
  }
  if (params.location) {
    out.locationCode(params.location);
  }
  if (params.locationCode) {
    out.locationCode(params.locationCode);
  }
  if (params.chan) {
    out.channelCode(params.chan);
  }
  if (params.channel) {
    out.channelCode(params.channel);
  }
  if (params.channelCode) {
    out.channelCode(params.channelCode);
  }
  if (params.start || params.starttime) {
    const s = params.start ? params.start : params.starttime;
    out.startTime(isoToDateTime(s));
  }
  if (params.end || params.endtime) {
    const e = params.end ? params.end : params.endtime;
    out.endTime(isoToDateTime(e));
  }
  if (params.quality) {
    out.quality(params.quality);
  }
  if (params.minimumlength) {
    out.minimumLength(parseInt(params.minimumlength));
  }
  if (params.repository) {
    out.repository(params.repository);
  }
  if (params.longestonly) {
    out.longestOnly(params.longestonly.toLowerCase() === "true");
  }
  if (params.format) {
    out.format(params.format);
  }
  if (params.nodata) {
    out.nodata(parseInt(params.nodata));
  }
  if (params.host) {
    out.host(params.host);
  }
  if (params.port) {
    out.port(parseInt(params.port));
  }
  if (params.specVersion) {
    out.specVersion(params.specVersion);
  }
  return out;
}

// src/fdsnevent.ts
var fdsnevent_exports = {};
__export(fdsnevent_exports, {
  EventQuery: () => EventQuery,
  SERVICE_NAME: () => SERVICE_NAME3,
  SERVICE_VERSION: () => SERVICE_VERSION3,
  USGS_HOST: () => USGS_HOST
});
var SERVICE_VERSION3 = 1;
var SERVICE_NAME3 = `fdsnws-event-${SERVICE_VERSION3}`;
var EventQuery = class extends FDSNCommon {
  /** @private */
  _eventId;
  /** @private */
  _startTime;
  /** @private */
  _endTime;
  /** @private */
  _updatedAfter;
  /** @private */
  _minMag;
  /** @private */
  _maxMag;
  /** @private */
  _magnitudeType;
  /** @private */
  _minDepth;
  /** @private */
  _maxDepth;
  /** @private */
  _minLat;
  /** @private */
  _maxLat;
  /** @private */
  _minLon;
  /** @private */
  _maxLon;
  /** @private */
  _latitude;
  /** @private */
  _longitude;
  /** @private */
  _minRadius;
  /** @private */
  _maxRadius;
  /** @private */
  _includeArrivals;
  /** @private */
  _includeAllOrigins;
  /** @private */
  _includeAllMagnitudes;
  /** @private */
  _limit;
  /** @private */
  _offset;
  /** @private */
  _orderBy;
  /** @private */
  _contributor;
  /** @private */
  _catalog;
  /** @private */
  _format;
  constructor(host) {
    if (!isNonEmptyStringArg(host)) {
      host = USGS_HOST;
    }
    super(host);
  }
  /**
   * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
   *  Setting this is probably a bad idea as the code may not be compatible with
   *  the web service.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  specVersion(value) {
    doStringGetterSetter(this, "specVersion", value);
    return this;
  }
  getSpecVersion() {
    return this._specVersion;
  }
  /**
   * Gets/Sets the protocol, http or https. This should match the protocol
   *  of the page loaded, but is autocalculated and generally need not be set.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  protocol(value) {
    doStringGetterSetter(this, "protocol", value);
    return this;
  }
  getProtocol() {
    return this._protocol;
  }
  /**
   * Gets/Sets the remote host to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  host(value) {
    doStringGetterSetter(this, "host", value);
    return this;
  }
  getHost() {
    return this._host;
  }
  /**
   * Gets/Sets the remote port to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  port(value) {
    doIntGetterSetter(this, "port", value);
    return this;
  }
  getPort() {
    return this._port;
  }
  /**
   * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
   * the status code when no matching data is found by the service.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  nodata(value) {
    doIntGetterSetter(this, "nodata", value);
    return this;
  }
  getNodata() {
    return this._nodata;
  }
  /**
   * Get/Set the timeout in seconds for the request. Default is 30.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  timeout(value) {
    doFloatGetterSetter(this, "timeoutSec", value);
    return this;
  }
  getTimeout() {
    return this._timeoutSec;
  }
  /**
   * Get/Set the eventid query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  eventId(value) {
    doStringGetterSetter(this, "eventId", value);
    return this;
  }
  getEventId() {
    return this._eventId;
  }
  /**
   * Get/Set the starttime query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  startTime(value) {
    doMomentGetterSetter(this, "startTime", value);
    return this;
  }
  getStartTime() {
    return this._startTime;
  }
  /**
   * Get/Set the endtime query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  endTime(value) {
    doMomentGetterSetter(this, "endTime", value);
    return this;
  }
  getEndTime() {
    return this._endTime;
  }
  /**
   * Sets startTime and endTime using the given time window
   *
   * @param   se time window
   * @returns     this
   */
  timeRange(se) {
    this.startTime(validStartTime(se));
    this.endTime(validEndTime(se));
    return this;
  }
  /**
   * Get/Set the updatedafter query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  updatedAfter(value) {
    doMomentGetterSetter(this, "updatedAfter", value);
    return this;
  }
  getUpdatedAfter() {
    return this._updatedAfter;
  }
  /**
   * Get/Set the minmag query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minMag(value) {
    doFloatGetterSetter(this, "minMag", value);
    return this;
  }
  getMinMag() {
    return this._minMag;
  }
  /**
   * Get/Set the maxmag query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxMag(value) {
    doFloatGetterSetter(this, "maxMag", value);
    return this;
  }
  getMaxMag() {
    return this._maxMag;
  }
  /**
   * Get/Set the magnitudetype query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  magnitudeType(value) {
    doStringGetterSetter(this, "magnitudeType", value);
    return this;
  }
  getMagnitudeType() {
    return this._magnitudeType;
  }
  /**
   * Get/Set the mindepth query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minDepth(value) {
    doFloatGetterSetter(this, "minDepth", value);
    return this;
  }
  getMinDepth() {
    return this._minDepth;
  }
  /**
   * Get/Set the maxdepth query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxDepth(value) {
    doFloatGetterSetter(this, "maxDepth", value);
    return this;
  }
  getMaxDepth() {
    return this._maxDepth;
  }
  /**
   * Get/Set the minlat query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minLat(value) {
    doFloatGetterSetter(this, "minLat", value);
    return this;
  }
  getMinLat() {
    return this._minLat;
  }
  /**
   * Get/Set the maxlat query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxLat(value) {
    doFloatGetterSetter(this, "maxLat", value);
    return this;
  }
  getMaxLat() {
    return this._maxLat;
  }
  /**
   * Get/Set the minlon query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minLon(value) {
    doFloatGetterSetter(this, "minLon", value);
    return this;
  }
  getMinLon() {
    return this._minLon;
  }
  /**
   * Get/Set the maxlon query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxLon(value) {
    doFloatGetterSetter(this, "maxLon", value);
    return this;
  }
  getMaxLon() {
    return this._maxLon;
  }
  /**
   * Get/Set the latitude query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  latitude(value) {
    doFloatGetterSetter(this, "latitude", value);
    return this;
  }
  getLatitude() {
    return this._latitude;
  }
  /**
   * Get/Set the longitude query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  longitude(value) {
    doFloatGetterSetter(this, "longitude", value);
    return this;
  }
  getLongitude() {
    return this._longitude;
  }
  /**
   * Get/Set the minradius query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minRadius(value) {
    doFloatGetterSetter(this, "minRadius", value);
    return this;
  }
  getMinRadius() {
    return this._minRadius;
  }
  /**
   * Get/Set the maxradius query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxRadius(value) {
    doFloatGetterSetter(this, "maxRadius", value);
    return this;
  }
  getMaxRadius() {
    return this._maxRadius;
  }
  /**
   * Get/Set the includearrivals query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  includeArrivals(value) {
    doBoolGetterSetter(this, "includeArrivals", value);
    return this;
  }
  getIncludeArrivals() {
    return this._includeArrivals;
  }
  /**
   * Get/Set the includeallorigins query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  includeAllOrigins(value) {
    doBoolGetterSetter(this, "includeAllOrigins", value);
    return this;
  }
  getIncludeAllOrigins() {
    return this._includeAllOrigins;
  }
  /**
   * Get/Set the includeallmagnitudes query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  includeAllMagnitudes(value) {
    doBoolGetterSetter(this, "includeAllMagnitudes", value);
    return this;
  }
  getIncludeAllMagnitudes() {
    return this._includeAllMagnitudes;
  }
  /**
   * Get/Set the format query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  format(value) {
    doStringGetterSetter(this, "format", value);
    return this;
  }
  getFormat() {
    return this._format;
  }
  /**
   * Get/Set the limit query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  limit(value) {
    doIntGetterSetter(this, "limit", value);
    return this;
  }
  getLimit() {
    return this._limit;
  }
  /**
   * Get/Set the offset query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  offset(value) {
    doIntGetterSetter(this, "offset", value);
    return this;
  }
  getOffset() {
    return this._offset;
  }
  /**
   * Get/Set the orderby query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  orderBy(value) {
    doStringGetterSetter(this, "orderBy", value);
    return this;
  }
  getOrderBy() {
    return this._orderBy;
  }
  /**
   * Get/Set the catalog query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  catalog(value) {
    doStringGetterSetter(this, "catalog", value);
    return this;
  }
  getCatalog() {
    return this._catalog;
  }
  /**
   * Get/Set the contributor query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  contributor(value) {
    doStringGetterSetter(this, "contributor", value);
    return this;
  }
  getContributor() {
    return this._contributor;
  }
  latLonRegion(value) {
    if (value instanceof LatLonBox) {
      this._minLat = value.south;
      this._maxLat = value.north;
      this._minLon = value.west;
      this._maxLon = value.east;
      this._latitude = void 0;
      this._longitude = void 0;
      this._minRadius = void 0;
      this._maxRadius = void 0;
    } else if (value instanceof LatLonRadius) {
      this._latitude = value.latitude;
      this._longitude = value.longitude;
      this._minRadius = value.minRadius;
      this._maxRadius = value.maxRadius;
      this._minLat = void 0;
      this._maxLat = void 0;
      this._minLon = void 0;
      this._maxLon = void 0;
    } else if (!isDef(value)) {
      this._latitude = void 0;
      this._longitude = void 0;
      this._minRadius = void 0;
      this._maxRadius = void 0;
      this._minLat = void 0;
      this._maxLat = void 0;
      this._minLon = void 0;
      this._maxLon = void 0;
    } else {
      throw new Error(
        `value argument is optional or LatLonRegion, but was type ${typeof value}, '${stringify(value)}' `
      );
    }
    return this;
  }
  /**
   * Checks to see if any parameter that would limit the data
   * returned is set. This is a crude, coarse check to make sure
   * the client doesn't ask for EVERYTHING the server has.
   *
   * @returns true is some parameter is set
   */
  isSomeParameterSet() {
    return isDef(this._eventId) || isDef(this._startTime) || isDef(this._endTime) || isDef(this._minLat) || isDef(this._maxLat) || isDef(this._minLon) || isDef(this._maxLon) || isDef(this._latitude) || isDef(this._longitude) || isDef(this._minRadius) || isDef(this._maxRadius) || isDef(this._minDepth) || isDef(this._maxDepth) || isDef(this._limit) || isDef(this._minMag) || isDef(this._maxMag) || isDef(this._updatedAfter) || isDef(this._catalog) || isDef(this._contributor);
  }
  /**
   * Queries the remote service and parses the returned xml.
   *
   *  @returns Promise to an Array of Quake objects.
   */
  query() {
    return this.queryEventParameters().then((eventParameters) => {
      return eventParameters.eventList;
    });
  }
  queryEventParameters() {
    return this.queryRawXml().then((rawXml) => {
      return parseQuakeML(rawXml, this._host);
    });
  }
  /**
   * Queries the remote server, to get QuakeML xml.
   *
   * @returns xml Document
   */
  queryRawXml() {
    if (!this.isSomeParameterSet()) {
      throw new Error(
        "Must set some parameter to avoid asking for everything."
      );
    }
    const url = this.formURL();
    const fetchInit = defaultFetchInitObj(XML_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then((response) => {
      if (response.status === 200) {
        return response.text();
      } else if (response.status === 204 || isDef(this._nodata) && response.status === this._nodata) {
        return FAKE_EMPTY_XML2;
      } else {
        throw new Error(`Status not successful: ${response.status}`);
      }
    }).then(function(rawXmlText) {
      return new DOMParser().parseFromString(rawXmlText, XML_MIME);
    });
  }
  /**
   * Forms the basic URL to contact the web service, without any query paramters
   *
   * @returns the url
   */
  formBaseURL() {
    let colon = ":";
    if (!this._host || this._host === USGS_HOST) {
      this._host = USGS_HOST;
      this._protocol = "https:";
    }
    if (this._protocol.endsWith(colon)) {
      colon = "";
    }
    return this._protocol + colon + "//" + this._host + (this._port === 80 ? "" : ":" + stringify(this._port)) + "/fdsnws/event/" + this._specVersion;
  }
  /**
   * Forms the URL to get catalogs from the web service, without any query paramters
   *
   * @returns the url
   */
  formCatalogsURL() {
    return this.formBaseURL() + "/catalogs";
  }
  /**
   * Queries the remote web service to get known catalogs
   *
   * @returns Promise to Array of catalog names
   */
  queryCatalogs() {
    const url = this.formCatalogsURL();
    const fetchInit = defaultFetchInitObj(XML_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then((response) => {
      if (response.status === 200) {
        return response.text();
      } else {
        throw new Error(`Status not 200: ${response.status}`);
      }
    }).then((rawXmlText) => {
      return new DOMParser().parseFromString(rawXmlText, XML_MIME);
    }).then((rawXml) => {
      if (!rawXml) {
        throw new Error("raw xml from DOMParser is null.");
      }
      const top = rawXml.documentElement;
      if (!top) {
        throw new Error("documentElement in xml from DOMParser is null.");
      }
      const catalogArray = top.getElementsByTagName("Catalog");
      const out = [];
      if (catalogArray) {
        for (let i = 0; i < catalogArray.length; i++) {
          const item = catalogArray.item(i);
          if (item && isDef(item.textContent)) {
            out.push(item.textContent);
          }
        }
      }
      return out;
    });
  }
  /**
   * Forms the URL to get contributors from the web service, without any query paramters
   *
   * @returns the url
   */
  formContributorsURL() {
    return this.formBaseURL() + "/contributors";
  }
  /**
   * Queries the remote web service to get known contributors
   *
   * @returns Promise to Array of contributor names
   */
  queryContributors() {
    const url = this.formContributorsURL();
    const fetchInit = defaultFetchInitObj(XML_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then((response) => {
      if (response.status === 200) {
        return response.text();
      } else {
        throw new Error(`Status not 200: ${response.status}`);
      }
    }).then(function(rawXmlText) {
      return new DOMParser().parseFromString(rawXmlText, XML_MIME);
    }).then(function(rawXml) {
      const top = rawXml.documentElement;
      if (!top) {
        throw new Error("documentElement in xml from DOMParser is null.");
      }
      const contributorArray = top.getElementsByTagName("Contributor");
      const out = [];
      if (contributorArray) {
        for (let i = 0; i < contributorArray.length; i++) {
          const item = contributorArray.item(i);
          if (item && isDef(item.textContent)) {
            out.push(item.textContent);
          }
        }
      }
      return out;
    });
  }
  /**
   * Forms the URL to get version from the web service, without any query paramters
   *
   * @returns the url
   */
  formVersionURL() {
    return this.formBaseURL() + "/version";
  }
  /**
   * Queries the remote web service to get its version
   *
   * @returns Promise to version string
   */
  queryVersion() {
    const url = this.formVersionURL();
    const fetchInit = defaultFetchInitObj(TEXT_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then(
      (response) => {
        if (response.status === 200) {
          return response.text();
        } else {
          throw new Error(`Status not 200: ${response.status}`);
        }
      }
    );
  }
  /**
   * Form URL to query the remote web service, encoding the query parameters.
   *
   * @returns url
   */
  formURL() {
    let colon = ":";
    if (this._protocol.endsWith(colon)) {
      colon = "";
    }
    let url = this.formBaseURL() + "/query?";
    if (this._eventId) {
      url = url + makeParam("eventid", this._eventId);
    }
    if (this._startTime) {
      url = url + makeParam("starttime", toIsoWoZ(this._startTime));
    }
    if (this._endTime) {
      url = url + makeParam("endtime", toIsoWoZ(this._endTime));
    }
    if (isNumArg(this._minMag)) {
      url = url + makeParam("minmag", this._minMag);
    }
    if (isNumArg(this._maxMag)) {
      url = url + makeParam("maxmag", this._maxMag);
    }
    if (isStringArg(this._magnitudeType)) {
      url = url + makeParam("magnitudetype", this._magnitudeType);
    }
    if (isNumArg(this._minDepth)) {
      url = url + makeParam("mindepth", this._minDepth);
    }
    if (isNumArg(this._maxDepth)) {
      url = url + makeParam("maxdepth", this._maxDepth);
    }
    if (isNumArg(this._minLat)) {
      url = url + makeParam("minlat", this._minLat);
    }
    if (isNumArg(this._maxLat)) {
      url = url + makeParam("maxlat", this._maxLat);
    }
    if (isNumArg(this._minLon)) {
      url = url + makeParam("minlon", this._minLon);
    }
    if (isNumArg(this._maxLon)) {
      url = url + makeParam("maxlon", this._maxLon);
    }
    if (isNumArg(this._minRadius) || isNumArg(this._maxRadius)) {
      if (isNumArg(this._latitude) && isNumArg(this._longitude)) {
        url = url + makeParam("latitude", this._latitude) + makeParam("longitude", this._longitude);
        if (isNumArg(this._minRadius)) {
          url = url + makeParam("minradius", this._minRadius);
        }
        if (isNumArg(this._maxRadius)) {
          url = url + makeParam("maxradius", this._maxRadius);
        }
      } else {
        throw new Error(
          `Cannot use minRadius or maxRadius without latitude and longitude: lat=${this._latitude} lon= ${this._longitude}`
        );
      }
    }
    if (this._includeArrivals) {
      if (this._host !== USGS_HOST) {
        url = url + "includearrivals=true&";
      } else {
        if (this._eventId) {
        } else {
          throw new Error(
            "USGS host, earthquake.usgs.gov, does not support includearrivals parameter."
          );
        }
      }
    }
    if (isObject(this._updatedAfter)) {
      url = url + makeParam("updatedafter", this._updatedAfter);
    }
    if (isDef(this._includeAllOrigins)) {
      url = url + makeParam("includeallorigins", this._includeAllOrigins);
    }
    if (isDef(this._includeAllMagnitudes)) {
      url = url + makeParam("includeallmagnitudes", this._includeAllMagnitudes);
    }
    if (isStringArg(this._format)) {
      url = url + makeParam("format", this._format);
    }
    if (isNumArg(this._limit)) {
      url = url + makeParam("limit", this._limit);
    }
    if (isNumArg(this._offset)) {
      url = url + makeParam("offset", this._offset);
    }
    if (isStringArg(this._orderBy)) {
      url = url + makeParam("orderby", this._orderBy);
    }
    if (isStringArg(this._catalog)) {
      url = url + makeParam("catalog", this._catalog);
    }
    if (isStringArg(this._contributor)) {
      url = url + makeParam("contributor", this._contributor);
    }
    if (isDef(this._nodata)) {
      url = url + makeParam("nodata", this._nodata);
    }
    if (url.endsWith("&") || url.endsWith("?")) {
      url = url.substr(0, url.length - 1);
    }
    return url;
  }
};

// src/fdsnstation.ts
var fdsnstation_exports = {};
__export(fdsnstation_exports, {
  IRIS_HOST: () => IRIS_HOST,
  LEVELS: () => LEVELS,
  LEVEL_CHANNEL: () => LEVEL_CHANNEL,
  LEVEL_NETWORK: () => LEVEL_NETWORK,
  LEVEL_RESPONSE: () => LEVEL_RESPONSE,
  LEVEL_STATION: () => LEVEL_STATION,
  SERVICE_NAME: () => SERVICE_NAME4,
  SERVICE_VERSION: () => SERVICE_VERSION4,
  StationQuery: () => StationQuery
});
var LEVEL_NETWORK = "network";
var LEVEL_STATION = "station";
var LEVEL_CHANNEL = "channel";
var LEVEL_RESPONSE = "response";
var LEVELS = [
  LEVEL_NETWORK,
  LEVEL_STATION,
  LEVEL_CHANNEL,
  LEVEL_RESPONSE
];
var SERVICE_VERSION4 = 1;
var SERVICE_NAME4 = `fdsnws-station-${SERVICE_VERSION4}`;
var StationQuery = class extends FDSNCommon {
  /** @private */
  _networkCode;
  /** @private */
  _stationCode;
  /** @private */
  _locationCode;
  /** @private */
  _channelCode;
  /** @private */
  _startTime;
  /** @private */
  _endTime;
  /** @private */
  _startBefore;
  /** @private */
  _endBefore;
  /** @private */
  _startAfter;
  /** @private */
  _endAfter;
  /** @private */
  _minLat;
  /** @private */
  _maxLat;
  /** @private */
  _minLon;
  /** @private */
  _maxLon;
  /** @private */
  _latitude;
  /** @private */
  _longitude;
  /** @private */
  _minRadius;
  /** @private */
  _maxRadius;
  /** @private */
  _includeRestricted;
  /** @private */
  _includeAvailability;
  /** @private */
  _format;
  /** @private */
  _updatedAfter;
  /** @private */
  _matchTimeseries;
  /**
   * Construct a query
   *
   * @param host the host to connect to , defaults to service.iris.edu
   */
  constructor(host) {
    super(host);
  }
  /**
   * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
   * Setting this is probably a bad idea as the code may not be compatible with
   * the web service.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  specVersion(value) {
    doStringGetterSetter(this, "specVersion", value);
    return this;
  }
  getSpecVersion() {
    return this._specVersion;
  }
  /**
   * Gets/Sets the protocol, http or https. This should match the protocol
   * of the page loaded, but is autocalculated and generally need not be set.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  protocol(value) {
    doStringGetterSetter(this, "protocol", value);
    return this;
  }
  getProtocol() {
    return this._protocol;
  }
  /**
   * Gets/Sets the remote host to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  host(value) {
    doStringGetterSetter(this, "host", value);
    return this;
  }
  getHost() {
    return this._host;
  }
  /**
   * Gets/Sets the remote port to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  port(value) {
    doIntGetterSetter(this, "port", value);
    return this;
  }
  getPort() {
    return this._port;
  }
  /**
   * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
   * the status code when no matching data is found by the service.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  nodata(value) {
    doIntGetterSetter(this, "nodata", value);
    return this;
  }
  getNodata() {
    return this._nodata;
  }
  /**
   * Get/Set the network query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  networkCode(value) {
    doStringGetterSetter(this, "networkCode", value);
    return this;
  }
  getNetworkCode() {
    return this._networkCode;
  }
  /**
   * Get/Set the station query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  stationCode(value) {
    doStringGetterSetter(this, "stationCode", value);
    return this;
  }
  getStationCode() {
    return this._stationCode;
  }
  /**
   * Get/Set the location code query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  locationCode(value) {
    doStringGetterSetter(this, "locationCode", value);
    return this;
  }
  getLocationCode() {
    return this._locationCode;
  }
  /**
   * Get/Set the channel query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  channelCode(value) {
    doStringGetterSetter(this, "channelCode", value);
    return this;
  }
  getChannelCode() {
    return this._channelCode;
  }
  /**
   * Get/Set the starttime query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  startTime(value) {
    doMomentGetterSetter(this, "startTime", value);
    return this;
  }
  getStartTime() {
    return this._startTime;
  }
  /**
   * Get/Set the endtime query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  endTime(value) {
    doMomentGetterSetter(this, "endTime", value);
    return this;
  }
  getEndTime() {
    return this._endTime;
  }
  /**
   * Sets startTime and endTime using the given time window
   *
   * @param   se time window
   * @returns     this
   */
  timeRange(se) {
    this.startTime(validStartTime(se));
    this.endTime(validEndTime(se));
    return this;
  }
  /**
   * Get/Set the startbefore query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  startBefore(value) {
    doMomentGetterSetter(this, "startBefore", value);
    return this;
  }
  getStartBefore() {
    return this._startBefore;
  }
  /**
   * Get/Set the endbefore query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  endBefore(value) {
    doMomentGetterSetter(this, "endBefore", value);
    return this;
  }
  getEndBefore() {
    return this._endBefore;
  }
  /**
   * Get/Set the startafter query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  startAfter(value) {
    doMomentGetterSetter(this, "startAfter", value);
    return this;
  }
  getStartAfter() {
    return this._startAfter;
  }
  /**
   * Get/Set the endafter query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  endAfter(value) {
    doMomentGetterSetter(this, "endAfter", value);
    return this;
  }
  getEndAfter() {
    return this._endAfter;
  }
  /**
   * Get/Set the minlat query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minLat(value) {
    doFloatGetterSetter(this, "minLat", value);
    return this;
  }
  getMinLat() {
    return this._minLat;
  }
  /**
   * Get/Set the maxlon query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxLat(value) {
    doFloatGetterSetter(this, "maxLat", value);
    return this;
  }
  getMaxLat() {
    return this._maxLat;
  }
  /**
   * Get/Set the minlon query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minLon(value) {
    doFloatGetterSetter(this, "minLon", value);
    return this;
  }
  getMinLon() {
    return this._minLon;
  }
  /**
   * Get/Set the maxlon query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxLon(value) {
    doFloatGetterSetter(this, "maxLon", value);
    return this;
  }
  getMaxLon() {
    return this._maxLon;
  }
  /**
   * Get/Set the latitude query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  latitude(value) {
    doFloatGetterSetter(this, "latitude", value);
    return this;
  }
  getLatitude() {
    return this._latitude;
  }
  /**
   * Get/Set the longitude query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  longitude(value) {
    doFloatGetterSetter(this, "longitude", value);
    return this;
  }
  getLongitude() {
    return this._longitude;
  }
  /**
   * Get/Set the minradius query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minRadius(value) {
    doFloatGetterSetter(this, "minRadius", value);
    return this;
  }
  getMinRadius() {
    return this._minRadius;
  }
  /**
   * Get/Set the maxradius query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxRadius(value) {
    doFloatGetterSetter(this, "maxRadius", value);
    return this;
  }
  getMaxRadius() {
    return this._maxRadius;
  }
  /**
   * Get/Set the includerestricted query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  includeRestricted(value) {
    doBoolGetterSetter(this, "includeRestricted", value);
    return this;
  }
  getIncludeRestricted() {
    return this._includeRestricted;
  }
  /**
   * Get/Set the includeavailability query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  includeAvailability(value) {
    doBoolGetterSetter(this, "includeAvailability", value);
    return this;
  }
  getIncludeAvailability() {
    return this._includeAvailability;
  }
  /**
   * Get/Set the format query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  format(value) {
    doStringGetterSetter(this, "format", value);
    return this;
  }
  getFormat() {
    return this._format;
  }
  /**
   * Get/Set the updatedafter query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  updatedAfter(value) {
    doMomentGetterSetter(this, "updatedAfter", value);
    return this;
  }
  getUpdatedAfter() {
    return this._updatedAfter;
  }
  /**
   * Get/Set the matchtimeseries query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  matchTimeseries(value) {
    doBoolGetterSetter(this, "matchTimeseries", value);
    return this;
  }
  getMatchTimeseries() {
    return this._matchTimeseries;
  }
  latLonRegion(value) {
    if (value instanceof LatLonBox) {
      this._minLat = value.south;
      this._maxLat = value.north;
      this._minLon = value.west;
      this._maxLon = value.east;
      this._latitude = void 0;
      this._longitude = void 0;
      this._minRadius = void 0;
      this._maxRadius = void 0;
    } else if (value instanceof LatLonRadius) {
      this._latitude = value.latitude;
      this._longitude = value.longitude;
      this._minRadius = value.minRadius;
      this._maxRadius = value.maxRadius;
      this._minLat = void 0;
      this._maxLat = void 0;
      this._minLon = void 0;
      this._maxLon = void 0;
    } else if (!isDef(value)) {
      this._latitude = void 0;
      this._longitude = void 0;
      this._minRadius = void 0;
      this._maxRadius = void 0;
      this._minLat = void 0;
      this._maxLat = void 0;
      this._minLon = void 0;
      this._maxLon = void 0;
    } else {
      throw new Error(
        `value argument is optional or LatLonRegion, but was type ${typeof value}, '${stringify(value)}' `
      );
    }
    return this;
  }
  nslcCodes(channelId) {
    this.networkCode(channelId.networkCode);
    this.stationCode(channelId.stationCode);
    this.locationCode(channelId.locationCode);
    this.channelCode(channelId.channelCode);
    return this;
  }
  /**
   * Get/Set the timeout in seconds for the request. Default is 30.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  timeout(value) {
    doFloatGetterSetter(this, "timeoutSec", value);
    return this;
  }
  getTimeout() {
    return this._timeoutSec;
  }
  /**
   * Checks to see if any parameter that would limit the data
   * returned is set. This is a crude, coarse check to make sure
   * the client doesn't ask for EVERYTHING the server has.
   *
   * @returns true if some parameter set
   */
  isSomeParameterSet() {
    return isDef(this._networkCode) && this._networkCode.length > 0 && this._networkCode !== "*" || isDef(this._stationCode) && this._stationCode.length > 0 && this._stationCode !== "*" || isDef(this._locationCode) && this._locationCode.length > 0 && this._locationCode !== "*" || isDef(this._channelCode) && this._channelCode.length > 0 && this._channelCode !== "*" || isDef(this._startTime) || isDef(this._endTime) || isDef(this._startBefore) || isDef(this._endBefore) || isDef(this._startAfter) || isDef(this._endAfter) || isDef(this._minLat) || isDef(this._maxLat) || isDef(this._minLon) || isDef(this._maxLon) || isDef(this._latitude) || isDef(this._longitude) || isDef(this._minRadius) || isDef(this._maxRadius) || isDef(this._updatedAfter);
  }
  /**
   * Queries the remote web service for networks.
   *
   * @returns a Promise to an Array of Network objects.
   */
  queryNetworks() {
    return this.query(LEVEL_NETWORK);
  }
  /**
   * Queries the remote web service for stations. The stations
   * are contained within their respective Networks.
   *
   * @returns a Promise to an Array of Network objects.
   */
  queryStations() {
    return this.query(LEVEL_STATION);
  }
  /**
   * Queries the remote web service for channels. The Channels
   * are contained within their respective Stations which are in Networks.
   *
   * @returns a Promise to an Array of Network objects.
   */
  queryChannels() {
    return this.query(LEVEL_CHANNEL);
  }
  /**
   * Queries the remote web service for responses. The Responses
   * are contained within their respective Channels,
   * which are in Stations which are in Networks.
   *
   * @returns a Promise to an Array of Network objects.
   */
  queryResponses() {
    return this.query(LEVEL_RESPONSE);
  }
  /**
   * Queries the remote web service at the given level.
   *
   * @param level the level to query at, networ, station, channel or response.
   * @returns a Promise to an Array of Network objects.
   */
  query(level) {
    if (!LEVELS.includes(level)) {
      throw new Error("Unknown level: '" + level + "'");
    }
    return this.queryRawXml(level).then(function(rawXml) {
      return parseStationXml(rawXml);
    });
  }
  /**
   * Execute POST request for networks, using params defined in this, and with
   * channel lines of the form:
   *
   * NET STA LOC CHA STARTTIME ENDTIME
   *
   * Note that empty LOC should be encoded as dash-dash
   *
   * @param  postLines array of channel selection lines
   * @returns a Promise to an Array of Network objects.
   */
  postQueryNetworks(postLines) {
    return this.postQueryRawXml(LEVEL_NETWORK, postLines).then(
      function(rawXml) {
        return parseStationXml(rawXml);
      }
    );
  }
  /**
   * Execute POST request for stations, using params defined in this, and with
   * channel lines of the form:
   *
   * NET STA LOC CHA STARTTIME ENDTIME
   *
   * Note that empty LOC should be encoded as dash-dash
   *
   * @param  postLines array of channel selection lines
   * @returns a Promise to an Array of Network objects.
   */
  postQueryStations(postLines) {
    return this.postQueryRawXml(LEVEL_STATION, postLines).then(
      function(rawXml) {
        return parseStationXml(rawXml);
      }
    );
  }
  /**
   * Execute POST request for channels, using params defined in this, and with
   * channel lines of the form:
   *
   * NET STA LOC CHA STARTTIME ENDTIME
   *
   * Note that empty LOC should be encoded as dash-dash
   *
   * @param  postLines array of channel selection lines
   * @returns a Promise to an Array of Network objects.
   */
  postQueryChannels(postLines) {
    return this.postQueryRawXml(LEVEL_CHANNEL, postLines).then(
      function(rawXml) {
        return parseStationXml(rawXml);
      }
    );
  }
  /**
   * Execute POST request for responses, using params defined in this, and with
   * channel lines of the form:
   *
   * NET STA LOC CHA STARTTIME ENDTIME
   *
   * Note that empty LOC should be encoded as dash-dash
   *
   * @param  postLines array of channel selection lines
   * @returns a Promise to an Array of Network objects.
   */
  postQueryResponses(postLines) {
    return this.postQueryRawXml(LEVEL_RESPONSE, postLines).then(
      function(rawXml) {
        return parseStationXml(rawXml);
      }
    );
  }
  /**
   * Execute POST request using params defined in this, for given level, and with
   * channel lines of the form:
   *
   * NET STA LOC CHA STARTTIME ENDTIME
   *
   * Note that empty LOC should be encoded as dash-dash
   *
   * @param  level     level to request, one of network, station, channel, response
   * @param  postLines array of channel selection lines
   * @returns a Promise to an Array of Network objects.
   */
  postQuery(level, postLines) {
    if (!LEVELS.includes(level)) {
      throw new Error("Unknown level: '" + level + "'");
    }
    return this.postQueryRawXml(level, postLines).then(function(rawXml) {
      return parseStationXml(rawXml);
    });
  }
  /**
   * Queries the remote web service at the given level for raw xml.
   * Note that in the case of a nodata status code, xml that represents a
   * valid stationxml but with zero &lt;Network&gt; elements will be returned
   * as this simplifies parsing.
   *
   * @param level the level to query at, network, station, channel or response.
   * @returns a Promise to an xml Document.
   */
  queryRawXml(level) {
    return this.queryRawXmlText(level).then(function(rawXmlText) {
      return new DOMParser().parseFromString(rawXmlText, "text/xml");
    });
  }
  /**
   * Queries the remote web service at the given level for unparsed xml as text.
   * Note that in the case of a nodata status code, text that represents a
   * valid stationxml but with zero &lt;Network&gt; elements will be returned
   * as this simplifies parsing.
   *
   * @param level the level to query at, network, station, channel or response.
   * @returns a Promise to string.
   */
  queryRawXmlText(level) {
    if (!this.isSomeParameterSet()) {
      throw new Error(
        "Must set some parameter to avoid asking for everything."
      );
    }
    const url = this.formURL(level);
    const fetchInit = defaultFetchInitObj(XML_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then(
      (response) => {
        if (response.status === 200) {
          return response.text();
        } else if (response.status === 204 || isDef(this._nodata) && response.status === this._nodata) {
          return FAKE_EMPTY_XML;
        } else {
          throw new Error(`Status not successful: ${response.status}`);
        }
      }
    );
  }
  /**
   * Execute POST request using params defined in this, for given level, and with
   * channel lines of the form:
   *
   * NET STA LOC CHA STARTTIME ENDTIME
   *
   * Note that empty LOC should be encoded as dash-dash
   *
   * @param  level     level to request, one of network, station, channel, response
   * @param  postLines array of channel selection lines
   * @returns           string suitable for POST to fdsn station web service.
   */
  postQueryRawXml(level, postLines) {
    if (postLines.length === 0) {
      return Promise.resolve(FAKE_EMPTY_XML).then(function(rawXmlText) {
        return new DOMParser().parseFromString(rawXmlText, "text/xml");
      });
    } else {
      const fetchInit = defaultFetchInitObj(XML_MIME);
      fetchInit.method = "POST";
      fetchInit.body = this.createPostBody(level, postLines);
      return doFetchWithTimeout(
        this.formPostURL(),
        fetchInit,
        this._timeoutSec * 1e3
      ).then((response) => {
        if (response.status === 200) {
          return response.text();
        } else if (response.status === 204 || isDef(this._nodata) && response.status === this._nodata) {
          return FAKE_EMPTY_XML;
        } else {
          throw new Error(`Status not successful: ${response.status}`);
        }
      }).then(function(rawXmlText) {
        return new DOMParser().parseFromString(rawXmlText, "text/xml");
      });
    }
  }
  /**
   * Creates post body using params defined in this, for given level, and with
   * optional channel lines of the form:
   *
   * NET STA LOC CHA STARTTIME ENDTIME
   *
   * Note that empty LOC should be encoded as dash-dash
   *
   * @param  level     level to request, one of network, station, channel, response
   * @param  postLines optional array of channel selection lines
   * @returns           string suitable for POST to fdsn station web service.
   */
  createPostBody(level, postLines = []) {
    let out = "";
    if (!isStringArg(level)) {
      throw new Error(
        "level not specified, should be one of network, station, channel, response."
      );
    }
    out = out + makePostParam("level", level);
    if (isObject(this._startBefore)) {
      out = out + makePostParam("startbefore", toIsoWoZ(this._startBefore));
    }
    if (isObject(this._startAfter)) {
      out = out + makePostParam("startafter", toIsoWoZ(this._startAfter));
    }
    if (isObject(this._endBefore)) {
      out = out + makePostParam("endbefore", toIsoWoZ(this._endBefore));
    }
    if (isObject(this._endAfter)) {
      out = out + makePostParam("endafter", toIsoWoZ(this._endAfter));
    }
    if (isNumArg(this._minLat)) {
      out = out + makePostParam("minlat", this._minLat);
    }
    if (isNumArg(this._maxLat)) {
      out = out + makePostParam("maxlat", this._maxLat);
    }
    if (isNumArg(this._minLon)) {
      out = out + makePostParam("minlon", this._minLon);
    }
    if (isNumArg(this._maxLon)) {
      out = out + makePostParam("maxlon", this._maxLon);
    }
    if (isNumArg(this._latitude)) {
      out = out + makePostParam("lat", this._latitude);
    }
    if (isNumArg(this._longitude)) {
      out = out + makePostParam("lon", this._longitude);
    }
    if (isNumArg(this._minRadius)) {
      out = out + makePostParam("minradius", this._minRadius);
    }
    if (isNumArg(this._maxRadius)) {
      out = out + makePostParam("maxradius", this._maxRadius);
    }
    if (isDef(this._includeRestricted)) {
      out = out + makePostParam("includerestricted", this._includeRestricted);
    }
    if (isDef(this._includeAvailability)) {
      out = out + makePostParam("includeavailability", this._includeAvailability);
    }
    if (isObject(this._updatedAfter)) {
      out = out + makePostParam("updatedafter", toIsoWoZ(this._updatedAfter));
    }
    if (isDef(this._matchTimeseries)) {
      out = out + makePostParam("matchtimeseries", this._matchTimeseries);
    }
    if (isStringArg(this._format)) {
      out = out + makePostParam("format", this._format);
    }
    if (isNumArg(this._nodata)) {
      out = out + makePostParam("nodata", this._nodata);
    }
    postLines.forEach((line) => out = out + line.trim() + "\n");
    return out;
  }
  /**
   * Forms the URL to get version from the web service, without any query paramters
   *
   * @returns the url
   */
  formVersionURL() {
    return this.formBaseURL() + "/version";
  }
  /**
   * Queries the remote web service to get its version
   *
   * @returns Promise to version string
   */
  queryVersion() {
    const url = this.formVersionURL();
    const fetchInit = defaultFetchInitObj(TEXT_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then(
      (response) => {
        if (response.status === 200) {
          return response.text();
        } else {
          throw new Error(`Status not 200: ${response.status}`);
        }
      }
    );
  }
  /**
   * Forms the basic URL to contact the web service, without any query paramters
   *
   * @returns the url
   */
  formBaseURL() {
    let colon = ":";
    if (this._protocol.endsWith(colon)) {
      colon = "";
    }
    return this._protocol + colon + "//" + this._host + (this._port === 80 ? "" : ":" + String(this._port)) + "/fdsnws/station/" + this._specVersion;
  }
  formPostURL() {
    return this.formBaseURL() + "/query";
  }
  /**
   * Form URL to query the remote web service, encoding the query parameters.
   *
   * @param level network, station, channel or response
   * @returns url
   */
  formURL(level) {
    let url = this.formBaseURL() + "/query?";
    if (!isStringArg(level)) {
      throw new Error(
        "level not specified, should be one of network, station, channel, response."
      );
    }
    url = url + makeParam("level", level);
    if (isStringArg(this._networkCode) && this._networkCode.length > 0 && this._networkCode !== "*") {
      url = url + makeParam("net", this._networkCode);
    }
    if (isStringArg(this._stationCode) && this._stationCode.length > 0 && this._stationCode !== "*") {
      url = url + makeParam("sta", this._stationCode);
    }
    if (isStringArg(this._locationCode) && this._locationCode.length > 0 && this._locationCode !== "*") {
      url = url + makeParam("loc", this._locationCode);
    }
    if (isStringArg(this._channelCode) && this._channelCode.length > 0 && this._channelCode !== "*") {
      url = url + makeParam("cha", this._channelCode);
    }
    if (isObject(this._startTime)) {
      url = url + makeParam("starttime", toIsoWoZ(this._startTime));
    }
    if (isObject(this._endTime)) {
      url = url + makeParam("endtime", toIsoWoZ(this._endTime));
    }
    if (isObject(this._startBefore)) {
      url = url + makeParam("startbefore", toIsoWoZ(this._startBefore));
    }
    if (isObject(this._startAfter)) {
      url = url + makeParam("startafter", toIsoWoZ(this._startAfter));
    }
    if (isObject(this._endBefore)) {
      url = url + makeParam("endbefore", toIsoWoZ(this._endBefore));
    }
    if (isObject(this._endAfter)) {
      url = url + makeParam("endafter", toIsoWoZ(this._endAfter));
    }
    if (isNumArg(this._minLat)) {
      url = url + makeParam("minlat", this._minLat);
    }
    if (isNumArg(this._maxLat)) {
      url = url + makeParam("maxlat", this._maxLat);
    }
    if (isNumArg(this._minLon)) {
      url = url + makeParam("minlon", this._minLon);
    }
    if (isNumArg(this._maxLon)) {
      url = url + makeParam("maxlon", this._maxLon);
    }
    if (isNumArg(this._latitude)) {
      url = url + makeParam("lat", this._latitude);
    }
    if (isNumArg(this._longitude)) {
      url = url + makeParam("lon", this._longitude);
    }
    if (isNumArg(this._minRadius)) {
      url = url + makeParam("minradius", this._minRadius);
    }
    if (isNumArg(this._maxRadius)) {
      url = url + makeParam("maxradius", this._maxRadius);
    }
    if (isDef(this._includeRestricted)) {
      url = url + makeParam("includerestricted", this._includeRestricted);
    }
    if (isDef(this._includeAvailability)) {
      url = url + makeParam("includeavailability", this._includeAvailability);
    }
    if (isObject(this._updatedAfter)) {
      url = url + makeParam("updatedafter", toIsoWoZ(this._updatedAfter));
    }
    if (isDef(this._matchTimeseries)) {
      url = url + makeParam("matchtimeseries", this._matchTimeseries);
    }
    if (isStringArg(this._format)) {
      url = url + makeParam("format", this._format);
    }
    if (isNumArg(this._nodata)) {
      url = url + makeParam("nodata", this._nodata);
    }
    if (url.endsWith("&") || url.endsWith("?")) {
      url = url.substr(0, url.length - 1);
    }
    return url;
  }
};

// src/fdsndatacenters.ts
var FDSN_HOST = "www.fdsn.org";
var DataCentersQuery = class extends FDSNCommon {
  /** @private */
  _name;
  /** @private */
  _services;
  /** @private */
  _includedatasets;
  constructor(host) {
    if (!isNonEmptyStringArg(host)) {
      host = FDSN_HOST;
    }
    super(host);
  }
  /**
   * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
   * Setting this is probably a bad idea as the code may not be compatible with
   * the web service.
   *
   * @param value optional new value if setting
   * @returns the query when setting, the current value os services if no arguments
   */
  specVersion(value) {
    doStringGetterSetter(this, "specVersion", value);
    return this;
  }
  getSpecVersion() {
    return this._specVersion;
  }
  /**
   * Gets/Sets the protocol, http or https. This should match the protocol
   * of the page loaded, but is autocalculated and generally need not be set.
   *
   * @param value optional new value if setting
   * @returns the query when setting, the current value os services if no arguments
   */
  protocol(value) {
    doStringGetterSetter(this, "protocol", value);
    return this;
  }
  getProtocol() {
    return this._protocol;
  }
  /**
   * Gets/Sets the remote host to connect to. This defaults to
   * www.fdsn.org and generally should not be set.
   *
   * @param value optional new value if setting
   * @returns the query when setting, the current value os services if no arguments
   */
  host(value) {
    doStringGetterSetter(this, "host", value);
    return this;
  }
  getHost() {
    return this._host;
  }
  /**
   * Gets/Sets the remote port to connect to. This defaults to
   * the standard port for the protocol and generally should not be set.
   *
   * @param value optional new value if setting
   * @returns the query when setting, the current value os services if no arguments
   */
  port(value) {
    doIntGetterSetter(this, "port", value);
    return this;
  }
  getPort() {
    return this._port;
  }
  /**
   * limits results to the named data center, default is all data centers
   *
   * @param   value names to search for
   * @returns the query when setting, the current value os services if no arguments
   */
  name(value) {
    doStringGetterSetter(this, "name", value);
    return this;
  }
  getName() {
    return this._name;
  }
  /**
   * limits results to services that match the glob style pattern
   *
   * @param  value glob style pattern to match against
   * @returns the query when setting, the current value os services if no arguments
   */
  services(value) {
    doStringGetterSetter(this, "services", value);
    return this;
  }
  getServices() {
    return this._services;
  }
  /**
   * whether the results include detailed information about
   * the data sets offered by each center, default is false
   *
   * @param  value true to include datasets
   * @returns the query when setting, the current value os services if no arguments
   */
  includeDataSets(value) {
    doBoolGetterSetter(this, "includedatasets", value);
    return this;
  }
  getIncludeDataSets() {
    return this._includedatasets;
  }
  /**
   * Get/Set the timeout in seconds for the request. Default is 30.
   *
   * @param  value timeout seconds
   * @returns the query when setting, the current value os services if no arguments
   */
  timeout(value) {
    doFloatGetterSetter(this, "timeoutSec", value);
    return this;
  }
  getTimeout() {
    return this._timeoutSec;
  }
  /**
   * queries the fdsn registry web service, returning the result as a parsed json object.
   *
   * @returns Promise to the json object.
   */
  queryJson() {
    const url = this.formURL();
    const fetchInit = defaultFetchInitObj(JSON_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then((response) => {
      const contentType = response.headers.get("content-type");
      if (isNonEmptyStringArg(contentType) && contentType.includes(JSON_MIME)) {
        return response.json();
      }
      throw new TypeError(`Oops, we did not get JSON! ${contentType}`);
    }).then((jsonValue) => {
      if (isValidRootType2(jsonValue)) {
        return jsonValue;
      } else {
        throw new TypeError(`Oops, we did not get roottype JSON!`);
      }
    });
  }
  /**
   * queries the registry to find fdsn availability compatible web services within
   * a datacenter of the given name, optionally within the repository with
   * the repo name.
   *
   * @param   name     datacenter name
   * @param   repoName optional repository name
   * @returns           Promise to Array of fdsnavailability.AvailabilityQuery objects
   */
  findFdsnAvailability(name, repoName) {
    if (name && name.length > 0) {
      this.name(name);
    }
    this.services(SERVICE_NAME);
    return this.queryJson().then((json) => {
      const out = this.extractCompatibleServices(
        json,
        SERVICE_NAME,
        repoName
      );
      const sList = out.map((service) => {
        if ("url" in service && typeof service.url === "string") {
          const url = new URL(service.url);
          const q = new AvailabilityQuery(url.hostname);
          if (url.port && url.port.length > 0) {
            q.port(Number.parseInt(url.port));
          }
          return q;
        } else {
          return null;
        }
      });
      return sList.flatMap((f) => f ? [f] : []);
    });
  }
  /**
   * queries the registry to find fdsn dataselect compatible web services within
   * a datacenter of the given name, optionally within the repository with
   * the repo name.
   *
   * @param   name     datacenter name
   * @param   repoName optional repository name
   * @returns           Promise to Array of fdsndataselect.DataSelectQuery objects
   */
  findFdsnDataSelect(name, repoName) {
    if (name && name.length > 0) {
      this.name(name);
    }
    this.services(SERVICE_NAME2);
    return this.queryJson().then((json) => {
      const out = this.extractCompatibleServices(
        json,
        SERVICE_NAME2,
        repoName
      );
      const sList = out.map((service) => {
        if ("url" in service && typeof service.url === "string") {
          const url = new URL(service.url);
          const q = new DataSelectQuery(url.hostname);
          if (url.port && url.port.length > 0) {
            q.port(Number.parseInt(url.port));
          }
          return q;
        } else {
          return null;
        }
      });
      return sList.flatMap((f) => f ? [f] : []);
    });
  }
  /**
   * queries the registry to find a fdsn event compatible web services within
   * a datacenter of the given name, optionally within the repository with
   * the repo name.
   *
   * @param   dcname     datacenter name
   * @param   repoName optional repository name
   * @returns           Promise to Array of fdsnevent.EventQuery objects
   */
  findFdsnEvent(dcname, repoName) {
    if (dcname && dcname.length > 0) {
      this.name(dcname);
    }
    this.services(SERVICE_NAME3);
    return this.queryJson().then((json) => {
      const out = this.extractCompatibleServices(
        json,
        SERVICE_NAME3,
        repoName
      );
      const sList = out.map((service) => {
        if ("url" in service && typeof service.url === "string") {
          const url = new URL(service.url);
          const q = new EventQuery(url.hostname);
          if (url.port && url.port.length > 0) {
            q.port(Number.parseInt(url.port));
          }
          return q;
        } else {
          return null;
        }
      });
      return sList.flatMap((f) => f ? [f] : []);
    });
  }
  /**
   * queries the registry to find a fdsn station compatible web services within
   * a datacenter of the given name, optionally within the repository with
   * the repo name.
   *
   * @param   dcname     datacenter name
   * @param   repoName optional repository name
   * @returns           Promise to Array of fdsnstation.StationQuery objects
   */
  findFdsnStation(dcname, repoName) {
    if (dcname && dcname.length > 0) {
      this.name(dcname);
    }
    this.services(SERVICE_NAME4);
    return this.queryJson().then((json) => {
      const out = this.extractCompatibleServices(
        json,
        SERVICE_NAME4,
        repoName
      );
      const sList = out.map((service) => {
        if ("url" in service && typeof service.url === "string") {
          const url = new URL(service.url);
          const q = new StationQuery(url.hostname);
          if (url.port && url.port.length > 0) {
            q.port(Number.parseInt(url.port));
          }
          return q;
        } else {
          return null;
        }
      });
      return sList.flatMap((f) => f ? [f] : []);
    });
  }
  /**
   * Extracts services comaptible with the given service name, optionally within
   * the given repository, from the json.
   *
   * @param   json           json containing services
   * @param   compatibleName service name to be compatible with
   * @param   repoName       optional repository within the json to search
   * @returns                array of services found
   */
  extractCompatibleServices(json, compatibleName, repoName) {
    const out = [];
    json.datacenters.forEach((dc) => {
      dc.repositories.forEach((repo) => {
        if (!isDef(repoName) || repoName === repo.name) {
          repo.services.forEach((service) => {
            if (service.name === compatibleName || isDef(service.compatibleWith) && service.compatibleWith.includes(compatibleName)) {
              out.push(service);
            }
          });
        }
      });
    });
    return out;
  }
  /**
   * Forms the base of the url for accessing the datacenters service.
   *
   * @returns         URL
   */
  formBaseURL() {
    let colon = ":";
    if (this._protocol.endsWith(colon)) {
      colon = "";
    }
    return this._protocol + colon + "//" + this._host + (this._port === 80 ? "" : stringify(this._port)) + "/ws/datacenters/" + this._specVersion;
  }
  /**
   * Forms version url, not part of spec and so may not be supported.
   *
   * @returns         version
   */
  formVersionURL() {
    return this.formBaseURL() + "/version";
  }
  /**
   * Queries the remote web service to get its version
   *
   * @returns Promise to version string
   */
  queryVersion() {
    const url = this.formVersionURL();
    const fetchInit = defaultFetchInitObj(TEXT_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then(
      (response) => {
        if (response.status === 200) {
          return response.text();
        } else {
          throw new Error(`Status not 200: ${response.status}`);
        }
      }
    );
  }
  /**
   * forms a url to the fdsn registry based on the configured parameters.
   *
   * @returns the url
   */
  formURL() {
    const method = "query";
    let url = this.formBaseURL() + `/${method}?`;
    if (this._name) {
      url = url + makeParam("name", this._name);
    }
    if (this._services) {
      url = url + makeParam("services", this._services);
    }
    if (this._includedatasets) {
      url = url + makeParam("includedatasets", this._includedatasets);
    }
    if (url.endsWith("&") || url.endsWith("?")) {
      url = url.substr(0, url.length - 1);
    }
    return url;
  }
};
function isValidRootType2(jsonValue) {
  if (!jsonValue || typeof jsonValue !== "object") {
    throw new TypeError("json is not object");
  }
  const jsonObj = jsonValue;
  if (Array.isArray(jsonObj.datacenters) && typeof jsonObj.version === "number") {
    return true;
  } else {
    throw new TypeError("json is not valid for FDSN DataCenters");
  }
}

// src/filter.ts
var filter_exports = {};
__export(filter_exports, {
  BAND_PASS: () => BAND_PASS,
  HIGH_PASS: () => HIGH_PASS,
  LOW_PASS: () => LOW_PASS,
  add: () => add,
  amplitude: () => amplitude,
  applyFilter: () => applyFilter,
  createButterworth: () => createButterworth,
  createChebyshevI: () => createChebyshevI,
  createChebyshevII: () => createChebyshevII,
  differentiate: () => differentiate,
  envelope: () => envelope,
  gainCorrect: () => gainCorrect,
  getPassband: () => getPassband,
  hilbert: () => hilbert,
  integrate: () => integrate,
  lineFit: () => lineFit,
  mul: () => mul,
  rMean: () => rMean,
  removeTrend: () => removeTrend
});
import { Duration as Duration9 } from "luxon";
var BAND_PASS = "BANDPASS";
var LOW_PASS = "LOWPASS";
var HIGH_PASS = "HIGHPASS";
function amplitude(real, imag) {
  return Math.hypot(real, imag);
}
function rMean(seis) {
  if (seis instanceof Seismogram) {
    const meanVal = seis.mean();
    const rmeanSeismogram = new Seismogram(
      seis.segments.map((s) => {
        const demeanY = s.y.map(function(d) {
          return d - meanVal;
        });
        const out = s.cloneWithNewData(demeanY);
        return out;
      })
    );
    return rmeanSeismogram;
  } else {
    throw new Error("rMean arg not a Seismogram");
  }
}
function lineFit(seis, referenceTime) {
  if (seis.numPoints === 0) {
    throw new Error(`cannot lineFit a seismogram with no points`);
  }
  const rn = seis.numPoints;
  let sumx = 0;
  let sumy = 0;
  let sumxy = 0;
  let sumx2 = 0;
  let sumy2 = 0;
  referenceTime = referenceTime ? referenceTime : seis.start;
  const x1 = referenceTime.toMillis() / 1e3;
  seis.segments.forEach((seg) => {
    const seg_start_x = seg.start.toMillis() / 1e3 - x1;
    const dx = 1 / seg.sampleRate;
    const Y = seg.y;
    for (let i = 0; i < Y.length; i++) {
      const yi = Y[i];
      const xi = seg_start_x + dx * i;
      sumx = sumx + xi;
      sumy = sumy + yi;
      sumxy = sumxy + xi * yi;
      sumx2 = sumx2 + xi * xi;
      sumy2 = sumy2 + yi * yi;
    }
  });
  const d = rn * sumx2 - sumx * sumx;
  const b = d !== 0 ? (sumx2 * sumy - sumx * sumxy) / d : 0;
  const a = d !== 0 ? (rn * sumxy - sumx * sumy) / d : 0;
  const sig2 = (sumy2 + rn * b * b + a * a * sumx2 - 2 * b * sumy - 2 * a * sumxy + 2 * b * a * sumx) / (seis.numPoints - 1);
  const sig = Math.sqrt(sig2);
  const siga2 = rn * sig2 / d;
  const sigb2 = sig2 * sumx2 / d;
  const siga = Math.sqrt(siga2);
  const sigb = Math.sqrt(sigb2);
  let cc = (rn * sumxy - sumx * sumy) / Math.sqrt(d * (rn * sumy2 - sumy * sumy));
  cc = Math.abs(cc);
  return {
    slope: a,
    intercept: b,
    reference_time: referenceTime,
    sigma: sig,
    sigma_a: siga,
    sigma_b: sigb,
    correlation: cc
  };
}
function removeTrend(seis, fitLine) {
  if (seis instanceof Seismogram) {
    const linfit = fitLine ? fitLine : lineFit(seis);
    if (Number.isNaN(linfit.slope) || Number.isNaN(linfit.intercept)) {
      throw new Error(
        `Can't remove trend with NaN, slope: ${linfit.slope} int: ${linfit.intercept}`
      );
    }
    const ref_secs = linfit.reference_time.toMillis() / 1e3;
    const rtr_segments = seis.segments.map((seg) => {
      const start_secs = seg.start.toMillis() / 1e3;
      const start_offset = start_secs - ref_secs;
      const dx = 1 / seg.sampleRate;
      const rtr_y = seg.y.map((y, idx) => {
        const out = y - (start_offset + dx * idx) * linfit.slope - linfit.intercept;
        return out;
      });
      const rtr_seg = seg.cloneWithNewData(rtr_y);
      return rtr_seg;
    });
    return new Seismogram(rtr_segments);
  } else {
    throw new Error("removeTrend arg not a Seismogram");
  }
}
function gainCorrect(seis, instrumentSensitivity) {
  const gain = instrumentSensitivity.sensitivity;
  const out = mul(seis, 1 / gain);
  out.segments.forEach((s) => s.yUnit = instrumentSensitivity.inputUnits);
  return out;
}
function mul(seis, factor) {
  if (seis instanceof Seismogram) {
    const gainSeismogram = new Seismogram(
      seis.segments.map((s) => {
        let gainY;
        if (s.y instanceof Int32Array || s.y instanceof Float32Array) {
          gainY = Float32Array.from(s.y);
        } else {
          gainY = Float64Array.from(s.y);
        }
        gainY = gainY.map(function(d) {
          return d * factor;
        });
        const outS = s.cloneWithNewData(gainY);
        return outS;
      })
    );
    return gainSeismogram;
  } else {
    throw new Error(`Expected Seismogram but was ${typeof seis}`);
  }
}
function add(seis, factor) {
  if (seis instanceof Seismogram) {
    const gainSeismogram = new Seismogram(
      seis.segments.map((s) => {
        let gainY;
        if (s.y instanceof Int32Array || s.y instanceof Float32Array) {
          gainY = Float32Array.from(s.y);
        } else {
          gainY = Float64Array.from(s.y);
        }
        gainY = gainY.map(function(d) {
          return d + factor;
        });
        const outS = s.cloneWithNewData(gainY);
        return outS;
      })
    );
    return gainSeismogram;
  } else {
    throw new Error(`Expected Seismogram but was ${typeof seis}`);
  }
}
function getPassband(type) {
  if (type === LOW_PASS) {
    return LOWPASS;
  } else if (type === BAND_PASS) {
    return PassbandType.BANDPASS;
  } else if (type === HIGH_PASS) {
    return PassbandType.HIGHPASS;
  } else {
    throw new Error(`unknown pass band: ${type}`);
  }
}
function createButterworth(numPoles, passband, lowFreqCorner, highFreqCorner, delta) {
  const passbandtype = getPassband(passband);
  return new Butterworth(
    numPoles,
    passbandtype,
    lowFreqCorner,
    highFreqCorner,
    delta
  );
}
function createChebyshevI(numPoles, epsilon, passband, lowFreqCorner, highFreqCorner, delta) {
  const passbandtype = getPassband(passband);
  return new ChebyshevI(
    numPoles,
    epsilon,
    passbandtype,
    lowFreqCorner,
    highFreqCorner,
    delta
  );
}
function createChebyshevII(numPoles, epsilon, passband, lowFreqCorner, highFreqCorner, delta) {
  const passbandtype = getPassband(passband);
  return new ChebyshevII(
    numPoles,
    epsilon,
    passbandtype,
    lowFreqCorner,
    highFreqCorner,
    delta
  );
}
function applyFilter(iirFilter, seis) {
  if (Math.abs(iirFilter.getDelta() - seis.samplePeriod) / seis.samplePeriod > 1e-3) {
    throw new Error(
      `Filter, delta=${iirFilter.getDelta()}, has different delta from seis, ${1 / seis.sampleRate}`
    );
  }
  const filteredSegments = [];
  for (let i = 0; i < seis.segments.length; i++) {
    const outData = Float32Array.from(seis.segments[i].y);
    iirFilter.filterInPlace(outData);
    filteredSegments.push(seis.segments[i].cloneWithNewData(outData));
  }
  return new Seismogram(filteredSegments);
}
function envelope(seis) {
  if (seis.isContiguous()) {
    const seisY = seis.y;
    const s = hilbert(seis);
    const hilbertY = s.y;
    let outY;
    if (seis.y instanceof Int32Array || seis.y instanceof Float32Array) {
      outY = new Float32Array(seisY.length);
    } else {
      outY = new Float64Array(seisY.length);
    }
    for (let n = 0; n < seisY.length; n++) {
      outY[n] = Math.sqrt(hilbertY[n] * hilbertY[n] + seisY[n] * seisY[n]);
    }
    return seis.cloneWithNewData(outY);
  } else {
    throw new Error("Cannot take envelope of non-contiguous seismogram");
  }
}
function hilbert(seis, n, lowEdge, highEdge) {
  if (seis.isContiguous()) {
    let seisY;
    if (seis.y instanceof Float32Array) {
      seisY = seis.y;
    } else {
      seisY = Float32Array.from(seis.y);
    }
    if (!isDef(n)) {
      n = 10;
    }
    if (!isDef(lowEdge)) {
      lowEdge = 0.05;
    }
    if (!isDef(highEdge)) {
      highEdge = 0.95;
    }
    const hilbert2 = new CenteredHilbertTransform(n, lowEdge, highEdge);
    const coeff = hilbert2.getCoefficients();
    for (const c of coeff) {
      if (Number.isNaN(c)) {
        throw new Error(`Hilbert FIR coeff includes NaN: ${coeff.join()}`);
      }
    }
    const hilbertY = hilbert2.filter(seisY);
    const s = seis.cloneWithNewData(hilbertY);
    return s;
  } else {
    throw new Error("Cannot take hilbert of non-contiguous seismogram");
  }
}
function differentiate(seis) {
  if (seis instanceof Seismogram) {
    const diffSeismogram = new Seismogram(
      seis.segments.map((s) => {
        const origY = s.y;
        const sampRate = 1 * s.sampleRate;
        const diffY = new Float32Array(origY.length - 1);
        for (let i = 0; i < diffY.length; i++) {
          diffY[i] = (origY[i + 1] - origY[i]) * sampRate;
        }
        const out = s.cloneWithNewData(diffY);
        out.startTime = out.startTime.plus(
          Duration9.fromMillis(1e3 / out.sampleRate / 2)
        );
        out.yUnit = out.yUnit + "/s";
        return out;
      })
    );
    return diffSeismogram;
  } else {
    throw new Error("diff arg not a Seismogram");
  }
}
function integrate(seis, integrationConst = 0) {
  let prior = integrationConst;
  if (seis instanceof Seismogram) {
    const intSeismogram = new Seismogram(
      seis.segments.map((s) => {
        const origY = s.y;
        const sampPeriod = s.samplePeriod;
        const intY = new Float32Array(origY.length + 1);
        intY[0] = prior;
        for (let i = 1; i < intY.length; i++) {
          prior += sampPeriod * origY[i - 1];
          intY[i] = prior;
        }
        const out = s.cloneWithNewData(intY);
        out.startTime = out.startTime.minus(
          Duration9.fromMillis(1e3 / out.sampleRate / 2)
        );
        if (out.yUnit.endsWith("/s")) {
          out.yUnit = out.yUnit.slice(0, out.yUnit.length - 2);
        } else {
          out.yUnit += "s";
        }
        return out;
      })
    );
    return intSeismogram;
  } else {
    throw new Error("integrate arg not a Seismogram");
  }
}

// src/fft.ts
var fft_exports = {};
__export(fft_exports, {
  FFTResult: () => FFTResult,
  calcDFT: () => calcDFT,
  fftForward: () => fftForward,
  findPowerTwo: () => findPowerTwo,
  inverseDFT: () => inverseDFT
});
function fftForward(seis) {
  let sdd;
  if (seis instanceof Seismogram) {
    sdd = SeismogramDisplayData.fromSeismogram(seis);
  } else {
    sdd = seis;
  }
  if (isDef(sdd.seismogram)) {
    const seismogram = sdd.seismogram;
    if (seismogram.isContiguous()) {
      const result = FFTResult.createFromPackedFreq(
        calcDFT(seismogram.y),
        seismogram.numPoints,
        seismogram.sampleRate
      );
      result.seismogramDisplayData = sdd;
      return result;
    } else {
      throw new Error("Can only take FFT is seismogram is contiguous.");
    }
  } else {
    throw new Error("Can not take FFT is seismogram is null.");
  }
}
function calcDFT(timeseries) {
  let [N, log2N] = findPowerTwo(timeseries.length);
  if (N < 16) {
    log2N = 4;
    N = 16;
  }
  const dft = new RDFT(log2N);
  const inArray = new Float32Array(N);
  inArray.fill(0);
  for (let i = 0; i < timeseries.length; i++) {
    inArray[i] = timeseries[i];
  }
  const out = new Float32Array(N).fill(0);
  dft.evaluate(inArray, out);
  return out;
}
function inverseDFT(packedFreq, numPoints) {
  if (numPoints > packedFreq.length) {
    throw new Error(
      `Not enough points in packed freq array for ${numPoints}, only ${packedFreq.length}`
    );
  }
  let [N, log2N] = findPowerTwo(packedFreq.length);
  if (N < 16) {
    log2N = 4;
    N = 16;
  }
  if (N !== packedFreq.length) {
    throw new Error(`power of two check fails: ${N} ${packedFreq.length}`);
  }
  const dft = new RDFT(log2N);
  const out = new Float32Array(N).fill(0);
  dft.evaluateInverse(packedFreq, out);
  return out.slice(0, numPoints);
}
function findPowerTwo(fftlength) {
  let log2N = 1;
  let N = 2;
  while (N < fftlength) {
    log2N += 1;
    N = 2 * N;
  }
  return [N, log2N];
}
var FFTResult = class _FFTResult {
  /** number of points in the original timeseries, may be less than fft size. */
  origLength;
  packedFreq;
  /** number of points in the fft, usually power of 2 larger than origLength. */
  numPoints;
  /** sample rate of the original time series, maybe be null. */
  sampleRate;
  /** optional units of the original data for display purposes. */
  inputUnits;
  /**
   * optional reference to SeismogramDisplayData when calculated from a seismogram.
   *  Useful for creating title, etc.
   */
  seismogramDisplayData;
  constructor(origLength, sampleRate) {
    this.origLength = origLength;
    this.sampleRate = sampleRate;
    this.packedFreq = new Float32Array(0);
    this.numPoints = 0;
  }
  /**
   * Factory method to create FFTResult from packed array.
   *
   * @param   packedFreq real and imag values in packed format
   * @param   origLength length of the original timeseries before padding.
   * @param   sampleRate sample rate of original data
   * @returns            FFTResult
   */
  static createFromPackedFreq(packedFreq, origLength, sampleRate) {
    const fftResult = new _FFTResult(origLength, sampleRate);
    fftResult.packedFreq = packedFreq;
    fftResult.numPoints = packedFreq.length;
    const [N, log2N] = findPowerTwo(packedFreq.length);
    if (N < origLength) {
      throw new Error(
        `Not enough freq points, ${packedFreq.length}, for orig length of ${origLength}, must be > and power two, (${N}, ${log2N})`
      );
    }
    return fftResult;
  }
  /**
   * Factory method to create from array of complex numbers.
   *
   * @param   complexArray real and imag values as array of Complex objects.
   * @param   origLength   length of the original timeseries before padding.
   * @param   sampleRate sample rate of original data
   * @returns               FFTResult
   */
  static createFromComplex(complexArray, origLength, sampleRate) {
    const N = 2 * (complexArray.length - 1);
    const modFreq = new Float32Array(N).fill(0);
    modFreq[0] = complexArray[0].real();
    for (let i = 1; i < complexArray.length - 1; i++) {
      modFreq[i] = complexArray[i].real();
      modFreq[N - i] = complexArray[i].imag();
    }
    modFreq[N / 2] = complexArray[complexArray.length - 1].real();
    return _FFTResult.createFromPackedFreq(modFreq, origLength, sampleRate);
  }
  /**
   * Factory method to create from amp and phase arrays
   *
   * @param   amp        amplitude values
   * @param   phase      phase values
   * @param   origLength length of the original timeseries before padding.
   * @param   sampleRate sample rate of original data
   * @returns             FFTResult
   */
  static createFromAmpPhase(amp, phase, origLength, sampleRate) {
    if (amp.length !== phase.length) {
      throw new Error(
        `amp and phase must be same length: ${amp.length} ${phase.length}`
      );
    }
    const modComplex = new Array(amp.length);
    for (let i = 0; i < amp.length; i++) {
      modComplex[i] = complexFromPolar(amp[i], phase[i]);
    }
    return _FFTResult.createFromComplex(modComplex, origLength, sampleRate);
  }
  /**
   * The minimum non-zero frequency in the fft
   *
   * @returns fundamental frequency
   */
  get fundamentalFrequency() {
    if (this.sampleRate) {
      return this.sampleRate / this.numPoints;
    } else {
      throw new Error(
        "sample rate not set on FFTResult, needed to calc min frequency"
      );
    }
  }
  asComplex() {
    const complexArray = [];
    const L = this.packedFreq.length;
    complexArray.push(new Complex(this.packedFreq[0], 0));
    for (let i = 1; i < this.packedFreq.length / 2; i++) {
      const c = new Complex(this.packedFreq[i], this.packedFreq[L - i]);
      complexArray.push(c);
    }
    complexArray.push(new Complex(this.packedFreq[L / 2], 0));
    return complexArray;
  }
  asAmpPhase() {
    const amp = new Float32Array(1 + this.packedFreq.length / 2);
    const phase = new Float32Array(1 + this.packedFreq.length / 2);
    let c = new Complex(this.packedFreq[0], 0);
    amp[0] = c.abs();
    phase[0] = c.angle();
    const L = this.packedFreq.length;
    for (let i = 1; i < this.packedFreq.length / 2; i++) {
      c = new Complex(this.packedFreq[i], this.packedFreq[L - i]);
      amp[i] = c.abs();
      phase[i] = c.angle();
    }
    c = new Complex(this.packedFreq[L / 2], 0);
    amp[this.packedFreq.length / 2] = c.abs();
    phase[this.packedFreq.length / 2] = c.angle();
    return [amp, phase];
  }
  /**
   * calculates the inverse fft of this.packedFreq
   *
   * @returns time domain representation
   */
  fftInverse() {
    return inverseDFT(this.packedFreq, this.origLength);
  }
  frequencies() {
    const out = new Float32Array(this.numPoints / 2 + 1).fill(0);
    for (let i = 0; i < out.length; i++) {
      out[i] = i * this.fundamentalFrequency;
    }
    return out;
  }
  get numFrequencies() {
    return this.numPoints / 2 + 1;
  }
  get minFrequency() {
    return this.fundamentalFrequency;
  }
  get maxFrequency() {
    return this.sampleRate / 2;
  }
  amplitudes() {
    const [amp] = this.asAmpPhase();
    return amp;
  }
  phases() {
    const [, phase] = this.asAmpPhase();
    return phase;
  }
  clone() {
    const out = _FFTResult.createFromPackedFreq(
      this.packedFreq.slice(),
      this.origLength,
      this.sampleRate
    );
    out.seismogramDisplayData = this.seismogramDisplayData;
    return out;
  }
};

// src/mseedarchive.ts
var mseedarchive_exports = {};
__export(mseedarchive_exports, {
  Allowed_Flags: () => Allowed_Flags,
  MSeedArchive: () => MSeedArchive,
  loadDataRecords: () => loadDataRecords,
  maxSampleRate: () => maxSampleRate,
  maxTimeForRecord: () => maxTimeForRecord,
  minSampleRate: () => minSampleRate
});
import { Duration as Duration10, Interval as Interval5 } from "luxon";
var Allowed_Flags = ["n", "s", "l", "c", "Y", "j", "H"];
var MSeedArchive = class {
  _rootUrl;
  _pattern;
  _recordSize;
  _timeoutSec;
  constructor(rootUrl, pattern) {
    this._rootUrl = rootUrl;
    this._pattern = pattern;
    this._recordSize = 512;
    this._timeoutSec = 30;
    this.checkPattern(this._pattern);
  }
  get rootUrl() {
    return this._rootUrl;
  }
  get pattern() {
    return this._pattern;
  }
  get recordSize() {
    return this._recordSize;
  }
  /* eslint-disable jsdoc/no-multi-asterisks */
  /**
   * checks pattern for allowed flags as not all that are supported
   * by ringserver are supported here. Must only include:
   * * n network code, white space removed
   * * s station code, white space removed
   * * l  location code, white space removed
   * * c  channel code, white space removed
   * * Y  year, 4 digits
   * * j  day of year, 3 digits zero padded
   * * H  hour, 2 digits zero padded
   *
   * @param p mseed archive pattern string
   * @returns true if all flags are allowed
   */
  checkPattern(p) {
    const regexp = /%[a-zA-Z]/g;
    const allFlags = p.match(regexp);
    if (!allFlags) {
      return false;
    } else {
      for (const f of allFlags) {
        if (Allowed_Flags.indexOf(f.slice(1)) === -1) {
          throw new Error(`${f} not allowed in pattern`);
        }
      }
    }
    return true;
  }
  /* eslint-enable jsdoc/no-multi-asterisks */
  /**
   * Loads seismograms from the remote miniseed archive via
   * http(s). Files downloaded include all that might overlap
   * the given time window based on record size,
   * the minimum sample rate
   * for the channel band code and the given time window.
   *
   * @param   channelTimeList request channels and time windows
   * @returns Promise to the same SeismogramDisplayData array, but with seismograms populated
   */
  loadSeismograms(channelTimeList) {
    const promiseArray = channelTimeList.map((ct) => {
      if (isDef(ct.channel)) {
        const request = ct;
        const dataRecords = this.loadDataForChannel(
          ct.channel,
          ct.startTime,
          ct.endTime
        );
        return Promise.all([request, dataRecords]).then((pArray) => {
          return {
            request: pArray[0],
            dataRecords: pArray[1]
          };
        });
      } else if (isDef(ct.sourceId)) {
        const request = ct;
        const dataRecords = this.loadData(
          ct.sourceId.networkCode,
          ct.sourceId.stationCode,
          ct.sourceId.locationCode,
          ct.sourceId.formChannelCode(),
          ct.startTime,
          ct.endTime
        );
        return Promise.all([request, dataRecords]).then((pArray) => {
          return {
            request: pArray[0],
            dataRecords: pArray[1]
          };
        });
      } else {
        throw new Error("channel is missing in loadSeismograms ");
      }
    });
    return Promise.all(promiseArray).then((pArray) => {
      const out = [];
      pArray.forEach((p) => {
        const seisArray = seismogramPerChannel(p.dataRecords);
        for (const seis of seisArray) {
          const cutSeis = seis.cut(
            Interval5.fromDateTimes(p.request.startTime, p.request.endTime)
          );
          p.request.seismogram = cutSeis;
          out.push(p.request);
        }
      });
      return out;
    });
  }
  /**
   * Loads miniseed records based on channel and time window.
   *
   * @param   channel   channel to request
   * @param   startTime start time
   * @param   endTime   end time
   * @returns Promise to array of miniseed records
   */
  loadDataForChannel(channel, startTime, endTime) {
    return this.loadData(
      channel.station.network.networkCode,
      channel.station.stationCode,
      channel.locationCode,
      channel.channelCode,
      startTime,
      endTime,
      channel.sampleRate
    );
  }
  /**
   * Loads miniseed records based on string channel codes.
   *
   * @param   net        network code
   * @param   sta        station code
   * @param   loc        location code
   * @param   chan       channel code
   * @param   startTime  start time
   * @param   endTime    end time
   * @param   sampleRate known sample rate for this channel
   * @returns             Promise to array of miniseed records
   */
  loadData(net, sta, loc, chan, startTime, endTime, sampleRate) {
    const basePattern = this.fillBasePattern(net, sta, loc, chan);
    if (!isDef(sampleRate)) {
      sampleRate = minSampleRate(chan);
    }
    const recordTime = maxTimeForRecord(this._recordSize, sampleRate);
    let t = startTime.minus(recordTime);
    const urlList = [];
    while (t < endTime) {
      const url = this.rootUrl + "/" + this.fillTimePattern(basePattern, t);
      t = t.plus(Duration10.fromObject({ hour: 1 }));
      urlList.push(url);
    }
    if (t.plus(recordTime) > endTime) {
      const url = this.rootUrl + "/" + this.fillTimePattern(basePattern, t);
      urlList.push(url);
    }
    return loadDataRecords(urlList).then((dataRecords) => {
      if (dataRecords) {
        dataRecords = dataRecords.filter(
          (dr) => dr.header.endTime >= startTime && dr.header.startTime <= endTime
        );
      } else {
        dataRecords = [];
      }
      return dataRecords;
    });
  }
  /**
   * Replaces codes from channel in base pattern.
   *
   * @param   net  string to replace '%n'
   * @param   sta  string to replace '%s'
   * @param   loc  string to replace '%l'
   * @param   chan string to replace '%c'
   * @returns       new string with channel replacements made
   */
  fillBasePattern(net, sta, loc, chan) {
    return this.pattern.replace(/%n/g, net).replace(/%s/g, sta).replace(/%l/g, loc).replace(/%c/g, chan);
  }
  /**
   * Replaces time entries ( %Y, %j, %H ) in pattern.
   *
   * @param   basePattern pattern to replace in
   * @param   t           DateTime in time
   * @returns              string with time replaces
   */
  fillTimePattern(basePattern, t) {
    return basePattern.replace(/%Y/g, t.toFormat("yyyy")).replace(/%j/g, t.toFormat("ooo")).replace(/%H/g, t.toFormat("HH"));
  }
};
function loadDataRecords(urlList, fetchInit, timeoutSec2) {
  const promiseArray = urlList.map((url) => {
    return doFetchWithTimeout(url, fetchInit, timeoutSec2).then((fetchResponse) => {
      if (fetchResponse.ok) {
        if (fetchResponse.status === 200 || fetchResponse.status === 304) {
          return fetchResponse.arrayBuffer().then((ab) => {
            let dataRecords = [];
            if (ab.byteLength > 0) {
              dataRecords = parseDataRecords(ab);
            }
            return dataRecords;
          });
        } else if (fetchResponse.status === 404) {
          return [];
        } else {
          log(
            "no data: status=" + fetchResponse.status + " " + fetchResponse.url
          );
          return [];
        }
      } else if (fetchResponse.status === 404) {
        return [];
      } else {
        throw new Error(
          "fetch error: " + fetchResponse.ok + " " + fetchResponse.status + " " + fetchResponse.url
        );
      }
    }).catch((err) => {
      log("caught fetch err, continuing with empty: " + String(err));
      return [];
    });
  });
  return Promise.all(promiseArray).then((pArray) => {
    let dataRecords = [];
    pArray.forEach((p) => {
      dataRecords = dataRecords.concat(p);
    });
    return dataRecords;
  });
}
function maxSampleRate(chan) {
  const f = chan.slice(0, 1);
  switch (f) {
    case "F":
    case "G":
      return 5e3;
    case "D":
    case "C":
      return 1e3;
    case "E":
    case "H":
      return 250;
    case "S":
    case "B":
      return 80;
    case "M":
      return 10;
    case "L":
      return 1;
    case "V":
      return 0.1;
    case "U":
      return 0.01;
    case "R":
      return 1e-3;
    case "P":
      return 1e-4;
    case "Q":
      return 1e-6;
    default:
      throw new Error("Unknown band code " + f + " in " + chan);
  }
}
function minSampleRate(chan) {
  const f = chan.slice(0, 1);
  switch (f) {
    case "F":
    case "G":
      return 1e3;
    case "D":
    case "C":
      return 2500;
    case "E":
    case "H":
      return 80;
    case "S":
    case "B":
      return 10;
    case "M":
      return 1;
    case "L":
      return 1;
    case "V":
      return 0.1;
    case "U":
      return 0.01;
    case "R":
      return 1e-4;
    case "P":
      return 1e-5;
    case "Q":
      return 1e-7;
    default:
      throw new Error("Unknown band code " + f + " in " + chan);
  }
}
function maxTimeForRecord(recordSize, sampleRate) {
  return Duration10.fromMillis(1e3 * ((recordSize - 40) * 2) / sampleRate);
}

// src/ringserverweb.ts
var ringserverweb_exports = {};
__export(ringserverweb_exports, {
  DATALINK_PATH: () => DATALINK_PATH,
  IRIS_HOST: () => IRIS_HOST3,
  NslcWithType: () => NslcWithType,
  RingserverConnection: () => RingserverConnection,
  SEEDLINK_PATH: () => SEEDLINK_PATH,
  StreamStat: () => StreamStat2,
  nslcSplit: () => nslcSplit,
  stationsFromStreams: () => stationsFromStreams
});
import { DateTime as DateTime12 } from "luxon";
var SEEDLINK_PATH = "/seedlink";
var DATALINK_PATH = "/datalink";
var IRIS_HOST3 = "rtserve.iris.washington.edu";
var ORG = "Organization: ";
var RingserverConnection = class {
  /** @private */
  _host;
  /** @private */
  _port;
  /** @private */
  _prefix;
  /** @private */
  _timeoutSec;
  constructor(host, port) {
    const hostStr = isNonEmptyStringArg(host) ? host : IRIS_HOST3;
    if (hostStr.startsWith("http")) {
      const rs_url = new URL(hostStr);
      this._host = rs_url.hostname;
      this._port = parseInt(rs_url.port);
      if (!Number.isInteger(this._port)) {
        this._port = 80;
      }
      this._prefix = rs_url.pathname;
    } else {
      this._host = hostStr;
      this._port = 80;
      this._prefix = "";
    }
    if (isNumArg(port)) {
      this._port = port;
    }
    this._timeoutSec = 30;
  }
  /**
   * Gets/Sets the remote host to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  host(value) {
    doStringGetterSetter(this, "host", value);
    return this;
  }
  getHost() {
    return this._host;
  }
  /**
   * Gets/Sets the remote port to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  port(value) {
    doIntGetterSetter(this, "port", value);
    return this;
  }
  getPort() {
    return this._port;
  }
  /**
   * Get/Set the timeout in seconds for the request. Default is 30.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  timeout(value) {
    doFloatGetterSetter(this, "timeoutSec", value);
    return this;
  }
  getTimeout() {
    return this._timeoutSec;
  }
  /**
   * Pulls id result from ringserver /id parsed into an object with
   * 'ringserverVersion' and 'serverId' fields.
   *
   * @returns Result as a Promise.
   */
  pullId() {
    return this.pullRaw(this.formIdURL()).then((raw) => {
      const lines = raw.split("\n");
      let organization = lines[1];
      if (organization.startsWith(ORG)) {
        organization = organization.substring(ORG.length);
      }
      return {
        ringserverVersion: lines[0],
        serverId: organization
      };
    });
  }
  /**
   *  Use numeric level (1-6) to pull just IDs from ringserver.
   *  In a default ringserver,
   *  level=1 would return all networks like
   *  CO
   *  and level=2 would return all stations like
   *  CO_JSC
   *  If level is falsy/missing, level=6 is used.
   *  The optional matchPattern is a regular expression, so for example
   *  '.+_JSC_00_HH.' would get all HH? channels from any station name JSC.
   *
   * @param level 1-6
   * @param matchPattern regular expression to match
   * @returns Result as a Promise.
   */
  pullStreamIds(level, matchPattern) {
    let queryParams = "level=6";
    if (isNumArg(level) && level > 0) {
      queryParams = "level=" + level;
    }
    if (matchPattern) {
      queryParams = queryParams + "&match=" + matchPattern;
    }
    const url = this.formStreamIdsURL(queryParams);
    return this.pullRaw(url).then((raw) => {
      return raw.split("\n").filter((line) => line.length > 0);
    });
  }
  /**
   * Pull streams, including start and end times, from the ringserver.
   * The optional matchPattern is a regular expression, so for example
   * '.+_JSC_00_HH.' would get all HH? channels from any station name JSC.
   * Result returned is an Promise.
   *
   * @param matchPattern regular expression to match
   * @returns promise to object with 'accessTime' as a DateTime
   * and 'streams' as an array of StreamStat objects.
   */
  pullStreams(matchPattern) {
    let queryParams = "";
    if (matchPattern) {
      queryParams = "match=" + matchPattern;
    }
    const url = this.formStreamsURL(queryParams);
    return this.pullRaw(url).then((raw) => {
      const lines = raw.split("\n");
      const out = {
        accessTime: DateTime12.utc(),
        streams: []
      };
      for (const line of lines) {
        if (line.length === 0) {
          continue;
        }
        const vals = line.split(/\s+/);
        if (vals.length === 0) {
          continue;
        } else if (vals.length >= 2) {
          out.streams.push(new StreamStat2(vals[0], vals[1], vals[2]));
        } else {
          log("Bad /streams line, skipping: '" + line + "'");
        }
      }
      return out;
    });
  }
  /**
   * Utility method to pull raw result from ringserver url.
   * Result returned is an Promise.
   *
   * @param url the url
   * @returns promise to string result
   */
  pullRaw(url) {
    const fetchInit = defaultFetchInitObj(TEXT_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then(
      (response) => {
        if (response.status === 200) {
          return response.text();
        } else {
          throw new Error(`Status not 200: ${response.status}`);
        }
      }
    );
  }
  getDataLinkURL() {
    let proto = "ws:";
    if (checkProtocol() === "https:") {
      proto = "wss:";
    }
    return proto + "//" + this._host + (this._port === 80 ? "" : ":" + this._port) + this._prefix + DATALINK_PATH;
  }
  getSeedLinkURL() {
    let proto = "ws:";
    if (checkProtocol() === "https:") {
      proto = "wss:";
    }
    return proto + "//" + this._host + (this._port === 80 ? "" : ":" + this._port) + this._prefix + SEEDLINK_PATH;
  }
  /**
   * Forms base url from protocol, host and port.
   *
   * @returns the string url
   */
  formBaseURL() {
    if (this._port === 0) {
      this._port = 80;
    }
    return checkProtocol() + "//" + this._host + (this._port === 80 ? "" : ":" + this._port) + this._prefix;
  }
  /**
   * Forms the ringserver id url.
   *
   * @returns the id url
   */
  formIdURL() {
    return this.formBaseURL() + "/id";
  }
  /**
   * Forms the ringserver streams url using the query parameters.
   *
   * @param queryParams optional string of query parameters
   * @returns the streams url
   */
  formStreamsURL(queryParams) {
    return this.formBaseURL() + "/streams" + (isNonEmptyStringArg(queryParams) && queryParams.length > 0 ? "?" + queryParams : "");
  }
  /**
   * Forms the ringserver stream ids url using the query parameters.
   *
   * @param queryParams optional string of query parameters
   * @returns the stream ids url
   */
  formStreamIdsURL(queryParams) {
    return this.formBaseURL() + "/streamids" + (queryParams && queryParams.length > 0 ? "?" + queryParams : "");
  }
};
function stationsFromStreams(streams) {
  const out = /* @__PURE__ */ new Map();
  for (const s of streams) {
    const nslc_type = nslcSplit(s.key);
    const nslc = nslc_type.nslc;
    const staKey = nslc.networkCode + "." + nslc.stationCode;
    let stat = out.get(staKey);
    if (!isDef(stat)) {
      stat = new StreamStat2(staKey, s.startRaw, s.endRaw);
      out.set(staKey, stat);
    } else {
      if (stat.start > s.start) {
        stat.start = s.start;
        stat.startRaw = s.startRaw;
      }
      if (stat.end < s.end) {
        stat.end = s.end;
        stat.endRaw = s.endRaw;
      }
    }
  }
  return Array.from(out.values());
}
var NslcWithType = class {
  type;
  nslc;
  constructor(type, nslc) {
    this.type = type;
    this.nslc = nslc;
  }
};
function nslcSplit(id) {
  const split = id.split("/");
  const nslc = split[0].split("_");
  if (nslc.length === 4) {
    return new NslcWithType(
      split[1],
      new NslcId(nslc[0], nslc[1], nslc[2], nslc[3])
    );
  } else {
    throw new Error("tried to split, did not find 4 elements in array: " + id);
  }
}
var StreamStat2 = class {
  key;
  startRaw;
  endRaw;
  start;
  end;
  constructor(key, start, end) {
    this.key = key;
    this.startRaw = start;
    this.endRaw = end;
    if (this.startRaw.indexOf(".") !== -1 && this.startRaw.indexOf(".") < this.startRaw.length - 4) {
      this.startRaw = this.startRaw.substring(
        0,
        this.startRaw.indexOf(".") + 4
      );
    }
    if (this.startRaw.charAt(this.startRaw.length - 1) !== "Z") {
      this.startRaw = this.startRaw + "Z";
    }
    if (this.endRaw.indexOf(".") !== -1 && this.endRaw.indexOf(".") < this.endRaw.length - 4) {
      this.endRaw = this.endRaw.substring(0, this.endRaw.indexOf(".") + 4);
    }
    if (this.endRaw.charAt(this.endRaw.length - 1) !== "Z") {
      this.endRaw = this.endRaw + "Z";
    }
    this.start = isoToDateTime(this.startRaw);
    this.end = isoToDateTime(this.endRaw);
    this.startRaw = start;
    this.endRaw = end;
  }
  /**
   * Calculates latency time difference between last packet and current time.
   *
   * @param accessTime time latency is calculated relative to
   * @returns latency
   */
  calcLatency(accessTime) {
    if (!accessTime) accessTime = DateTime12.utc();
    return this.end.diff(accessTime);
  }
};

// src/sacpolezero.ts
var sacpolezero_exports = {};
__export(sacpolezero_exports, {
  SacPoleZero: () => SacPoleZero,
  geomspace: () => geomspace,
  linspace: () => linspace,
  logspace: () => logspace
});
var SacPoleZero = class _SacPoleZero {
  /**
   * Complex poles
   */
  poles;
  /**
   * Complex zeros
   */
  zeros;
  /**
   * Scalar overall gain
   */
  constant;
  /** number of zeros added to convert to displacement, for debugging */
  gamma;
  /** hertz/radian factor, for debugging */
  mulFactor;
  /** sensitivity accounting for gamma, for debugging */
  sd;
  /** normalization factor for poles and zeros accounting for gamma, for debugging */
  A0;
  constructor(poles, zeros, constant) {
    this.poles = poles;
    this.zeros = zeros;
    this.constant = constant;
    this.mulFactor = 1;
    this.sd = 1;
    this.A0 = 1;
  }
  toString() {
    const s = ["sacPoleZero:"];
    s.push("ZEROS " + this.zeros.length);
    for (let i = 0; i < this.zeros.length; i++) {
      s.push("    " + this.zeros[i].real() + " " + this.zeros[i].imag());
    }
    s.push("POLES " + this.poles.length);
    for (let i = 0; i < this.poles.length; i++) {
      s.push("    " + this.poles[i].real() + " " + this.poles[i].imag());
    }
    s.push("CONSTANT " + this.constant);
    if (isNumArg(this.gamma) && isNumArg(this.mulFactor) && isNumArg(this.sd) && isNumArg(this.A0)) {
      s.push("*    gamma: " + this.gamma);
      s.push("*    mulFactor: " + this.mulFactor);
      s.push("*    sd: " + this.sd);
      s.push("*    A0: " + this.A0);
    }
    return s.join("\n");
  }
  evalPoleZeroInverse(freq) {
    const s = new Complex(0, 2 * Math.PI * freq);
    let zeroOut = new Complex(1, 0);
    let poleOut = new Complex(1, 0);
    for (let i = 0; i < this.poles.length; i++) {
      poleOut = poleOut.timesComplex(s.minusComplex(this.poles[i]));
    }
    for (let i = 0; i < this.zeros.length; i++) {
      if (s.real() === this.zeros[i].real() && s.imag() === this.zeros[i].imag()) {
        return new Complex(0, 0);
      }
      zeroOut = zeroOut.timesComplex(s.minusComplex(this.zeros[i]));
    }
    const out = poleOut.overComplex(zeroOut);
    return out.overReal(this.constant);
  }
  trimZeros(gamma) {
    for (let i = 0; i < gamma; i++) {
      const z = this.zeros[this.zeros.length - 1 - i];
      if (z.real() !== 0 || z.imag() !== 0) {
        throw new Error(
          `Attempt to trim ${gamma} zeros from SacPoleZero, but zero isn't 0+i0: ${stringify(z)}`
        );
      }
    }
    let trimmedZeros = this.zeros.slice().reverse();
    for (let i = 0; i < gamma; i++) {
      const idx = trimmedZeros.findIndex(
        (d) => d.real() === 0 && d.imag() === 0
      );
      trimmedZeros.splice(idx, 1);
    }
    trimmedZeros = trimmedZeros.reverse();
    this.zeros = trimmedZeros;
  }
  toText() {
    const s = [];
    s.push("ZEROS " + this.zeros.length);
    for (let i = 0; i < this.zeros.length; i++) {
      s.push("    " + this.zeros[i].real() + " " + this.zeros[i].imag());
    }
    s.push("POLES " + this.poles.length);
    for (let i = 0; i < this.poles.length; i++) {
      s.push("    " + this.poles[i].real() + " " + this.poles[i].imag());
    }
    s.push("CONSTANT " + this.constant);
    return s.join("\n");
  }
  /**
   * Caclulates the frequency response from the given poles and zeros.
   *
   * @param freqs frequencies to compute
   * @returns  frequency response
   */
  calcForDisplay(freqs) {
    const out = freqs.map((freq) => {
      let respAtS = this.evalPoleZeroInverse(freq);
      respAtS = new Complex(1, 0).overComplex(respAtS);
      return respAtS;
    });
    return out;
  }
  /**
   * Parses a string in sac polezero format into a SacPoleZero.
   *
   * @param data string to parse
   * @returns SacPoleZero instance
   */
  static parse(data) {
    const pz = {
      zeros: Array(0),
      poles: Array(0),
      constant: 1
    };
    const lines = data.split("\n");
    let numZeros = 0;
    let numPoles = 0;
    let i = 0;
    while (i < lines.length) {
      let l = lines[i];
      let items = l.trim().split(/ +/);
      if (items[0] === "ZEROS") {
        numZeros = parseInt(items[1]);
        i++;
        l = lines[i];
        items = l.trim().split(/ +/);
        while (i < lines.length && pz.zeros.length < numZeros) {
          if (items[0] === "POLES") {
            for (let z = pz.zeros.length; z < numZeros; z++) {
              pz.zeros.push(new Complex(0, 0));
            }
            break;
          } else {
            const real = parseFloat(items[0]);
            const imag = parseFloat(items[1]);
            pz.zeros.push(new Complex(real, imag));
          }
          i++;
          l = lines[i];
          items = l.trim().split(/ +/);
        }
      }
      if (items[0] === "POLES") {
        numPoles = parseInt(items[1]);
        i++;
        l = lines[i];
        items = l.trim().split(/ +/);
        while (i < lines.length && pz.poles.length < numPoles) {
          if (items[0] === "CONSTANT") {
            for (let z = pz.poles.length; z < numPoles; z++) {
              pz.poles.push(new Complex(0, 0));
            }
            break;
          } else {
            const real = parseFloat(items[0]);
            const imag = parseFloat(items[1]);
            pz.poles.push(new Complex(real, imag));
          }
          i++;
          l = lines[i];
          items = l.trim().split(/ +/);
        }
      }
      if (items[0] === "CONSTANT") {
        pz.constant = parseFloat(items[1]);
      }
      i++;
    }
    return new _SacPoleZero(pz.poles, pz.zeros, pz.constant);
  }
};
function geomspace(start, stop, num) {
  const log_start = Math.log(start);
  const log_stop = Math.log(stop);
  return logspace(log_start, log_stop, num);
}
function logspace(start, stop, num) {
  return linspace(start, stop, num).map((n) => Math.pow(10, n));
}
function linspace(start, stop, num) {
  const delta = (stop - start) / (num - 1);
  const out = [];
  for (let i = 0; i < num; i++) {
    out.push(start + i * delta);
  }
  return out;
}

// src/seedlink.ts
var seedlink_exports = {};
__export(seedlink_exports, {
  SEEDLINK_PROTOCOL: () => SEEDLINK_PROTOCOL,
  SeedlinkConnection: () => SeedlinkConnection
});
var SEEDLINK_PROTOCOL = "SeedLink3.1";
var SeedlinkConnection = class {
  url;
  requestConfig;
  receiveMiniseedFn;
  errorHandler;
  closeFn;
  webSocket;
  command;
  constructor(url, requestConfig, receiveMiniseedFn, errorHandler) {
    this.url = url;
    this.requestConfig = requestConfig;
    this.receiveMiniseedFn = receiveMiniseedFn;
    this.errorHandler = errorHandler;
    this.closeFn = null;
    this.command = "DATA";
    this.webSocket = null;
  }
  setTimeCommand(startTime) {
    this.command = "TIME " + startTime.toFormat("yyyy,LL,dd,HH,mm,ss");
  }
  setOnError(errorHandler) {
    this.errorHandler = errorHandler;
  }
  setOnClose(closeFn) {
    this.closeFn = closeFn;
  }
  connect() {
    if (this.webSocket) {
      this.webSocket.close();
      this.webSocket = null;
    }
    try {
      const webSocket = new WebSocket(this.url, SEEDLINK_PROTOCOL);
      this.webSocket = webSocket;
      webSocket.binaryType = "arraybuffer";
      webSocket.onopen = () => {
        this.sendHello().then(() => {
          return this.sendCmdArray(this.requestConfig);
        }).then(() => {
          return this.sendCmdArray([this.command]);
        }).then((val) => {
          webSocket.onmessage = (event) => {
            this.handle(event);
          };
          webSocket.send("END\r");
          return val;
        }).catch((err) => {
          this.close();
          const insureErr = err instanceof Error ? err : new Error(stringify(err));
          if (this.errorHandler) {
            this.errorHandler(insureErr);
          } else {
            throw insureErr;
          }
        });
      };
      webSocket.onerror = (event) => {
        this.handleError(new Error("" + stringify(event)));
        this.close();
      };
      webSocket.onclose = (closeEvent) => {
        if (this.closeFn) {
          this.closeFn(closeEvent);
        }
        if (this.webSocket) {
          this.webSocket = null;
        }
      };
    } catch (err) {
      if (this.errorHandler) {
        this.errorHandler(toError(err));
      } else {
        throw err;
      }
    }
  }
  close() {
    if (this.webSocket) {
      this.webSocket.close();
    }
  }
  handle(event) {
    if (event.data instanceof ArrayBuffer) {
      const data = event.data;
      if (data.byteLength < 64) {
      } else {
        this.handleMiniseed(data);
      }
    } else {
      this.handleError(new Error("Unknown message type" + String(event)));
    }
  }
  handleMiniseed(data) {
    try {
      if (data.byteLength < 64) {
        this.errorHandler(
          new Error(
            "message too small to be miniseed: " + data.byteLength + " " + dataViewToString(new DataView(data))
          )
        );
        return;
      }
      const slHeader = new DataView(data, 0, 8);
      if (slHeader.getInt8(0) === 83 && slHeader.getInt8(1) === 76) {
        let seqStr = "";
        for (let i = 0; i < 6; i++) {
          seqStr = seqStr + String.fromCharCode(slHeader.getInt8(2 + i));
        }
        const dataView = new DataView(data, 8, data.byteLength - 8);
        const out = {
          rawsequence: seqStr,
          sequence: parseInt(seqStr, 16),
          miniseed: parseSingleDataRecord(dataView)
        };
        this.receiveMiniseedFn(out);
      } else {
        throw new Error(
          "Not a seedlink packet, no starting SL: " + slHeader.getInt8(0) + " " + slHeader.getInt8(1)
        );
      }
    } catch (e) {
      this.errorHandler(toError(e));
      this.close();
    }
  }
  isConnected() {
    return this.webSocket !== null;
  }
  /**
   * Sends initial HELLO to server and waits for response.
   *
   * @returns            Promise that resolves to the response from the server.
   */
  sendHello() {
    const webSocket = this.webSocket;
    const promise = new Promise(function(resolve, reject) {
      if (webSocket) {
        webSocket.onmessage = function(event) {
          if (event.data instanceof ArrayBuffer) {
            const data = event.data;
            const replyMsg = dataViewToString(new DataView(data));
            const lines = replyMsg.trim().split("\r");
            if (lines.length === 2) {
              resolve([lines[0], lines[1]]);
            } else {
              reject("not 2 lines: " + replyMsg);
            }
          } else {
            reject("event.data not ArrayBuffer?");
          }
        };
        webSocket.send("HELLO\r");
      } else {
        reject("webSocket has been closed");
      }
    });
    return promise;
  }
  /**
   * Sends an array of commands, each as a Promise waiting for the 'OK' response
   * before sending the next.
   *
   * @param   cmd array of commands to send
   * @returns      Promise that resolves to the 'OK' returned by the last
   *   command if successful, or rejects on the first failure.
   */
  sendCmdArray(cmd) {
    return cmd.reduce((accum, next) => {
      return accum.then(() => {
        return this.createCmdPromise(next);
      });
    }, Promise.resolve("OK"));
  }
  /**
   * creates a Promise that sends a command and waits resolved with the result.
   *
   * @param   mycmd command string to send.
   * @returns        Promise that resolves to the reply from the server.
   */
  createCmdPromise(mycmd) {
    const webSocket = this.webSocket;
    const promise = new Promise(function(resolve, reject) {
      if (webSocket) {
        webSocket.onmessage = function(event) {
          if (event.data instanceof ArrayBuffer) {
            const data = event.data;
            const replyMsg = dataViewToString(new DataView(data)).trim();
            if (replyMsg === "OK") {
              resolve(replyMsg);
            } else {
              reject("msg not OK: " + replyMsg);
            }
          } else {
            reject("event.data not ArrayBuffer?");
          }
        };
        webSocket.send(mycmd + "\r\n");
      } else {
        reject("webSocket has been closed");
      }
    });
    return promise;
  }
  /**
   * handle errors that arise
   *
   * @private
   * @param   error the error
   */
  handleError(error) {
    if (this.errorHandler) {
      this.errorHandler(error);
    } else {
      log("seedlink handleError: " + error.message);
    }
  }
};

// src/seedlink4.ts
var seedlink4_exports = {};
__export(seedlink4_exports, {
  AUTH_COMMAND: () => AUTH_COMMAND,
  BYE_COMMAND: () => BYE_COMMAND,
  DATA_COMMAND: () => DATA_COMMAND,
  ENDFETCH_COMMAND: () => ENDFETCH_COMMAND,
  END_COMMAND: () => END_COMMAND,
  HELLO_COMMAND: () => HELLO_COMMAND,
  INFO_COMMAND: () => INFO_COMMAND,
  MINISEED_2_FORMAT: () => MINISEED_2_FORMAT,
  MINISEED_3_FORMAT: () => MINISEED_3_FORMAT,
  SEEDLINK4_PROTOCOL: () => SEEDLINK4_PROTOCOL,
  SELECT_COMMAND: () => SELECT_COMMAND,
  SEPacket: () => SEPacket,
  SE_PACKET_SIGNATURE: () => SE_PACKET_SIGNATURE,
  SLPROTO_COMMAND: () => SLPROTO_COMMAND,
  STATION_COMMAND: () => STATION_COMMAND,
  SeedlinkConnection: () => SeedlinkConnection2,
  USERAGENT_COMMAND: () => USERAGENT_COMMAND
});
var SEEDLINK4_PROTOCOL = "SLPROTO4.0";
var MINISEED_2_FORMAT = "2";
var MINISEED_3_FORMAT = "3";
var SE_PACKET_SIGNATURE = "SE";
var END_COMMAND = "END";
var ENDFETCH_COMMAND = "ENDFETCH";
var AUTH_COMMAND = "AUTH";
var BYE_COMMAND = "BYE";
var DATA_COMMAND = "DATA";
var HELLO_COMMAND = "HELLO";
var INFO_COMMAND = "INFO";
var SELECT_COMMAND = "SELECT";
var SLPROTO_COMMAND = "SLPROTO";
var STATION_COMMAND = "STATION";
var USERAGENT_COMMAND = "USERAGENT";
var useLittleEndian = true;
var SEPacket = class _SEPacket {
  dataFormat;
  dataSubformat;
  payloadLength;
  sequence;
  stationId;
  _miniseed;
  _mseed3;
  _json;
  _rawPayload;
  constructor(dataFormat, dataSubformat, payloadLength, sequence, stationId) {
    this.dataFormat = dataFormat;
    this.dataSubformat = dataSubformat;
    this.payloadLength = payloadLength;
    this.sequence = sequence;
    this.stationId = stationId;
    this._miniseed = null;
    this._mseed3 = null;
    this._json = null;
    this._rawPayload = null;
  }
  static parse(data) {
    let sePacket;
    if (data.byteLength < 16) {
      throw new Error(
        "message too small to be SE packet: " + data.byteLength + " " + dataViewToString(new DataView(data))
      );
    }
    const slHeader = new DataView(data, 0, 16);
    const sig = String.fromCharCode(slHeader.getUint8(0), slHeader.getUint8(1));
    if (sig === SE_PACKET_SIGNATURE) {
      const dataFormat = slHeader.getUint8(2);
      const dataSubformat = slHeader.getUint8(3);
      const payloadLength = slHeader.getUint32(4, useLittleEndian);
      const sequenceNum = slHeader.getBigUint64(8, useLittleEndian);
      const stationIdLength = slHeader.getUint8(16);
      const stationIdDV = new DataView(data, 17, stationIdLength);
      const stationId = dataViewToString(stationIdDV);
      const dataView = new DataView(
        data,
        17 + stationIdLength,
        data.byteLength - 16
      );
      sePacket = new _SEPacket(
        String.fromCharCode(dataFormat),
        String.fromCharCode(dataSubformat),
        payloadLength,
        sequenceNum,
        stationId
      );
      sePacket._rawPayload = dataView;
      if (dataFormat === 50) {
        sePacket._miniseed = parseSingleDataRecord(dataView);
      } else if (dataFormat === 51) {
        sePacket._mseed3 = MSeed3Record.parseSingleDataRecord(dataView);
      } else if (dataFormat === 74) {
        const jsonData = JSON.parse(dataViewToString(dataView));
        sePacket._json = jsonData;
      }
    } else {
      throw new Error(
        "Not a seedlink4 packet, no starting SE: " + slHeader.getInt8(0) + " " + slHeader.getInt8(1)
      );
    }
    return sePacket;
  }
  /**
   * is this packet a miniseed packet
   *
   * @returns          true if it is miniseed
   */
  isMiniseed() {
    return isDef(this._miniseed) || this.dataFormat === MINISEED_2_FORMAT;
  }
  /**
   * Parsed payload as a miniseed data record, if the streamid
   * ends with '/MSEED', null otherwise.
   *
   * @returns miniseed DataRecord or null
   */
  asMiniseed() {
    if (!isDef(this._rawPayload)) {
      throw new Error(
        `payload is missing in packet from ${this.stationId}, seq: ${this.sequence}`
      );
    }
    if (!isDef(this._miniseed)) {
      if (this.dataFormat === MINISEED_2_FORMAT && isDef(this._rawPayload)) {
        this._miniseed = parseSingleDataRecord(this._rawPayload);
      } else {
        this._miniseed = null;
      }
    }
    return this._miniseed;
  }
  /**
   * is this packet a miniseed3 packet
   *
   * @returns          true if it is miniseed3
   */
  isMiniseed3() {
    return isDef(this._mseed3) || this.dataFormat === MINISEED_3_FORMAT;
  }
  /**
   * Parsed payload as a miniseed3 data record, if the data format is 3, null otherwise.
   *
   * @returns miniseed3 DataRecord or null
   */
  asMiniseed3() {
    if (!isDef(this._rawPayload)) {
      throw new Error(
        `payload is missing in packet from ${this.stationId}, seq: ${this.sequence}`
      );
    }
    if (!isDef(this._mseed3)) {
      if (this.dataFormat === MINISEED_3_FORMAT && isDef(this._rawPayload)) {
        this._mseed3 = MSeed3Record.parseSingleDataRecord(
          this._rawPayload
        );
      } else if (this.isMiniseed()) {
        const ms2 = this.asMiniseed();
        if (ms2) {
          this._mseed3 = convertMS2Record(ms2);
        }
      } else {
        this._mseed3 = null;
      }
    }
    return this._mseed3;
  }
};
var SeedlinkConnection2 = class {
  url;
  requestConfig;
  receivePacketFn;
  errorHandler;
  closeFn;
  webSocket;
  endCommand;
  agent;
  agentVersion;
  constructor(url, requestConfig, receivePacketFn, errorHandler) {
    this.webSocket = null;
    this.url = url;
    this.requestConfig = requestConfig;
    this.receivePacketFn = receivePacketFn;
    this.errorHandler = errorHandler;
    this.closeFn = null;
    this.endCommand = END_COMMAND;
    this.agent = "seisplotjs";
    this.agentVersion = version;
  }
  setAgent(agent) {
    this.agent = agent.trim().replaceAll(/\w+/g, "_");
  }
  createDataTimeCommand(startTime, endTime) {
    const endTimeStr = isDef(endTime) ? endTime.toISO() : "";
    return `DATA ALL ${startTime.toISO()} ${endTimeStr}`;
  }
  setOnError(errorHandler) {
    this.errorHandler = errorHandler;
  }
  setOnClose(closeFn) {
    this.closeFn = closeFn;
  }
  connect() {
    this.interactiveConnect().then(() => {
      return this.sendHello();
    }).then((lines) => {
      if (this.checkProto(lines)) {
        return true;
      } else {
        throw new Error(`${SEEDLINK4_PROTOCOL} not found in HELLO response`);
      }
    }).then(() => {
      return this.sendCmdArray([
        `${USERAGENT_COMMAND} ${this.agent}/${this.agentVersion} (seisplotjs/${version})`
      ]);
    }).then(() => {
      return this.sendCmdArray(this.requestConfig);
    }).then(() => {
      return this.sendCmdArray([this.endCommand]);
    }).then((val) => {
      if (this.webSocket === null) {
        throw new Error("websocket is null");
      }
      this.webSocket.onmessage = (event) => {
        this.handle(event);
      };
      this.webSocket.send(`${this.endCommand}\r`);
      return val;
    }).catch((err) => {
      this.close();
      const insureErr = err instanceof Error ? err : new Error(stringify(err));
      if (this.errorHandler) {
        this.errorHandler(insureErr);
      } else {
        throw insureErr;
      }
    });
  }
  interactiveConnect() {
    if (this.webSocket) {
      this.webSocket.close();
      this.webSocket = null;
    }
    return new Promise((resolve, reject) => {
      try {
        const webSocket = new WebSocket(this.url, SEEDLINK4_PROTOCOL);
        this.webSocket = webSocket;
        webSocket.binaryType = "arraybuffer";
        webSocket.onopen = () => {
          resolve(this);
        };
        webSocket.onerror = (event) => {
          this.handleError(new Error("" + stringify(event)));
          reject(event);
        };
        webSocket.onclose = (closeEvent) => {
          if (this.closeFn) {
            this.closeFn(closeEvent);
          }
          if (this.webSocket) {
            this.webSocket = null;
          }
        };
      } catch (err) {
        this.close();
        if (this.errorHandler) {
          this.errorHandler(toError(err));
        }
        reject(err);
      }
    }).then(function(sl4) {
      return sl4;
    });
  }
  checkProto(lines) {
    const sl = lines[0].split("::");
    const caps = sl[1].trim().split(" ");
    for (const c of caps) {
      if (c === SEEDLINK4_PROTOCOL) {
        return true;
      }
    }
    return false;
  }
  close() {
    if (this.webSocket) {
      this.webSocket.close();
    }
    this.webSocket = null;
  }
  handle(event) {
    if (event.data instanceof ArrayBuffer) {
      const rawdata = event.data;
      const data = new Uint8Array(rawdata);
      if (data[0] === 83 && data[1] === 69) {
        this.handleSEPacket(event);
      } else {
        this.close();
        this.errorHandler(
          new Error(
            `Packet does not look like SE packet: ${data[0]} ${data[1]}`
          )
        );
      }
    } else {
      this.close();
      this.errorHandler(new Error("event.data is not ArrayBuffer"));
    }
  }
  handleSEPacket(event) {
    if (event.data instanceof ArrayBuffer) {
      const data = event.data;
      try {
        const out = SEPacket.parse(data);
        this.receivePacketFn(out);
      } catch (e) {
        this.close();
        this.errorHandler(toError(e));
      }
    } else {
      this.close();
      this.errorHandler(new Error("event.data is not ArrayBuffer"));
    }
  }
  isConnected() {
    return this.webSocket !== null;
  }
  /**
   * Sends initial HELLO to server and waits for response.
   *
   * @returns            Promise that resolves to the response from the server.
   */
  sendHello() {
    const webSocket = this.webSocket;
    const promise = new Promise((resolve, reject) => {
      if (webSocket) {
        webSocket.onmessage = (event) => {
          if (event.data instanceof ArrayBuffer) {
            const data = event.data;
            const replyMsg = dataViewToString(new DataView(data));
            const lines = replyMsg.trim().split("\r");
            if (lines.length === 2) {
              resolve(lines);
            } else {
              reject("not 2 lines: " + replyMsg);
            }
          } else {
            this.close();
            this.errorHandler(new Error("event.data is not ArrayBuffer"));
          }
        };
        webSocket.send(`${HELLO_COMMAND}\r`);
      } else {
        reject("webSocket has been closed");
      }
    });
    return promise;
  }
  /**
   * Sends an array of commands, each as a Promise waiting for the 'OK' response
   * before sending the next.
   *
   * @param   cmd array of commands to send
   * @returns      Promise that resolves to the 'OK' returned by the last
   *   command if successful, or rejects on the first failure.
   */
  sendCmdArray(cmd) {
    return cmd.reduce((accum, next) => {
      return accum.then(() => {
        return this.createCmdPromise(next);
      });
    }, Promise.resolve("OK"));
  }
  /**
   * creates a Promise that sends a command and waits resolved with the result.
   *
   * @param   mycmd command string to send.
   * @returns        Promise that resolves to the reply from the server.
   */
  createCmdPromise(mycmd) {
    const mythis = this;
    const webSocket = this.webSocket;
    const promise = new Promise(function(resolve, reject) {
      if (webSocket) {
        webSocket.onmessage = (event) => {
          if (event.data instanceof ArrayBuffer) {
            const data = event.data;
            const replyMsg = dataViewToString(new DataView(data)).trim();
            if (replyMsg === "OK") {
              resolve(replyMsg);
            } else {
              reject("msg not OK: " + replyMsg);
            }
          } else {
            mythis.close();
            mythis.errorHandler(new Error("event.data is not ArrayBuffer"));
          }
        };
        webSocket.send(mycmd + "\r\n");
      } else {
        reject("webSocket has been closed");
      }
    });
    return promise;
  }
  /**
   * handle errors that arise
   *
   * @private
   * @param   error the error
   */
  handleError(error) {
    if (this.errorHandler) {
      this.errorHandler(error);
    } else {
      log("seedlink4 handleError: " + error.message);
    }
  }
};

// src/seismogramloader.ts
var seismogramloader_exports = {};
__export(seismogramloader_exports, {
  SeismogramLoader: () => SeismogramLoader
});
import { Duration as Duration12, Interval as Interval6 } from "luxon";

// src/irisfedcatalog.ts
var SERVICE_VERSION5 = 1;
var SERVICE_NAME5 = `irisws-fedcatalog-${SERVICE_VERSION5}`;
var IRIS_HOST4 = "service.iris.edu";
var TARGET_DATASELECT = "dataselect";
var FAKE_EMPTY_TEXT = "\n";
var FedCatalogDataCenter = class {
  dataCenter;
  services;
  stationService;
  dataSelectService;
  postLines;
  stationQuery;
  dataSelectQuery;
  level;
  constructor() {
    this.dataCenter = "";
    this.stationService = "";
    this.dataSelectService = "";
    this.postLines = [];
    this.services = /* @__PURE__ */ new Map();
    this.stationQuery = null;
    this.dataSelectQuery = null;
    this.level = LEVEL_NETWORK;
  }
  /**
   * Uses the response from the FedCat server to make the actual FDSNStation
   * query that returns StationXML. If the original FedCat query did not return
   * a Station service, or it was not asked for, then the array will be empty.
   *
   * @returns promise to networks
   */
  queryNetworkList() {
    if (this.stationQuery) {
      return this.stationQuery.postQuery(this.level, this.postLines);
    } else {
      return Promise.all([]);
    }
  }
  queryStationRawXml() {
    if (isDef(this.stationQuery)) {
      return this.stationQuery.postQueryRawXml(this.level, this.postLines);
    } else {
      throw new Error("this.stationQuery does not exist.");
    }
  }
  querySDDList() {
    if (isDef(this.dataSelectQuery)) {
      const sddList = this.postLines.map((line) => {
        const items = line.split(" ");
        const start = isoToDateTime(items[4]);
        const end = isoToDateTime(items[5]);
        return SeismogramDisplayData.fromCodesAndTimes(
          items[0],
          items[1],
          items[2],
          items[3],
          start,
          end
        );
      });
      return this.dataSelectQuery.postQuerySeismograms(sddList);
    } else {
      return Promise.all([]);
    }
  }
};
var FedCatalogResult = class {
  params;
  queries;
  constructor() {
    this.params = /* @__PURE__ */ new Map();
    this.queries = [];
  }
};
var FedCatalogQuery = class _FedCatalogQuery extends FDSNCommon {
  /** @private */
  _targetService;
  /** @private */
  _level;
  /** @private */
  _networkCode;
  /** @private */
  _stationCode;
  /** @private */
  _locationCode;
  /** @private */
  _channelCode;
  /** @private */
  _startTime;
  /** @private */
  _endTime;
  /** @private */
  _startBefore;
  /** @private */
  _endBefore;
  /** @private */
  _startAfter;
  /** @private */
  _endAfter;
  /** @private */
  _minLat;
  /** @private */
  _maxLat;
  /** @private */
  _minLon;
  /** @private */
  _maxLon;
  /** @private */
  _latitude;
  /** @private */
  _longitude;
  /** @private */
  _minRadius;
  /** @private */
  _maxRadius;
  /** @private */
  _includeRestricted;
  /** @private */
  _includeAvailability;
  /** @private */
  _format;
  /** @private */
  _updatedAfter;
  /** @private */
  _matchTimeseries;
  fedCatResult;
  /**
   * Construct a query
   *
   * @param host the host to connect to , defaults to service.iris.edu
   */
  constructor(host) {
    if (!isNonEmptyStringArg(host)) {
      host = IRIS_HOST4;
    }
    super(host);
    this.fedCatResult = null;
  }
  /**
   * Constructs a station FedCatalogQuery using the parameters in a StationQuery.
   *
   * @param   stationQuery query to pull parameters from
   * @returns               fedcatalog query
   */
  static fromStationQuery(stationQuery) {
    const out = new _FedCatalogQuery();
    if (!(stationQuery instanceof StationQuery)) {
      throw new Error(
        "1st arg must be a StationQuery: " + stringify(stationQuery)
      );
    }
    if (!stationQuery.isSomeParameterSet()) {
      throw new Error(
        "Some parameters must be set in the stationQuery to avoid asking for everything."
      );
    }
    if (isStringArg(stationQuery._networkCode)) {
      out.networkCode(stationQuery._networkCode);
    }
    if (isStringArg(stationQuery._stationCode)) {
      out.stationCode(stationQuery._stationCode);
    }
    if (isStringArg(stationQuery._locationCode)) {
      out.locationCode(stationQuery._locationCode);
    }
    if (isStringArg(stationQuery._channelCode)) {
      out.channelCode(stationQuery._channelCode);
    }
    if (isObject(stationQuery._startTime)) {
      out.startTime(stationQuery._startTime);
    }
    if (isObject(stationQuery._endTime)) {
      out.endTime(stationQuery._endTime);
    }
    if (isObject(stationQuery._startBefore)) {
      out.startBefore(stationQuery._startBefore);
    }
    if (isObject(stationQuery._startAfter)) {
      out.startAfter(stationQuery._startAfter);
    }
    if (isObject(stationQuery._endBefore)) {
      out.endBefore(stationQuery._endBefore);
    }
    if (isObject(stationQuery._endAfter)) {
      out.endAfter(stationQuery._endAfter);
    }
    if (isNumArg(stationQuery._minLat)) {
      out.minLat(stationQuery._minLat);
    }
    if (isNumArg(stationQuery._maxLat)) {
      out.maxLat(stationQuery._maxLat);
    }
    if (isNumArg(stationQuery._minLon)) {
      out.minLon(stationQuery._minLon);
    }
    if (isNumArg(stationQuery._maxLon)) {
      out.maxLon(stationQuery._maxLon);
    }
    if (isNumArg(stationQuery._latitude)) {
      out.latitude(stationQuery._latitude);
    }
    if (isNumArg(stationQuery._longitude)) {
      out.longitude(stationQuery._longitude);
    }
    if (isNumArg(stationQuery._minRadius)) {
      out.minRadius(stationQuery._minRadius);
    }
    if (isNumArg(stationQuery._maxRadius)) {
      out.maxRadius(stationQuery._maxRadius);
    }
    if (isDef(stationQuery._includeRestricted)) {
      out.includeRestricted(stationQuery._includeRestricted);
    }
    if (isDef(stationQuery._includeAvailability)) {
      out.includeAvailability(stationQuery._includeAvailability);
    }
    if (isObject(stationQuery._updatedAfter)) {
      out.updatedAfter(stationQuery._updatedAfter);
    }
    if (isDef(stationQuery._matchTimeseries)) {
      out.matchTimeseries(stationQuery._matchTimeseries);
    }
    return out;
  }
  /**
   * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
   *  Setting this is probably a bad idea as the code may not be compatible with
   *  the web service.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  specVersion(value) {
    doStringGetterSetter(this, "specVersion", value);
    return this;
  }
  getSpecVersion() {
    return this._specVersion;
  }
  /**
   * Gets/Sets the protocol, http or https. This should match the protocol
   *  of the page loaded, but is autocalculated and generally need not be set.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  protocol(value) {
    doStringGetterSetter(this, "protocol", value);
    return this;
  }
  getProtocol() {
    return this._protocol;
  }
  /**
   * Gets/Sets the remote host to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  host(value) {
    doStringGetterSetter(this, "host", value);
    return this;
  }
  getHost() {
    return this._host;
  }
  /**
   * Gets/Sets the remote port to connect to.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  port(value) {
    doIntGetterSetter(this, "port", value);
    return this;
  }
  getPort() {
    return this._port;
  }
  /**
   * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
   * the status code when no matching data is found by the service.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  nodata(value) {
    doIntGetterSetter(this, "nodata", value);
    return this;
  }
  getNodata() {
    return this._nodata;
  }
  /**
   * Get/Set the targetservice query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  targetService(value) {
    doStringGetterSetter(this, "targetService", value);
    return this;
  }
  getTargetService() {
    return this._targetService;
  }
  /**
   * Get/Set the network query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  networkCode(value) {
    doStringGetterSetter(this, "networkCode", value);
    return this;
  }
  getNetworkCode() {
    return this._networkCode;
  }
  /**
   * Get/Set the station query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  stationCode(value) {
    doStringGetterSetter(this, "stationCode", value);
    return this;
  }
  getStationCode() {
    return this._stationCode;
  }
  /**
   * Get/Set the location code query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  locationCode(value) {
    doStringGetterSetter(this, "locationCode", value);
    return this;
  }
  getLocationCode() {
    return this._locationCode;
  }
  /**
   * Get/Set the channel query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  channelCode(value) {
    doStringGetterSetter(this, "channelCode", value);
    return this;
  }
  getChannelCode() {
    return this._channelCode;
  }
  nslcCodes(channelId) {
    this.networkCode(channelId.networkCode);
    this.stationCode(channelId.stationCode);
    this.locationCode(channelId.locationCode);
    this.channelCode(channelId.channelCode);
    return this;
  }
  /**
   * Get/Set the starttime query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  startTime(value) {
    doMomentGetterSetter(this, "startTime", value);
    return this;
  }
  getStartTime() {
    return this._startTime;
  }
  /**
   * Get/Set the endtime query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  endTime(value) {
    doMomentGetterSetter(this, "endTime", value);
    return this;
  }
  getEndTime() {
    return this._endTime;
  }
  /**
   * Sets startTime and endTime using the given time window
   *
   * @param   se time window
   * @returns     this
   */
  timeRange(se) {
    this.startTime(validStartTime(se));
    this.endTime(validEndTime(se));
    return this;
  }
  /**
   * Get/Set the startbefore query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  startBefore(value) {
    doMomentGetterSetter(this, "startBefore", value);
    return this;
  }
  getStartBefore() {
    return this._startBefore;
  }
  /**
   * Get/Set the endbefore query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  endBefore(value) {
    doMomentGetterSetter(this, "endBefore", value);
    return this;
  }
  getEndBefore() {
    return this._endBefore;
  }
  /**
   * Get/Set the startafter query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  startAfter(value) {
    doMomentGetterSetter(this, "startAfter", value);
    return this;
  }
  getStartAfter() {
    return this._startAfter;
  }
  /**
   * Get/Set the endafter query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  endAfter(value) {
    doMomentGetterSetter(this, "endAfter", value);
    return this;
  }
  getEndAfter() {
    return this._endAfter;
  }
  /**
   * Get/Set the minlat query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minLat(value) {
    doFloatGetterSetter(this, "minLat", value);
    return this;
  }
  getMinLat() {
    return this._minLat;
  }
  /**
   * Get/Set the maxlon query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxLat(value) {
    doFloatGetterSetter(this, "maxLat", value);
    return this;
  }
  getMaxLat() {
    return this._maxLat;
  }
  /**
   * Get/Set the minlon query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minLon(value) {
    doFloatGetterSetter(this, "minLon", value);
    return this;
  }
  getMinLon() {
    return this._minLon;
  }
  /**
   * Get/Set the maxlon query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxLon(value) {
    doFloatGetterSetter(this, "maxLon", value);
    return this;
  }
  getMaxLon() {
    return this._maxLon;
  }
  /**
   * Get/Set the latitude query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  latitude(value) {
    doFloatGetterSetter(this, "latitude", value);
    return this;
  }
  getLatitude() {
    return this._latitude;
  }
  /**
   * Get/Set the longitude query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  longitude(value) {
    doFloatGetterSetter(this, "longitude", value);
    return this;
  }
  getLongitude() {
    return this._longitude;
  }
  /**
   * Get/Set the minradius query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  minRadius(value) {
    doFloatGetterSetter(this, "minRadius", value);
    return this;
  }
  getMinRadius() {
    return this._minRadius;
  }
  /**
   * Get/Set the maxradius query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  maxRadius(value) {
    doFloatGetterSetter(this, "maxRadius", value);
    return this;
  }
  getMaxRadius() {
    return this._maxRadius;
  }
  /**
   * Get/Set the includerestricted query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  includeRestricted(value) {
    doBoolGetterSetter(this, "includeRestricted", value);
    return this;
  }
  getIncludeRestricted() {
    return this._includeRestricted;
  }
  /**
   * Get/Set the includeavailability query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  includeAvailability(value) {
    doBoolGetterSetter(this, "includeAvailability", value);
    return this;
  }
  getIncludeAvailability() {
    return this._includeAvailability;
  }
  /**
   * Get/Set the format query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  format(value) {
    doStringGetterSetter(this, "format", value);
    return this;
  }
  getFormat() {
    return this._format;
  }
  /**
   * Get/Set the updatedafter query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  updatedAfter(value) {
    doMomentGetterSetter(this, "updatedAfter", value);
    return this;
  }
  getUpdatedAfter() {
    return this._updatedAfter;
  }
  /**
   * Get/Set the matchtimeseries query parameter.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  matchTimeseries(value) {
    doBoolGetterSetter(this, "matchTimeseries", value);
    return this;
  }
  getMatchTimeseries() {
    return this._matchTimeseries;
  }
  /**
   * Get/Set the timeout in seconds for the request. Default is 30.
   *
   * @param value optional new value if setting
   * @returns new value if getting, this if setting
   */
  timeout(value) {
    doFloatGetterSetter(this, "timeoutSec", value);
    return this;
  }
  getTimeout() {
    return this._timeoutSec;
  }
  /**
   * Checks to see if any parameter that would limit the data
   * returned is set. This is a crude, coarse check to make sure
   * the client doesn't ask for EVERYTHING the server has.
   *
   * @returns true if some parameter set
   */
  isSomeParameterSet() {
    return isDef(this._networkCode) && this._networkCode.length > 0 && this._networkCode !== "*" || isDef(this._stationCode) && this._stationCode.length > 0 && this._stationCode !== "*" || isDef(this._locationCode) && this._locationCode.length > 0 && this._locationCode !== "*" || isDef(this._channelCode) && this._channelCode.length > 0 && this._channelCode !== "*" || isDef(this._startTime) || isDef(this._endTime) || isDef(this._startBefore) || isDef(this._endBefore) || isDef(this._startAfter) || isDef(this._endAfter) || isDef(this._minLat) || isDef(this._maxLat) || isDef(this._minLon) || isDef(this._maxLon) || isDef(this._latitude) || isDef(this._longitude) || isDef(this._minRadius) || isDef(this._maxRadius) || isDef(this._updatedAfter);
  }
  /**
   * Queries the remote web service for networks.
   *
   * @returns a Promise to an Array of Network objects.
   */
  queryNetworks() {
    return this.queryFdsnStation(LEVEL_NETWORK);
  }
  /**
   * Queries the remote web service for stations. The stations
   * are contained within their respective Networks.
   *
   * @returns a Promise to an Array of Network objects.
   */
  queryStations() {
    return this.queryFdsnStation(LEVEL_STATION);
  }
  /**
   * Queries the remote web service for channels. The Channels
   * are contained within their respective Stations which are in Networks.
   *
   * @returns a Promise to an Array of Network objects.
   */
  queryChannels() {
    return this.queryFdsnStation(LEVEL_CHANNEL);
  }
  /**
   * Queries the remote web service for responses. The Responses
   * are contained within their respective Channels,
   * which are in Stations which are in Networks.
   *
   * @returns a Promise to an Array of Network objects.
   */
  queryResponses() {
    return this.queryFdsnStation(LEVEL_RESPONSE);
  }
  /**
   * Queries the remote station web service at the given level.
   *
   * @param level the level to query at, networ, station, channel or response.
   * @returns a Promise to an Array of Network objects.
   */
  queryFdsnStation(level) {
    return this.setupQueryFdsnStation(level).then((fedCatalogResult) => {
      return Promise.all(
        fedCatalogResult.queries.map((query) => query.queryNetworkList())
      );
    }).then((netArrayArray) => {
      const out = [];
      netArrayArray.forEach((netArray) => {
        netArray.forEach((net) => {
          out.push(net);
        });
      });
      return out;
    });
  }
  setupQueryFdsnStation(level) {
    if (!LEVELS.includes(level)) {
      throw new Error("Unknown level: '" + level + "'");
    }
    this._level = level;
    this.targetService("station");
    return this.queryRaw().then(function(fedCatalogResult) {
      for (const r of fedCatalogResult.queries) {
        const stationQuery = new StationQuery();
        r.stationQuery = stationQuery;
        fedCatalogResult.params.forEach((v, k) => {
          const field = `_${k}`;
          if (field === "_level") {
          } else if (field in stationQuery) {
            stationQuery[field] = v;
          } else {
            throw new Error(
              `field ${field} does not exist in StationQuery class`
            );
          }
        });
        if (!r.services.has("STATIONSERVICE") || !isDef(r.services.get("STATIONSERVICE"))) {
          throw new Error(
            "QueryResult does not have STATIONSERVICE in services"
          );
        }
        const urlString = r.services.get("STATIONSERVICE");
        if (isDef(urlString)) {
          const serviceURL = new URL(urlString);
          stationQuery.host(serviceURL.hostname);
          if (serviceURL.port) {
            stationQuery.port(parseInt(serviceURL.port));
          }
        } else {
          throw new Error(
            "QueryResult does have STATIONSERVICE in services, but is undef"
          );
        }
      }
      return fedCatalogResult;
    });
  }
  /**
   * For each item in a parsed result returned from a FedCat service, create a
   * DataSelectQuery with host and port, or url filled in correctly, ready to
   * be called with the result lines.
   *
   * @param   fedCatalogResult result from a FedCat web service
   * @returns               result with dataSelectQuery added to each item
   */
  setupForFdnsDataSelect(fedCatalogResult) {
    for (const r of fedCatalogResult.queries) {
      const dataSelectQuery = new DataSelectQuery();
      r.dataSelectQuery = dataSelectQuery;
      fedCatalogResult.params.forEach((k, v) => {
        const field = `_${k}`;
        if (field in dataSelectQuery) {
          dataSelectQuery[field] = v;
        } else {
          throw new Error(
            `field ${field} does not exist in DataSelectQuery class`
          );
        }
      });
      if (!r.services.has("DATASELECTSERVICE")) {
        throw new Error(
          "QueryResult does not have DATASELECTSERVICE in services"
        );
      }
      const urlString = r.services.get("DATASELECTSERVICE");
      if (isDef(urlString)) {
        const serviceURL = new URL(urlString);
        dataSelectQuery.host(serviceURL.hostname);
        if (serviceURL.port) {
          dataSelectQuery.port(parseInt(serviceURL.port));
        }
      } else {
        throw new Error(
          "QueryResult does have DATASELECTSERVICE in services, but is undef"
        );
      }
    }
    return fedCatalogResult;
  }
  queryFdsnDataselect() {
    this.targetService(TARGET_DATASELECT);
    return this.queryRaw().then((fedCatalogResult) => {
      return this.setupForFdnsDataSelect(fedCatalogResult);
    }).then((fedCatalogResult) => {
      return this.postFdsnDataselectForFedCatResult(fedCatalogResult);
    });
  }
  postFdsnDataselectForFedCatResult(fedCatalogResult) {
    return Promise.all(
      fedCatalogResult.queries.map((query) => query.querySDDList())
    ).then((sddArrayArray) => {
      const out = [];
      sddArrayArray.forEach((sddArray) => {
        sddArray.forEach((sdd) => {
          out.push(sdd);
        });
      });
      return out;
    });
  }
  /**
   * query the fed cat server for dataselect using post, which allows for multiple
   * channel-timeranges at once. This assumes that there are not multiple
   * time ranges for the same channel as the results, encapsulated as
   * SeismogramDisplayData objects, are returned one seismogram
   * per channel, which may contain gaps. The original channel and timerange are
   * also populated with each result.
   *
   * @param   sddList array of SeismogramDisplayData objects
   * that will be filled in with the resulting seismogram
   * @returns Promise to the input Array of SeismogramDisplayData objects, each with the
   * seismogram containing the data returned from the server
   */
  postQuerySeismograms(sddList) {
    return this.postQueryRaw(sddList, TARGET_DATASELECT).then((fedCatalogResult) => {
      this.setupForFdnsDataSelect(fedCatalogResult);
      return this.postFdsnDataselectForFedCatResult(fedCatalogResult);
    }).then((sddResultArray) => {
      for (const sdd of sddList) {
        const codes = sdd.codes();
        const matchSdd = sddResultArray.find(
          (s) => s.codes() === codes && s.timeRange.overlaps(sdd.timeRange)
        );
        if (matchSdd) {
          sdd.seismogram = matchSdd.seismogram;
        }
      }
      return sddList;
    });
  }
  postQueryRaw(sddList, targetService) {
    if (sddList.length === 0) {
      return Promise.resolve(this.parseRequest(FAKE_EMPTY_TEXT));
    } else {
      const body = `targetservice=${targetService}
` + DataSelectQuery.createPostBody(sddList);
      return this.postQueryRawWithBody(body);
    }
  }
  postQueryRawWithBody(body) {
    const fetchInit = defaultFetchInitObj(TEXT_MIME);
    fetchInit.method = "POST";
    fetchInit.body = body;
    return doFetchWithTimeout(
      this.formPostURL(),
      fetchInit,
      this._timeoutSec * 1e3
    ).then((response) => {
      return this.handleHttpResponseCodes(response);
    }).then((rawText) => {
      return this.parseRequest(rawText);
    });
  }
  /**
   * Queries the remote web service.
   *
   * @returns a Promise to an parsed result.
   */
  queryRaw() {
    if (!this.isSomeParameterSet()) {
      throw new Error(
        "Must set some parameter to avoid asking for everything."
      );
    }
    const url = this.formURL();
    const fetchInit = defaultFetchInitObj(TEXT_MIME);
    this.fedCatResult = doFetchWithTimeout(
      url,
      fetchInit,
      this._timeoutSec * 1e3
    ).then((response) => {
      return this.handleHttpResponseCodes(response);
    }).then((rawText) => {
      return this.parseRequest(rawText);
    });
    return this.fedCatResult;
  }
  handleHttpResponseCodes(response) {
    if (response.status === 200) {
      return response.text();
    } else if (response.status === 204 || isDef(this._nodata) && response.status === this._nodata) {
      return Promise.resolve(FAKE_EMPTY_TEXT);
    } else {
      throw new Error(`Status not successful: ${response.status}`);
    }
  }
  parseRequest(requestText) {
    const out = new FedCatalogResult();
    const lines = requestText.split("\n");
    let inParams = true;
    let query = null;
    for (let l of lines) {
      l = l.trim();
      if (inParams) {
        if (l.length === 0) {
          inParams = false;
        } else {
          const keyval = l.split("=");
          out.params.set(keyval[0], keyval[1]);
        }
      } else {
        if (l.length === 0) {
          query = null;
        } else {
          if (query === null) {
            query = new FedCatalogDataCenter();
            if (this._level) {
              query.level = this._level;
            }
            out.queries.push(query);
          }
          if (l.indexOf("=") !== -1) {
            const keyval = l.split("=");
            if (keyval[0] === "DATACENTER") {
              query.dataCenter = keyval[1];
            } else if (keyval[0].endsWith("SERVICE")) {
              query.services.set(keyval[0], keyval[1]);
            } else {
              throw new Error(`Unexpected line in FedCatalog response: '${l}'`);
            }
          } else {
            query.postLines.push(l);
          }
        }
      }
    }
    return out;
  }
  /**
   * Forms the URL to get version from the web service, without any query paramters
   *
   * @returns the url
   */
  formVersionURL() {
    return this.formBaseURL() + "/version";
  }
  /**
   * Queries the remote web service to get its version
   *
   * @returns Promise to version string
   */
  queryVersion() {
    const url = this.formVersionURL();
    const fetchInit = defaultFetchInitObj(TEXT_MIME);
    return doFetchWithTimeout(url, fetchInit, this._timeoutSec * 1e3).then(
      (response) => {
        if (response.status === 200) {
          return response.text();
        } else {
          throw new Error(`Status not 200: ${response.status}`);
        }
      }
    );
  }
  /**
   * Forms the basic URL to contact the web service, without any query paramters
   *
   * @returns the url
   */
  formBaseURL() {
    let colon = ":";
    if (this._protocol.endsWith(colon)) {
      colon = "";
    }
    return this._protocol + colon + "//" + this._host + (this._port === 80 ? "" : ":" + String(this._port)) + "/irisws/fedcatalog/" + this._specVersion;
  }
  /**
   * Form URL to post the remote web service. No parameters are added
   * to the URL as those will be in the body of the post.
   *
   * @returns the URL for posting
   */
  formPostURL() {
    return this.formBaseURL() + "/query";
  }
  /**
   * Form URL to query the remote web service, encoding the query parameters.
   *
   * @returns url
   */
  formURL() {
    let url = this.formBaseURL() + "/query?";
    if (isStringArg(this._level)) {
      url = url + makeParam("level", this._level);
    }
    if (isStringArg(this._targetService)) {
      url = url + makeParam("targetservice", this._targetService);
    }
    if (isStringArg(this._networkCode) && this._networkCode.length > 0 && this._networkCode !== "*") {
      url = url + makeParam("net", this._networkCode);
    } else {
      url = url + makeParam("net", "*");
    }
    if (isStringArg(this._stationCode) && this._stationCode.length > 0 && this._stationCode !== "*") {
      url = url + makeParam("sta", this._stationCode);
    }
    if (isStringArg(this._locationCode) && this._locationCode.length > 0 && this._locationCode !== "*") {
      url = url + makeParam("loc", this._locationCode);
    }
    if (isStringArg(this._channelCode) && this._channelCode.length > 0 && this._channelCode !== "*") {
      url = url + makeParam("cha", this._channelCode);
    }
    if (isObject(this._startTime)) {
      url = url + makeParam("starttime", toIsoWoZ(this._startTime));
    }
    if (isObject(this._endTime)) {
      url = url + makeParam("endtime", toIsoWoZ(this._endTime));
    }
    if (isObject(this._startBefore)) {
      url = url + makeParam("startbefore", toIsoWoZ(this._startBefore));
    }
    if (isObject(this._startAfter)) {
      url = url + makeParam("startafter", toIsoWoZ(this._startAfter));
    }
    if (isObject(this._endBefore)) {
      url = url + makeParam("endbefore", toIsoWoZ(this._endBefore));
    }
    if (isObject(this._endAfter)) {
      url = url + makeParam("endafter", toIsoWoZ(this._endAfter));
    }
    if (isNumArg(this._minLat)) {
      url = url + makeParam("minlat", this._minLat);
    }
    if (isNumArg(this._maxLat)) {
      url = url + makeParam("maxlat", this._maxLat);
    }
    if (isNumArg(this._minLon)) {
      url = url + makeParam("minlon", this._minLon);
    }
    if (isNumArg(this._maxLon)) {
      url = url + makeParam("maxlon", this._maxLon);
    }
    if (isNumArg(this._latitude)) {
      url = url + makeParam("lat", this._latitude);
    }
    if (isNumArg(this._longitude)) {
      url = url + makeParam("lon", this._longitude);
    }
    if (isNumArg(this._minRadius)) {
      url = url + makeParam("minradius", this._minRadius);
    }
    if (isNumArg(this._maxRadius)) {
      url = url + makeParam("maxradius", this._maxRadius);
    }
    if (isDef(this._includeRestricted)) {
      url = url + makeParam("includerestricted", this._includeRestricted);
    }
    if (isDef(this._includeAvailability)) {
      url = url + makeParam("includeavailability", this._includeAvailability);
    }
    if (isObject(this._updatedAfter)) {
      url = url + makeParam("updatedafter", toIsoWoZ(this._updatedAfter));
    }
    if (isDef(this._matchTimeseries)) {
      url = url + makeParam("matchtimeseries", this._matchTimeseries);
    }
    if (isStringArg(this._format)) {
      url = url + makeParam("format", this._format);
    }
    if (isNumArg(this._nodata)) {
      url = url + makeParam("nodata", this._nodata);
    }
    if (url.endsWith("&") || url.endsWith("?")) {
      url = url.substr(0, url.length - 1);
    }
    return url;
  }
};

// src/seismogramloader.ts
var SeismogramLoader = class {
  stationQuery;
  withFedCatalog;
  withResponse;
  markOrigin;
  eventQuery;
  dataselectQuery;
  _startPhaseList;
  _endPhaseList;
  _markedPhaseList;
  _startOffset;
  _endOffset;
  constructor(stationQuery, eventQuery, dataselectQuery) {
    if (stationQuery instanceof StationQuery) {
      this.stationQuery = stationQuery;
    } else if (Array.isArray(stationQuery)) {
      this.stationQuery = Promise.resolve(stationQuery);
    } else {
      throw new Error(
        "1st arg must be a StationQuery or array of Networks: " + stringify(stationQuery)
      );
    }
    if (eventQuery instanceof EventQuery) {
      this.eventQuery = eventQuery;
    } else if (Array.isArray(eventQuery)) {
      this.eventQuery = Promise.resolve(eventQuery);
    } else {
      throw new Error(
        "2nd arg must be EventQuery or array of Quake: " + stringify(eventQuery)
      );
    }
    this.withFedCatalog = true;
    this.withResponse = false;
    this.markOrigin = true;
    this.dataselectQuery = null;
    if (isDef(dataselectQuery)) {
      this.dataselectQuery = dataselectQuery;
    }
    this._startPhaseList = ["p", "P", "Pdiff", "PKP"];
    this._endPhaseList = ["s", "S", "Sdiff", "SKS"];
    this._markedPhaseList = [];
    this._startOffset = Duration12.fromMillis(-30 * 1e3);
    this._endOffset = Duration12.fromMillis(60 * 1e3);
  }
  get startPhaseList() {
    return this._startPhaseList;
  }
  set startPhaseList(val) {
    if (Array.isArray(val)) {
      this._startPhaseList = val;
    } else if (isStringArg(val)) {
      this._startPhaseList = val.split(",");
    } else {
      throw new Error(
        "value argument is string or array of string, but was " + typeof val
      );
    }
  }
  get startOffset() {
    return this._startOffset;
  }
  set startOffset(val) {
    if (Duration12.isDuration(val)) {
      this._startOffset = val;
    } else if (typeof val === "number") {
      this.startOffsetSeconds(val);
    } else {
      throw new Error(
        "startOffset must be luxon Duration or number of seconds: " + stringify(val)
      );
    }
  }
  /**
   * Sets the startOffset Duration to be val seconds.
   *
   * @param  val  number of seconds, negative for before, positive for after
   * @returns     this
   */
  startOffsetSeconds(val) {
    this._startOffset = Duration12.fromMillis(val * 1e3);
    return this;
  }
  get endPhaseList() {
    return this._endPhaseList;
  }
  set endPhaseList(val) {
    if (Array.isArray(val)) {
      this._endPhaseList = val;
    } else if (isStringArg(val)) {
      this._endPhaseList = val.split(",");
    } else {
      throw new Error(
        "value argument is string or array of string, but was " + typeof val
      );
    }
  }
  get endOffset() {
    return this._endOffset;
  }
  set endOffset(val) {
    if (Duration12.isDuration(val)) {
      this._endOffset = val;
    } else if (typeof val === "number") {
      this.endOffsetSeconds(val);
    } else {
      throw new Error(
        "startOffset must be luxon Duration or number of seconds: " + stringify(val)
      );
    }
  }
  /**
   * Sets the endOffset Duration to be val seconds.
   *
   * @param  val  number of seconds, negative for before, positive for after
   * @returns     this
   */
  endOffsetSeconds(val) {
    this._endOffset = Duration12.fromMillis(val * 1e3);
    return this;
  }
  /**
   * Additional phase arrival travel times to be marked, but do not effect
   * the request time window.
   *
   * @returns array of phase names.
   */
  get markedPhaseList() {
    return this._markedPhaseList;
  }
  set markedPhaseList(val) {
    if (Array.isArray(val)) {
      this._markedPhaseList = val;
    } else if (isStringArg(val)) {
      this._markedPhaseList = val.split(",");
    } else {
      throw new Error(
        "value argument is string or array of string, but was " + typeof val
      );
    }
  }
  loadSeismograms() {
    return this.load().then((res) => res.waveforms);
  }
  /**
   * Loads a Dataset based on the input station and event queries.
   *
   * The raw traveltimes are included in the extras of the dataset with
   * key "traveltimes", which is a Map with the quake as the key.
   *
   * @returns a Dataset
   */
  load() {
    let networkListPromise;
    if (this.stationQuery instanceof StationQuery) {
      if (!this.stationQuery.isSomeParameterSet()) {
        throw new Error(
          "Must set some station parameter to avoid asking for everything."
        );
      }
      let fedcat;
      if (this.withFedCatalog) {
        fedcat = FedCatalogQuery.fromStationQuery(this.stationQuery);
      } else {
        fedcat = this.stationQuery;
      }
      if (this.withResponse) {
        networkListPromise = fedcat.queryResponses();
      } else {
        networkListPromise = fedcat.queryChannels();
      }
    } else {
      networkListPromise = this.stationQuery;
    }
    let quakeListPromise;
    if (this.eventQuery instanceof EventQuery) {
      if (!this.eventQuery.isSomeParameterSet()) {
        throw new Error(
          "Must set some event parameter to avoid asking for everything."
        );
      }
      quakeListPromise = this.eventQuery.query();
    } else {
      quakeListPromise = this.eventQuery;
    }
    let allPhaseList = [];
    allPhaseList = allPhaseList.concat(
      this.startPhaseList,
      this.endPhaseList,
      this.markedPhaseList
    );
    if (allPhaseList.includes("origin")) {
      this.markOrigin = true;
    }
    const allPhasesWithoutOrigin = allPhaseList.filter((p) => p !== "origin").join(",");
    return Promise.all([networkListPromise, quakeListPromise]).then(([netList, quakeList]) => {
      const ttpromiseList = [];
      for (const q of quakeList) {
        const allDistDeg = [];
        for (const s of allStations(netList)) {
          if (s.timeRange.contains(q.time)) {
            const daz = distaz(
              s.latitude,
              s.longitude,
              q.latitude,
              q.longitude
            );
            allDistDeg.push(daz.distanceDeg);
          }
        }
        if (allDistDeg.length > 0) {
          const taupQuery = new TraveltimeQuery();
          taupQuery.distdeg(allDistDeg);
          taupQuery.phases(allPhasesWithoutOrigin);
          ttpromiseList.push(Promise.all([q, taupQuery.queryJson()]));
        }
      }
      return Promise.all([Promise.all(ttpromiseList), netList, quakeList]);
    }).then(([ttList, netList, quakeList]) => {
      const ttMap = /* @__PURE__ */ new Map();
      for (const [q, tt] of ttList) {
        ttMap.set(q, tt);
      }
      return Promise.all([ttMap, netList, quakeList]);
    }).then(([ttMap, netList, quakeList]) => {
      const seismogramDataList = [];
      for (const [quake, ttjson] of ttMap) {
        for (const station of allStations(netList)) {
          if (!station.timeRange.contains(quake.time)) {
            continue;
          }
          const daz = distaz(
            station.latitude,
            station.longitude,
            quake.latitude,
            quake.longitude
          );
          const stationArrivals = [];
          for (const a of ttjson.arrivals) {
            if (Math.abs(a.distdeg % 360 - daz.distanceDeg % 360) < 1e-6 || Math.abs(360 - a.distdeg % 360 - daz.distanceDeg % 360) < 1e-6) {
              stationArrivals.push(a);
            }
          }
          const station_ttjson = {
            model: ttjson.model,
            sourcedepth: ttjson.sourcedepth,
            receiverdepth: ttjson.receiverdepth,
            phases: ttjson.phases,
            arrivals: stationArrivals
          };
          let startArrival = null;
          let endArrival = null;
          for (const pname of this.startPhaseList) {
            if (pname === "origin" && (startArrival === null || startArrival.time > 0)) {
              startArrival = createOriginArrival(daz.distanceDeg);
            } else {
              for (const a of stationArrivals) {
                if (a.phase === pname && (startArrival === null || startArrival.time > a.time)) {
                  startArrival = a;
                }
              }
            }
          }
          for (const pname of this.endPhaseList) {
            if (pname === "origin" && (endArrival === null || endArrival.time < 0)) {
              endArrival = createOriginArrival(daz.distanceDeg);
            } else {
              for (const a of stationArrivals) {
                if (a.phase === pname && (endArrival === null || endArrival.time < a.time)) {
                  endArrival = a;
                }
              }
            }
          }
          if (isDef(startArrival) && isDef(endArrival)) {
            const startTime = quake.time.plus(Duration12.fromMillis(1e3 * startArrival.time)).plus(this.startOffset);
            const endTime = quake.time.plus(Duration12.fromMillis(1e3 * endArrival.time)).plus(this.endOffset);
            const timeRange = Interval6.fromDateTimes(startTime, endTime);
            const phaseMarkers = createMarkersForTravelTimes(
              quake,
              station_ttjson
            );
            if (this.markOrigin) {
              phaseMarkers.push(createMarkerForOriginTime(quake));
            }
            for (const chan of station.channels) {
              if (!chan.timeRange.contains(quake.time)) {
                continue;
              }
              const sdd = SeismogramDisplayData.fromChannelAndTimeWindow(
                chan,
                timeRange
              );
              sdd.addQuake(quake);
              sdd.addTravelTimes(ttjson);
              sdd.addMarkers(phaseMarkers);
              seismogramDataList.push(sdd);
            }
          }
        }
      }
      let sddListPromise;
      if (this.dataselectQuery !== null) {
        sddListPromise = this.dataselectQuery.postQuerySeismograms(seismogramDataList);
      } else if (this.withFedCatalog) {
        const fedcatDS = new FedCatalogQuery();
        sddListPromise = fedcatDS.postQuerySeismograms(seismogramDataList);
      } else {
        sddListPromise = new DataSelectQuery().postQuerySeismograms(
          seismogramDataList
        );
      }
      return Promise.all([sddListPromise, ttMap, netList, quakeList]);
    }).then(([sddList, ttMap, networkList, quakeList]) => {
      const dataset = new Dataset();
      dataset.waveforms = sddList;
      dataset.catalog = quakeList;
      dataset.inventory = networkList;
      dataset.extra.set("traveltimes", ttMap);
      return dataset;
    });
  }
};

// src/sorting.ts
var sorting_exports = {};
__export(sorting_exports, {
  SORT_ALPHABETICAL: () => SORT_ALPHABETICAL,
  SORT_AZIMUTH: () => SORT_AZIMUTH,
  SORT_BACKAZIMUTH: () => SORT_BACKAZIMUTH,
  SORT_DISTANCE: () => SORT_DISTANCE,
  SORT_NONE: () => SORT_NONE,
  SORT_ORIGINTIME: () => SORT_ORIGINTIME,
  SORT_STARTTIME: () => SORT_STARTTIME,
  createSortValueFunction: () => createSortValueFunction,
  reorderXYZ: () => reorderXYZ,
  sort: () => sort
});
var SORT_NONE = "none";
var SORT_DISTANCE = "distance";
var SORT_AZIMUTH = "azimuth";
var SORT_BACKAZIMUTH = "backazimuth";
var SORT_ALPHABETICAL = "alphabetical";
var SORT_STARTTIME = "starttime";
var SORT_ORIGINTIME = "origin";
function sort(seisData, key) {
  if (key === SORT_NONE) {
    return seisData;
  }
  const cache = /* @__PURE__ */ new Map();
  const calcSortValue = createSortValueFunction(key);
  seisData.forEach((sdd) => {
    cache.set(sdd, calcSortValue(sdd));
  });
  return seisData.slice().sort((sddA, sddB) => {
    const valA = cache.get(sddA);
    const valB = cache.get(sddB);
    if (!valA && !valB) {
      return 0;
    } else if (!valA) {
      return 1;
    } else if (!valB) {
      return -1;
    } else if (valA < valB) {
      return -1;
    } else if (valA > valB) {
      return 1;
    } else {
      return 0;
    }
  });
}
function createSortValueFunction(key) {
  if (key === SORT_DISTANCE) {
    return (sdd) => {
      let out = Number.MAX_VALUE;
      if (sdd.hasQuake() && sdd.hasChannel()) {
        const distaz2 = sdd.distaz;
        out = distaz2 ? Math.min(Number.MAX_VALUE, distaz2.delta) : Number.MAX_VALUE;
      }
      return out;
    };
  } else if (key === SORT_AZIMUTH) {
    return (sdd) => {
      let out = Number.MAX_VALUE;
      if (sdd.hasQuake() && sdd.hasChannel()) {
        const distaz2 = sdd.distaz;
        out = distaz2 ? Math.min(Number.MAX_VALUE, distaz2.az) : Number.MAX_VALUE;
      }
      return out;
    };
  } else if (key === SORT_BACKAZIMUTH) {
    return (sdd) => {
      let out = Number.MAX_VALUE;
      if (sdd.hasQuake() && sdd.hasChannel()) {
        const distaz2 = sdd.distaz;
        out = distaz2 ? Math.min(Number.MAX_VALUE, distaz2.baz) : Number.MAX_VALUE;
      }
      return out;
    };
  } else if (key === SORT_ALPHABETICAL) {
    return (sdd) => sdd.sourceId.toString();
  } else if (key === SORT_STARTTIME) {
    return (sdd) => sdd.startTime;
  } else if (key === SORT_ORIGINTIME) {
    return (sdd) => {
      let out = WAY_FUTURE;
      if (sdd.hasQuake()) {
        out = sdd.quake ? sdd.quake.time : out;
      }
      return out;
    };
  } else {
    throw new Error(`unknown sorting key: ${key}`);
  }
}
function xyzCompareFun(a, b) {
  if (a.hasChannel() && b.hasChannel()) {
    if (Math.abs(a.channel.dip) > 85 || Math.abs(b.channel.dip) > 85) {
      if (a.channel.dip !== b.channel.dip) {
        return Math.abs(a.channel.dip) - Math.abs(b.channel.dip);
      }
    }
    const ninetyRot = Math.abs(
      (a.channel.azimuth - b.channel.azimuth + 90) % 360
    );
    if (ninetyRot < 5 || ninetyRot > 355) {
      return 1;
    }
    if (ninetyRot > 175 || ninetyRot < 185) {
      return -1;
    }
  }
  const aSID = a.sourceId;
  const bSID = b.sourceId;
  if (aSID && bSID) {
    return aSID.subsourceCode.localeCompare(bSID.subsourceCode);
  } else if (aSID === bSID) {
    return 0;
  } else {
    return -1;
  }
}
function reorderXYZ(sddList) {
  return sddList.slice().sort(xyzCompareFun);
}

// src/taper.ts
var taper_exports = {};
__export(taper_exports, {
  COSINE: () => COSINE,
  HAMMING: () => HAMMING,
  HANNING: () => HANNING,
  getCoefficients: () => getCoefficients,
  taper: () => taper
});
function taper(seis, width = 0.05, taperType = HANNING) {
  if (width > 0.5) {
    throw new Error(`Taper width cannot be larger than 0.5, width=${width}`);
  }
  if (seis.isContiguous()) {
    const data = seis.y;
    const outData = Float32Array.from(data);
    const w = Math.floor(data.length * width);
    const coeff = getCoefficients(taperType, w);
    const omega = coeff[0];
    const f0 = coeff[1];
    const f1 = coeff[2];
    for (let i = 0; i < w; i++) {
      const taperFactor = f0 - f1 * Math.cos(omega * i);
      outData[i] = outData[i] * taperFactor;
      outData[outData.length - i - 1] = outData[outData.length - i - 1] * taperFactor;
    }
    return seis.cloneWithNewData(outData);
  } else {
    throw new Error(
      `Cannot take taper of non-contiguous seismogram: ${seis.segments.length}`
    );
  }
}
function getCoefficients(type, length) {
  let omega, f0, f1;
  if (type === HANNING) {
    omega = Math.PI / length;
    f0 = 0.5;
    f1 = 0.5;
  } else if (type === HAMMING) {
    omega = Math.PI / length;
    f0 = 0.54;
    f1 = 0.46;
  } else {
    omega = Math.PI / 2 / length;
    f0 = 1;
    f1 = 1;
  }
  return [omega, f0, f1];
}
var HANNING = "HANNING";
var HAMMING = "HAMMING";
var COSINE = "COSINE";

// src/transfer.ts
var transfer_exports = {};
__export(transfer_exports, {
  METER: () => METER,
  METER_PER_SECOND: () => METER_PER_SECOND,
  METER_PER_SECOND_PER_SECOND: () => METER_PER_SECOND_PER_SECOND,
  applyFreqTaper: () => applyFreqTaper,
  calcFreqTaper: () => calcFreqTaper,
  calcGamma: () => calcGamma,
  calcResponse: () => calcResponse,
  calcResponseFromSacPoleZero: () => calcResponseFromSacPoleZero,
  calcScaleUnit: () => calcScaleUnit,
  calc_A0: () => calc_A0,
  combine: () => combine,
  convertPoleZeroToSacStyle: () => convertPoleZeroToSacStyle,
  convertToSacPoleZero: () => convertToSacPoleZero,
  evalPoleZeroInverse: () => evalPoleZeroInverse,
  transfer: () => transfer,
  transferSacPZ: () => transferSacPZ,
  transferSacPZSegment: () => transferSacPZSegment
});
import configureMeasurements from "convert-units";
import allMeasures from "convert-units/definitions/all";
var convert = configureMeasurements(allMeasures);
function transfer(seis, response, lowCut, lowPass, highPass, highCut) {
  if (!response) {
    throw new Error("Response not exist???");
  }
  const sacPoleZero = convertToSacPoleZero(response);
  return transferSacPZ(seis, sacPoleZero, lowCut, lowPass, highPass, highCut);
}
function transferSacPZ(seis, sacPoleZero, lowCut, lowPass, highPass, highCut) {
  const outSeis = [];
  for (let i = 0; i < seis.segments.length; i++) {
    const result = transferSacPZSegment(
      seis.segments[i],
      sacPoleZero,
      lowCut,
      lowPass,
      highPass,
      highCut
    );
    outSeis.push(result);
  }
  return new Seismogram(outSeis);
}
function transferSacPZSegment(seis, sacPoleZero, lowCut, lowPass, highPass, highCut) {
  const sampFreq = seis.sampleRate;
  const values = seis.y;
  let outData = Float32Array.from(values);
  outData.forEach((d, i) => outData[i] = d / sampFreq);
  let freqValues = calcDFT(outData);
  freqValues = combine(
    freqValues,
    sampFreq,
    sacPoleZero,
    lowCut,
    lowPass,
    highPass,
    highCut
  );
  outData = inverseDFT(freqValues, values.length);
  outData.forEach((d, i) => outData[i] = d * freqValues.length);
  const out = seis.cloneWithNewData(outData);
  out.yUnit = "m";
  return out;
}
function calcResponse(response, numPoints, sampleRate, unit) {
  const sacPoleZero = convertToSacPoleZero(response);
  const siUnit = unit.replaceAll("**", "");
  const unitQty = convert(1).getUnit(siUnit);
  let gamma = 0;
  if (unitQty === null) {
    throw new Error("unknown response unit: " + unit);
  } else if (unitQty.measure === "length") {
    gamma = 0;
  } else if (unitQty.measure === "speed") {
    gamma = 1;
  } else if (unitQty.measure === "acceleration") {
    gamma = 2;
  } else {
    throw new Error(
      "response unit is not displacement (m), velocity (m/s) or acceleration (m/s^2): " + unit
    );
  }
  sacPoleZero.trimZeros(gamma);
  const out = calcResponseFromSacPoleZero(sacPoleZero, numPoints, sampleRate);
  return out;
}
function calcResponseFromSacPoleZero(sacPoleZero, numPoints, sampleRate) {
  const deltaF = sampleRate / numPoints;
  const freqValues = new Float32Array(numPoints);
  let respAtS;
  respAtS = evalPoleZeroInverse(sacPoleZero, 0);
  respAtS = new Complex(1, 0).overComplex(respAtS);
  freqValues[0] = respAtS.real();
  let freq = sampleRate / 2;
  respAtS = evalPoleZeroInverse(sacPoleZero, freq);
  respAtS = new Complex(1, 0).overComplex(respAtS);
  freqValues[freqValues.length / 2] = respAtS.real();
  for (let i = 1; i < freqValues.length / 2; i++) {
    freq = i * deltaF;
    respAtS = evalPoleZeroInverse(sacPoleZero, freq);
    respAtS = new Complex(1, 0).overComplex(respAtS);
    if (respAtS.real() !== 0 || respAtS.imag() !== 0) {
      freqValues[i] = respAtS.real();
      freqValues[freqValues.length - i] = respAtS.imag();
    } else {
      freqValues[i] = 1e-10;
      freqValues[freqValues.length - i] = 0;
    }
  }
  const out = FFTResult.createFromPackedFreq(freqValues, numPoints, sampleRate);
  return out;
}
function combine(freqValues, sampFreq, sacPoleZero, lowCut, lowPass, highPass, highCut) {
  const deltaF = sampFreq / freqValues.length;
  freqValues[0] = 0;
  let freq = sampFreq / 2;
  let respAtS = evalPoleZeroInverse(sacPoleZero, freq);
  respAtS = respAtS.timesReal(
    deltaF * calcFreqTaper(freq, lowCut, lowPass, highPass, highCut)
  );
  freqValues[freqValues.length / 2] = respAtS.timesReal(freqValues[freqValues.length / 2]).real();
  for (let i = 1; i < freqValues.length / 2; i++) {
    freq = i * deltaF;
    respAtS = evalPoleZeroInverse(sacPoleZero, freq);
    respAtS = respAtS.timesReal(
      deltaF * calcFreqTaper(freq, lowCut, lowPass, highPass, highCut)
    );
    const freqComplex = new Complex(
      freqValues[i],
      freqValues[freqValues.length - i]
    ).timesComplex(respAtS);
    freqValues[i] = freqComplex.real();
    freqValues[freqValues.length - i] = freqComplex.imag();
  }
  return freqValues;
}
function evalPoleZeroInverse(sacPoleZero, freq) {
  return sacPoleZero.evalPoleZeroInverse(freq);
}
function calcFreqTaper(freq, lowCut, lowPass, highPass, highCut) {
  if (lowCut > lowPass || lowPass > highPass || highPass > highCut) {
    throw new Error(
      "must be lowCut > lowPass > highPass > highCut: " + lowCut + " " + lowPass + " " + highPass + " " + highCut
    );
  }
  if (freq <= lowCut || freq >= highCut) {
    return 0;
  }
  if (freq >= lowPass && freq <= highPass) {
    return 1;
  }
  if (freq > lowCut && freq < lowPass) {
    return 0.5 * (1 + Math.cos(Math.PI * (freq - lowPass) / (lowCut - lowPass)));
  }
  return 0.5 * (1 - Math.cos(Math.PI * (freq - highCut) / (highPass - highCut)));
}
function applyFreqTaper(fftResult, sampleRate, lowCut, lowPass, highPass, highCut) {
  const deltaF = fftResult.fundamentalFrequency;
  return FFTResult.createFromAmpPhase(
    fftResult.amplitudes().map(
      (v, i) => i === 0 ? 0 : v * calcFreqTaper(i * deltaF, lowCut, lowPass, highPass, highCut)
    ),
    fftResult.phases(),
    fftResult.origLength,
    fftResult.sampleRate
  );
}
var METER = convert().getUnit("m");
var METER_PER_SECOND = convert().getUnit("m/s");
var METER_PER_SECOND_PER_SECOND = convert().getUnit("m/s2");
function calcGamma(unit) {
  let gamma;
  const unitQty = convert(1).getUnit(unit);
  if (unitQty === null) {
    throw new Error("unknown response unit: " + unit);
  } else if (unitQty.measure === "length") {
    gamma = 0;
  } else if (unitQty.measure === "speed") {
    gamma = 1;
  } else if (unitQty.measure === "acceleration") {
    gamma = 2;
  } else {
    throw new Error(
      "response unit is not displacement (m), velocity (m/s) or acceleration (m/s^2): " + unit
    );
  }
  return gamma;
}
function calcScaleUnit(unit) {
  let scale;
  const unitQty = convert(1).getUnit(unit);
  if (unitQty === null) {
    throw new Error("unknown response unit: " + unit);
  } else if (unitQty.measure === "length") {
    scale = convert(1).from(unit).to("m");
  } else if (unitQty.measure === "speed") {
    scale = convert(1).from(unit).to("m/s");
  } else if (unitQty.measure === "acceleration") {
    scale = convert(1).from(unit).to("m/s2");
  } else {
    throw new Error(
      "response unit is not displacement (m), velocity (m/s) or acceleration (m/s^2): " + unit
    );
  }
  return scale;
}
function convertToSacPoleZero(response) {
  let polesZeros;
  if (response.stages[0].filter instanceof PolesZeros) {
    polesZeros = response.stages[0].filter;
  } else {
    throw new Error("can't find PolesZeros");
  }
  if (response.instrumentSensitivity === null) {
    throw new Error("response.instrumentSensitivity missing");
  }
  let unit = response.instrumentSensitivity.inputUnits;
  unit = unit.replaceAll("**", "");
  if (unit === "M") {
    unit = "m";
  }
  if (unit === "M/S" || unit === "M/SEC") {
    unit = "m/s";
  }
  if (unit === "M/S2" || unit === "M/SEC2") {
    unit = "m/s2";
  }
  const gamma = calcGamma(unit);
  const scaleUnit = calcScaleUnit(unit);
  const scale_sensitivity = scaleUnit * response.instrumentSensitivity.sensitivity;
  return convertPoleZeroToSacStyle(
    polesZeros,
    scale_sensitivity,
    response.instrumentSensitivity.frequency,
    gamma
  );
}
function convertPoleZeroToSacStyle(polesZeros, sensitivity, sensitivity_freq, gamma) {
  let mulFactor = 1;
  if (polesZeros.pzTransferFunctionType === "LAPLACE (HERTZ)") {
    mulFactor = 2 * Math.PI;
  }
  const zeros = [];
  for (let i = 0; i < polesZeros.zeros.length; i++) {
    zeros[i] = new Complex(
      polesZeros.zeros[i].real() * mulFactor,
      polesZeros.zeros[i].imag() * mulFactor
    );
  }
  for (let i = 0; i < gamma; i++) {
    zeros.push(new Complex(0, 0));
  }
  const poles = [];
  for (let i = 0; i < polesZeros.poles.length; i++) {
    poles[i] = new Complex(
      polesZeros.poles[i].real() * mulFactor,
      polesZeros.poles[i].imag() * mulFactor
    );
  }
  let constant = polesZeros.normalizationFactor;
  let sd = sensitivity;
  const fs = sensitivity_freq;
  sd *= Math.pow(2 * Math.PI * fs, gamma);
  let A0 = polesZeros.normalizationFactor;
  const fn = polesZeros.normalizationFrequency;
  A0 = A0 / Math.pow(2 * Math.PI * fn, gamma);
  if (polesZeros.pzTransferFunctionType === "LAPLACE (HERTZ)") {
    A0 *= Math.pow(
      2 * Math.PI,
      polesZeros.poles.length - polesZeros.zeros.length
    );
  }
  if (poles.length === 0 && zeros.length === 0) {
    constant = sd * A0;
  } else {
    constant = sd * calc_A0(poles, zeros, fs);
  }
  const sacPZ = new SacPoleZero(poles, zeros, constant);
  sacPZ.gamma = gamma;
  sacPZ.mulFactor = mulFactor;
  sacPZ.sd = sd;
  sacPZ.A0 = A0;
  return sacPZ;
}
function calc_A0(poles, zeros, ref_freq) {
  let numer = new Complex(1, 0);
  let denom = new Complex(1, 0);
  const f0 = new Complex(0, 2 * Math.PI * ref_freq);
  for (let i = 0; i < zeros.length; i++) {
    denom = denom.timesComplex(f0.minusComplex(zeros[i]));
  }
  for (let i = 0; i < poles.length; i++) {
    numer = numer.timesComplex(f0.minusComplex(poles[i]));
  }
  const a0 = numer.overComplex(denom).abs();
  return a0;
}

// src/usgsgeojson.ts
var usgsgeojson_exports = {};
__export(usgsgeojson_exports, {
  daySummaryAllUrl: () => daySummaryAllUrl,
  daySummaryM1_0Url: () => daySummaryM1_0Url,
  daySummaryM2_5Url: () => daySummaryM2_5Url,
  daySummaryM4_5Url: () => daySummaryM4_5Url,
  daySummarySignificantUrl: () => daySummarySignificantUrl,
  hourSummaryAllUrl: () => hourSummaryAllUrl,
  hourSummaryM1_0Url: () => hourSummaryM1_0Url,
  hourSummaryM2_5Url: () => hourSummaryM2_5Url,
  hourSummaryM4_5Url: () => hourSummaryM4_5Url,
  hourSummarySignificantUrl: () => hourSummarySignificantUrl,
  isValidUSGSGeoJsonQuake: () => isValidUSGSGeoJsonQuake,
  isValidUSGSGeoJsonSummary: () => isValidUSGSGeoJsonSummary,
  loadDaySummaryAll: () => loadDaySummaryAll,
  loadDaySummaryM1_0: () => loadDaySummaryM1_0,
  loadDaySummaryM2_5: () => loadDaySummaryM2_5,
  loadDaySummaryM4_5: () => loadDaySummaryM4_5,
  loadDaySummarySignificant: () => loadDaySummarySignificant,
  loadHourSummaryAll: () => loadHourSummaryAll,
  loadHourSummaryM1_0: () => loadHourSummaryM1_0,
  loadHourSummaryM2_5: () => loadHourSummaryM2_5,
  loadHourSummaryM4_5: () => loadHourSummaryM4_5,
  loadHourSummarySignificant: () => loadHourSummarySignificant,
  loadMonthSummaryAll: () => loadMonthSummaryAll,
  loadMonthSummaryM1_0: () => loadMonthSummaryM1_0,
  loadMonthSummaryM2_5: () => loadMonthSummaryM2_5,
  loadMonthSummaryM4_5: () => loadMonthSummaryM4_5,
  loadMonthSummarySignificant: () => loadMonthSummarySignificant,
  loadRawUSGSGeoJsonSummary: () => loadRawUSGSGeoJsonSummary,
  loadUSGSGeoJsonSummary: () => loadUSGSGeoJsonSummary,
  loadUSGSSummary: () => loadUSGSSummary,
  loadWeekSummaryAll: () => loadWeekSummaryAll,
  loadWeekSummaryM1_0: () => loadWeekSummaryM1_0,
  loadWeekSummaryM2_5: () => loadWeekSummaryM2_5,
  loadWeekSummaryM4_5: () => loadWeekSummaryM4_5,
  loadWeekSummarySignificant: () => loadWeekSummarySignificant,
  monthSummaryAllUrl: () => monthSummaryAllUrl,
  monthSummaryM1_0Url: () => monthSummaryM1_0Url,
  monthSummaryM2_5Url: () => monthSummaryM2_5Url,
  monthSummaryM4_5Url: () => monthSummaryM4_5Url,
  monthSummarySignificantUrl: () => monthSummarySignificantUrl,
  parseFeatureAsQuake: () => parseFeatureAsQuake,
  parseGeoJSON: () => parseGeoJSON,
  weekSummaryAllUrl: () => weekSummaryAllUrl,
  weekSummaryM1_0Url: () => weekSummaryM1_0Url,
  weekSummaryM2_5Url: () => weekSummaryM2_5Url,
  weekSummaryM4_5Url: () => weekSummaryM4_5Url,
  weekSummarySignificantUrl: () => weekSummarySignificantUrl
});
import { DateTime as DateTime13 } from "luxon";
var timeoutSec = 10;
var hourSummarySignificantUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_hour.geojson";
var hourSummaryM4_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/4.5_hour.geojson";
var hourSummaryM2_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/2.5_hour.geojson";
var hourSummaryM1_0Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/1.0_hour.geojson";
var hourSummaryAllUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/all_hour.geojson";
var daySummarySignificantUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_day.geojson";
var daySummaryM4_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/4.5_day.geojson";
var daySummaryM2_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/2.5_day.geojson";
var daySummaryM1_0Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/1.0_day.geojson";
var daySummaryAllUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/all_day.geojson";
var weekSummarySignificantUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_week.geojson";
var weekSummaryM4_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/4.5_week.geojson";
var weekSummaryM2_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/2.5_week.geojson";
var weekSummaryM1_0Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/1.0_week.geojson";
var weekSummaryAllUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/all_week.geojson";
var monthSummarySignificantUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_month.geojson";
var monthSummaryM4_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/4.5_month.geojson";
var monthSummaryM2_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/2.5_month.geojson";
var monthSummaryM1_0Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/1.0_month.geojson";
var monthSummaryAllUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/all_month.geojson";
function loadHourSummarySignificant() {
  return loadUSGSSummary(hourSummarySignificantUrl);
}
function loadHourSummaryM4_5() {
  return loadUSGSSummary(hourSummaryM4_5Url);
}
function loadHourSummaryM2_5() {
  return loadUSGSSummary(hourSummaryM2_5Url);
}
function loadHourSummaryM1_0() {
  return loadUSGSSummary(hourSummaryM1_0Url);
}
function loadHourSummaryAll() {
  return loadUSGSSummary(hourSummaryAllUrl);
}
function loadDaySummarySignificant() {
  return loadUSGSSummary(daySummarySignificantUrl);
}
function loadDaySummaryM4_5() {
  return loadUSGSSummary(daySummaryM4_5Url);
}
function loadDaySummaryM2_5() {
  return loadUSGSSummary(daySummaryM2_5Url);
}
function loadDaySummaryM1_0() {
  return loadUSGSSummary(daySummaryM1_0Url);
}
function loadDaySummaryAll() {
  return loadUSGSSummary(daySummaryAllUrl);
}
function loadWeekSummarySignificant() {
  return loadUSGSSummary(weekSummarySignificantUrl);
}
function loadWeekSummaryM4_5() {
  return loadUSGSSummary(weekSummaryM4_5Url);
}
function loadWeekSummaryM2_5() {
  return loadUSGSSummary(weekSummaryM2_5Url);
}
function loadWeekSummaryM1_0() {
  return loadUSGSSummary(weekSummaryM1_0Url);
}
function loadWeekSummaryAll() {
  return loadUSGSSummary(weekSummaryAllUrl);
}
function loadMonthSummarySignificant() {
  return loadUSGSSummary(monthSummarySignificantUrl);
}
function loadMonthSummaryM4_5() {
  return loadUSGSSummary(monthSummaryM4_5Url);
}
function loadMonthSummaryM2_5() {
  return loadUSGSSummary(monthSummaryM2_5Url);
}
function loadMonthSummaryM1_0() {
  return loadUSGSSummary(monthSummaryM1_0Url);
}
function loadMonthSummaryAll() {
  return loadUSGSSummary(monthSummaryAllUrl);
}
function loadUSGSSummary(url) {
  return loadUSGSGeoJsonSummary(url).then((eventParameters) => {
    return eventParameters.eventList;
  });
}
function loadUSGSGeoJsonSummary(url) {
  return loadRawUSGSGeoJsonSummary(url).then((geojson) => {
    return parseGeoJSON(geojson);
  });
}
function loadRawUSGSGeoJsonSummary(url) {
  const fetchInit = defaultFetchInitObj(JSON_MIME);
  return doFetchWithTimeout(url, fetchInit, timeoutSec * 1e3).then((response) => {
    if (response.status !== 200) {
      return [];
    } else {
      return response.json();
    }
  }).then((jsonValue) => {
    if (isValidUSGSGeoJsonSummary(jsonValue)) {
      return jsonValue;
    } else {
      throw new TypeError(`Oops, we did not get roottype JSON!`);
    }
  });
}
function parseGeoJSON(geojson) {
  const quakeList = [];
  const description = geojson.metadata.title;
  for (const f of geojson.features) {
    const quake = parseFeatureAsQuake(f);
    quakeList.push(quake);
  }
  const out = new EventParameters();
  out.creationInfo = new CreationInfo();
  out.creationInfo.agencyURI = geojson.metadata.url;
  out.creationInfo.creationTime = DateTime13.fromMillis(
    geojson.metadata.generated
  );
  out.eventList = quakeList;
  out.description = description;
  return out;
}
function parseFeatureAsQuake(feature) {
  const quake = new Quake();
  quake.publicId = `quakeml:earthquake.usgs.gov/fdsnws/event/1/query?eventid={feature.id}`;
  const p = feature.properties;
  if (p == null) {
    throw new Error("Geojson missing properties");
  }
  quake.descriptionList.push(new EventDescription(p.title));
  const origin = new Origin(
    DateTime13.fromMillis(p.time),
    feature.geometry.coordinates[1],
    feature.geometry.coordinates[0]
  );
  origin.depth = feature.geometry.coordinates[2] * 1e3;
  quake.originList.push(origin);
  const mag = new Magnitude(p.mag);
  mag.type = p.magType;
  quake.magnitudeList.push(mag);
  quake.preferredOrigin = origin;
  quake.preferredMagnitude = mag;
  return quake;
}
function isValidUSGSGeoJsonSummary(jsonValue) {
  if (!jsonValue || typeof jsonValue !== "object") {
    throw new TypeError("json is not object");
  }
  const jsonObj = jsonValue;
  if (!(typeof jsonObj.type === "string" && jsonObj.type === "FeatureCollection")) {
    throw new TypeError(
      "geojson is not valid for USGS GeoJson, type should be FeatureCollection"
    );
  }
  if (!(typeof jsonObj.metadata === "object")) {
    throw new TypeError(
      "geojson is not valid for USGS GeoJson, missing metadata"
    );
  }
  if (!Array.isArray(jsonObj.features)) {
    throw new TypeError(
      "geojson is not valid for USGS GeoJson, features should be array"
    );
  } else {
    if (!jsonObj.features.every(isValidUSGSGeoJsonQuake)) {
      throw new TypeError(
        "geojson is not valid for USGS GeoJson, feature should be USGSGeoJsonFeature"
      );
    }
  }
  return true;
}
function isValidUSGSGeoJsonQuake(jsonValue) {
  if (!jsonValue || typeof jsonValue !== "object") {
    throw new TypeError("json is not object");
  }
  const jsonObj = jsonValue;
  if (!(typeof jsonObj.type === "string" && jsonObj.type === "Feature")) {
    throw new TypeError(
      "geojson is not valid for USGS GeoJson, type should be Feature"
    );
  }
  if (!(typeof jsonObj.properties === "object")) {
    throw new TypeError(
      "geojson is not valid for USGS GeoJson, missing properties"
    );
  }
  if (!(typeof jsonObj.id === "string")) {
    throw new TypeError(
      "geojson is not valid for USGS GeoJson, id should be string"
    );
  }
  return true;
}

// src/vector.ts
var vector_exports = {};
__export(vector_exports, {
  DtoR: () => DtoR,
  RotatedSeismograms: () => RotatedSeismograms,
  rotate: () => rotate,
  vectorMagnitude: () => vectorMagnitude
});
var DtoR = Math.PI / 180;
var RotatedSeismograms = class {
  radial;
  transverse;
  azimuthRadial;
  azimuthTransverse;
  rotation;
  constructor(radial, azimuthRadial, transverse, azimuthTransverse, rotation) {
    this.radial = radial;
    this.azimuthRadial = azimuthRadial;
    this.transverse = transverse;
    this.azimuthTransverse = azimuthTransverse;
    this.rotation = rotation;
  }
};
function rotate(seisA, azimuthA, seisB, azimuthB, azimuth) {
  if (seisA.segments.length !== seisB.segments.length) {
    throw new Error(
      `Seismograms do not have same number of segments: ${seisA.segments.length} !== ${seisB.segments.length}`
    );
  }
  const rotOutRad = [];
  const rotOutTrans = [];
  for (let i = 0; i < seisA.segments.length; i++) {
    const result = rotateSeismogramSegment(
      seisA.segments[i],
      azimuthA,
      seisB.segments[i],
      azimuthB,
      azimuth
    );
    rotOutRad.push(result.radial);
    rotOutTrans.push(result.transverse);
  }
  const out = new RotatedSeismograms(
    new Seismogram(rotOutRad),
    azimuth % 360,
    new Seismogram(rotOutTrans),
    (azimuth + 90) % 360,
    azimuth - azimuthA
  );
  return out;
}
function rotateSeismogramSegment(seisA, azimuthA, seisB, azimuthB, azimuth) {
  if (seisA.y.length !== seisB.y.length) {
    throw new Error(
      `seisA and seisB should be of same lenght but was ${seisA.y.length} ${seisB.y.length}`
    );
  }
  if (!seisA.startTime.equals(seisB.startTime)) {
    throw new Error(
      `Expect startTime to be same, but was ${seisA.startTime.toISO()} ${seisB.startTime.toISO()}`
    );
  }
  if (seisA.sampleRate !== seisB.sampleRate) {
    throw new Error(
      `Expect sampleRate to be same, but was ${seisA.sampleRate} ${seisB.sampleRate}`
    );
  }
  if ((azimuthA + 90) % 360 !== azimuthB % 360) {
    throw new Error(
      `Expect azimuthB to be azimuthA + 90, but was ${azimuthA} ${azimuthB}`
    );
  }
  const rotRadian = 1 * DtoR * (azimuth - azimuthA);
  const cosTheta = Math.cos(rotRadian);
  const sinTheta = Math.sin(rotRadian);
  const x = new Float32Array(seisA.y.length);
  const y = new Float32Array(seisA.y.length);
  for (let i = 0; i < seisA.y.length; i++) {
    x[i] = cosTheta * seisB.yAtIndex(i) - sinTheta * seisA.yAtIndex(i);
    y[i] = sinTheta * seisB.yAtIndex(i) + cosTheta * seisA.yAtIndex(i);
  }
  const outSeisRad = seisA.cloneWithNewData(y);
  const outSeisTan = seisA.cloneWithNewData(x);
  if (seisA.sourceId) {
    outSeisRad.sourceId = seisA.sourceId.clone();
    outSeisRad.sourceId.subsourceCode = "R";
    outSeisTan.sourceId = seisA.sourceId.clone();
    outSeisTan.sourceId.subsourceCode = "T";
  }
  const out = {
    radial: outSeisRad,
    transverse: outSeisTan,
    azimuthRadial: azimuth % 360,
    azimuthTransverse: (azimuth + 90) % 360
  };
  return out;
}
function vectorMagnitude(seisA, seisB, seisC, orientCode) {
  if (seisA.segments.length !== seisB.segments.length || seisA.segments.length !== seisC.segments.length) {
    throw new Error(
      `Seismograms do not have same number of segments: ${seisA.segments.length}  !== ${seisB.segments.length}  !== ${seisC.segments.length}`
    );
  }
  const outData = [];
  for (let i = 0; i < seisA.segments.length; i++) {
    const result = vectorMagnitudeSegment(
      seisA.segments[i],
      seisB.segments[i],
      seisC.segments[i],
      orientCode
    );
    outData.push(result);
  }
  const outSeis = new Seismogram(outData);
  return outSeis;
}
function vectorMagnitudeSegment(seisA, seisB, seisC, orientCode) {
  if (seisA.y.length !== seisB.y.length || seisA.y.length !== seisC.y.length) {
    throw new Error(
      `seis should be of same length but was ${seisA.y.length} ${seisB.y.length} ${seisC.y.length}`
    );
  }
  if (seisA.sampleRate !== seisB.sampleRate || seisA.sampleRate !== seisC.sampleRate) {
    throw new Error(
      `Expect sampleRate to be same, but was ${seisA.sampleRate} ${seisB.sampleRate} ${seisC.sampleRate}`
    );
  }
  let y;
  if (seisA.y instanceof Float64Array) {
    y = new Float64Array(seisA.y.length);
  } else {
    y = new Float32Array(seisA.y.length);
  }
  for (let i = 0; i < seisA.y.length; i++) {
    y[i] = Math.sqrt(
      seisA.y[i] * seisA.y[i] + seisB.y[i] * seisB.y[i] + seisC.y[i] * seisC.y[i]
    );
  }
  const outSeis = seisA.cloneWithNewData(y);
  if (!isDef(orientCode)) {
    orientCode = "M";
  }
  outSeis.sourceId = seisA.sourceId.clone();
  outSeis.sourceId.subsourceCode = orientCode;
  return outSeis;
}

// src/index_node.ts
import * as OregonDSP2 from "oregondsp";
import * as luxon from "luxon";
var axisutil = null;
var animatedseismograph = null;
var components = null;
var cssutil = null;
var datechooser = null;
var infotable = null;
var fdsneventcomponent = null;
var fdsnstationcomponent = null;
var handlebarshelpers = null;
var helicorder = null;
var organizeddisplay = null;
var particlemotion = null;
var seismograph = null;
var seismographconfig = null;
var seismographutil = null;
var seismographconfigeditor = null;
var spectraplot = null;
var leaflet = null;
var leafletutil = null;
export {
  OregonDSP2 as OregonDSP,
  animatedseismograph,
  axisutil,
  components,
  cssutil,
  datalink_exports as datalink,
  dataset_exports as dataset,
  datechooser,
  distaz_exports as distaz,
  fdsnavailability_exports as fdsnavailability,
  fdsncommon_exports as fdsncommon,
  fdsndatacenters_exports as fdsndatacenters,
  fdsndataselect_exports as fdsndataselect,
  fdsnevent_exports as fdsnevent,
  fdsneventcomponent,
  fdsnsourceid_exports as fdsnsourceid,
  fdsnstation_exports as fdsnstation,
  fdsnstationcomponent,
  fft_exports as fft,
  filter_exports as filter,
  handlebarshelpers,
  helicorder,
  infotable,
  leaflet,
  leafletutil,
  luxon,
  miniseed_exports as miniseed,
  mseed3_exports as mseed3,
  mseedarchive_exports as mseedarchive,
  oregondsputil_exports as oregondsputil,
  organizeddisplay,
  particlemotion,
  quakeml_exports as quakeml,
  ringserverweb_exports as ringserverweb,
  sacpolezero_exports as sacPoleZero,
  scale_exports as scale,
  seedcodec_exports as seedcodec,
  seedlink_exports as seedlink,
  seedlink4_exports as seedlink4,
  seismogram_exports as seismogram,
  seismogramloader_exports as seismogramloader,
  seismogramsegment_exports as seismogramsegment,
  seismograph,
  seismographconfig,
  seismographconfigeditor,
  seismographutil,
  sorting_exports as sorting,
  spectraplot,
  stationxml_exports as stationxml,
  taper_exports as taper,
  transfer_exports as transfer,
  traveltime_exports as traveltime,
  usgsgeojson_exports as usgsgeojson,
  util_exports as util,
  vector_exports as vector,
  version
};
